#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009  John Schneiderman <JohnMS@member.fsf.org>        ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

from mediasong import MediaSong

import wx


class SearchWindow(wx.Panel):
    """ Interface to search for a song in the media library. """

    """
     ATTRIBUTES
    """
    # A list of the MediaSong objects displayed in the results panel.
    __displayedResults = []
    # Is the width of the media column.
    __MEDIA_WIDTH = 50
    # Approximate width of the vertical scroll bar
    __VERTICAL_SCROLL_BAR = 25

    """
     EVENT ID
    """
    # Identifier for the area where the user enters search terms.
    ID_SEARCH_ENTRY = wx.NewId()
    # Identifier for the result display.
    ID_SEARCH_RESULTS = wx.NewId() # Display for the results of a search
    # Search for audio media
    __SEARCH_AUDIO = wx.NewId()
    # Search for video media
    __SEARCH_VIDEO = wx.NewId()
    # Column ID for the media type in the results window
    __ID_MEDIA_COLUMN = 0
    # Column ID for the song title in the results window
    __ID_TITLE_COLUMN = 1
    # Column ID for the artist title in the results window
    __ID_ARTIST_COLUMN = 2

    def __init__(self, parent, searchAudio=True, searchVideo=True):
        """ Default constructor

         wx.Window parent: is the parent window.
         boolean searchAudio: True if the user wants to search for audio media.
         boolean searchVideo: True if the user wants to search for video media.
        """
        wx.Panel.__init__(self, parent, wx.NewId())
        self.__createControls(searchAudio, searchVideo)
        self.__bindEvents()
        self.__doLayout()

    def __createControls(self, searchAudio, searchVideo):
        """ Create the controls for the search window.

         boolean searchAudio: True if the user wants to search for audio media.
         boolean searchVideo: True if the user wants to search for video media.
        """
        # Create Search Box
        self.__srchCtrlSearch = wx.SearchCtrl(self, self.ID_SEARCH_ENTRY, \
            style=wx.TE_PROCESS_ENTER)
        self.__srchCtrlSearch.ShowCancelButton(True)
        self.__srchCtrlSearch.ShowSearchButton(True)
        self.__srchCtrlSearch.SetDescriptiveText("Search Media Library")

        # Create Search Box Menu
        self.__mnuSearch = wx.Menu("Search Options")
        self.__mnuSearch.AppendCheckItem(self.__SEARCH_AUDIO, "Audio Media")
        self.__mnuSearch.Check(self.__SEARCH_AUDIO, searchAudio)
        self.__mnuSearch.AppendCheckItem(self.__SEARCH_VIDEO, "Video Media")
        self.__mnuSearch.Check(self.__SEARCH_VIDEO, searchVideo)
        self.__srchCtrlSearch.SetMenu(self.__mnuSearch)

        # Create Search Result Displays
        self.__lstCtlSearchResults = wx.ListCtrl(self, \
            self.ID_SEARCH_RESULTS, style=wx.LC_REPORT | wx.SUNKEN_BORDER | \
             wx.LC_HRULES | wx.LC_VRULES)
        self.__lstCtlSearchResults.InsertColumn(self.__ID_MEDIA_COLUMN, \
            "Media", self.__MEDIA_WIDTH)
        self.__lstCtlSearchResults.InsertColumn(self.__ID_TITLE_COLUMN, \
            "Title")
        self.__lstCtlSearchResults.InsertColumn(self.__ID_ARTIST_COLUMN, \
            "Artist")

    def __bindEvents(self):
        """ Connects all the needed events.

         Connects each of the control objects created to their corresponding
           methods.
        """
        wx.EVT_LIST_INSERT_ITEM(self, self.ID_SEARCH_RESULTS, \
            self.__onListInsertItem)

    def __doLayout(self):
        """ Creates a visually pleasing look for the controls.

         All the control objects are placed on the window.
        """

        # Search Entry Layout
        self.__bxSzrSearchEntry = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrSearchEntry.Add(self.__srchCtrlSearch, 1, wx.EXPAND | \
            wx.ALL | wx.FIXED_MINSIZE, 3)

        # Search Result Layout
        self.__bxSzrSearchResults = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrSearchResults.Add(self.__lstCtlSearchResults, 1, \
            wx.EXPAND | wx.ALL, 3)

        # Create Interface
        self.__bxSzrInterface = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrInterface.Add(self.__bxSzrSearchEntry, 0, wx.EXPAND | \
            wx.ALL)
        self.__bxSzrInterface.AddSpacer(5)
        self.__bxSzrInterface.Add(self.__bxSzrSearchResults, 1, wx.EXPAND | \
            wx.ALL)
        self.SetSizer(self.__bxSzrInterface)

    def searchTerms(self):
        """ Accessor to the search terms.

         return string: the search terms entered by the user.
        """
        return self.__srchCtrlSearch.GetValue()

    def searchForAudios(self):
        """ Accessor to the search criteria

         return boolean: True if the user wants to search for audio media.
        """
        return self.__mnuSearch.IsChecked(self.__SEARCH_AUDIO)

    def searchForVideos(self):
        """ Accessor to the search criteria

         return boolean: True if the user wants to search for video media.
        """
        return self.__mnuSearch.IsChecked(self.__SEARCH_VIDEO)

    def clearSearchTerms(self):
        """ Clears the search terms. """
        self.__srchCtrlSearch.SetValue("")

    def selectedSongs(self):
        """ Accessor to the currently selected songs

         return list[MediaSong]: each of the songs selected by the user.
        """
        selection = []
        current = -1 # Current index
        while True:
            next = self.__getNextSelectedSong(current)
            if next == -1:
                return selection

            selection.append(self.__displayedResults[next])
            current = next

    def __getNextSelectedSong(self, current):
        """ Get the next selected song in the results display.

         int current: is the current index.
         return int: the index of the next selected item, or -1 when no more
           selections are left.
        """
        return self.__lstCtlSearchResults.GetNextItem(current, \
            wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)

    def clearSearchResults(self):
        """ Clears the display result window """
        del self.__displayedResults[:]
        self.__lstCtlSearchResults.DeleteAllItems()

    def addSearchResult(self, result):
        """ Adds a search result to display for the user.

         MediaSong result: is the song to add.
        """
        self.__displayedResults.append(result)
        index = len(self.__displayedResults) - 1
        # Title Information
        item = wx.ListItem()
        item.SetId(index)
        item.SetColumn(self.__ID_TITLE_COLUMN)
        item.SetText(result.title)
        self.__lstCtlSearchResults.InsertItem(item)
        # Artist Information
        item = wx.ListItem()
        item.SetId(index)
        item.SetColumn(self.__ID_ARTIST_COLUMN)
        item.SetText(result.artist)
        self.__lstCtlSearchResults.SetItem(item)
        # Video Information
        item = wx.ListItem()
        item.SetId(index)
        item.SetColumn(self.__ID_MEDIA_COLUMN)
        if result.blIsVideo:
            item.SetText("Video")
        else:
            item.SetText("Audio")
        self.__lstCtlSearchResults.SetItem(item)

    def __onListInsertItem(self, event):
        """ Handles the column width display as each item is inserted.

         wx.Event event: is the event object (Not Used)
        """
        self.__adjustResultColumns()

    def __adjustResultColumns(self):
        """ Adjusts the width of the result columns. """
        width = self.__bxSzrInterface.GetSize().GetWidth() - \
            self.__MEDIA_WIDTH - self.__VERTICAL_SCROLL_BAR
        self.__lstCtlSearchResults.SetColumnWidth(self.__ID_TITLE_COLUMN, \
            width / 2)
        self.__lstCtlSearchResults.SetColumnWidth(self.__ID_ARTIST_COLUMN, \
            width / 2)
        self.__lstCtlSearchResults.SetColumnWidth(self.__ID_MEDIA_COLUMN, \
            self.__MEDIA_WIDTH)
