#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009  John Schneiderman <JohnMS@member.fsf.org>        ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

import wx


class PlayerWindow(wx.Panel):
    """ The interface to control the song playing """

    """
     ATTRIBUTES
    """
    # The lowest possible volume percentage
    __VOLUME_MINIMUM = 0
    # The highest possible volume percentage
    __VOLUME_MAXIMUM = 100
    # The lowest possible position percentage into a song.
    __TRACK_MINIMUM = 0
    # The highest possible position percentage into a song.
    __TRACK_MAXIMUM = 100 # End of song

    """
     EVENT ID
    """
    # Used to indicate to play or pause a song.
    ID_PLAY = wx.NewId()
    # Used to indicate to stop a song.
    ID_STOP = wx.NewId()
    # Used to indicate to change the track position.
    ID_TRACK_POSITION = wx.NewId()
    # Used to indicate to change the volume level.
    ID_VOLUME = wx.NewId()

    def __init__(self, parent, id=wx.NewId(), volume=50):
        """ Default constructor

         wx.Window parent: is the parent window.
         wx.WindowID id: is the unique identification number for the window
         int volume: is the percentage volume level to display.
        """
        wx.Panel.__init__(self, parent, id)
        self.__createControls(volume)
        self.__doLayout()

    def __createControls(self, volume):
        """ Create the controls for the player window.

         int volume: is the percentage volume level to display.
        """

        # Track Time Information
        self.__sttcTxtTimePosition = wx.StaticText(self, wx.ID_ANY, "Current")
        self.__txtCtrlTimePosition = wx.TextCtrl(self, wx.ID_ANY, "00:0.0", \
            style=wx.TE_READONLY)
        self.__sttcTxtTimeTotal = wx.StaticText(self, wx.ID_ANY, "Total")
        self.__txtCtrlTimeTotal = wx.TextCtrl(self, wx.ID_ANY, "00:0.0", \
            style=wx.TE_READONLY)
        self.__sldrTrackPosition = wx.Slider(self, self.ID_TRACK_POSITION, \
            0, self.__TRACK_MINIMUM, self.__TRACK_MAXIMUM, \
            style=wx.SL_AUTOTICKS)

        # Create Player Buttons
        self.__bttnPlay = wx.Button(self, self.ID_PLAY, "Play")
        self.__bttnStop = wx.Button(self, self.ID_STOP, "Stop")

        # Volume Control
        self.__sldrVolume = wx.Slider(self, self.ID_VOLUME, volume, \
            self.__VOLUME_MINIMUM, self.__VOLUME_MAXIMUM, \
            style=wx.SL_VERTICAL | wx.SL_INVERSE | wx.SL_RIGHT | wx.SL_LABELS)


    def __doLayout(self):
        """ Places the controls in a defined layout. """

        # Track Time Layout
        self.__bxSzrTimePosition = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrTimePosition.Add(self.__sttcTxtTimePosition, 0, \
            wx.ALL | wx.ALIGN_RIGHT, 3)
        self.__bxSzrTimePosition.Add(self.__txtCtrlTimePosition, 0, \
            wx.ALL | wx.ALIGN_RIGHT, 3)
        self.__bxSzrTimeTotal = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrTimeTotal.Add(self.__sttcTxtTimeTotal, 0, wx.ALL | \
            wx.ALIGN_RIGHT, 3)
        self.__bxSzrTimeTotal.Add(self.__txtCtrlTimeTotal, 0, wx.ALL | \
            wx.ALIGN_RIGHT, 3)
        self.__bxSzrTrackTime = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrTrackTime.Add(self.__bxSzrTimePosition, 0, wx.ALL | \
            wx.ALIGN_RIGHT, 3)
        self.__bxSzrTrackTime.Add(self.__bxSzrTimeTotal, 0, wx.ALL | \
            wx.ALIGN_RIGHT, 3)
        self.__bxSzrTrackTime.AddSpacer(5)
        self.__bxSzrTrackTime.Add(self.__sldrTrackPosition, 1, wx.EXPAND | \
            wx.ALIGN_RIGHT)

        # Player Layout
        self.__bxSzrPlayerControls = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrPlayerControls.Add(self.__bttnPlay, 0, wx.ALL, 3)
        self.__bxSzrPlayerControls.Add(self.__bttnStop, 0, wx.ALL, 3)

        # Volume Layout
        self.__bxSzrVolume = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrVolume.Add(self.__sldrVolume, 1, wx.EXPAND)

        # Create Interface
        self.__bxSzrInterface = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrInterface.AddStretchSpacer(4)
        self.__bxSzrInterface.Add(self.__bxSzrTrackTime, 0, wx.EXPAND)
        self.__bxSzrInterface.AddSpacer(5)
        self.__bxSzrInterface.Add(self.__bxSzrPlayerControls, 0, wx.EXPAND)
        self.__bxSzrInterface.Add(self.__bxSzrVolume, 1, wx.EXPAND | \
            wx.FIXED_MINSIZE)
        self.__bxSzrInterface.AddStretchSpacer()
        self.SetSizer(self.__bxSzrInterface)

    def volumeLevel(self):
        """ Accessor to the displayed volume level """
        return self.__sldrVolume.GetValue()

    def setVolumeLevel(self, volume):
        """ Modifier to the displayed volume level. """
        self.__sldrVolume.SetValue(volume)

    def trackPosition(self):
        """ Accessor to the displayed track position. """
        return self.__sldrTrackPosition.GetValue()

    def setTrackPosition(self, secondsPassed, secondsTotal):
        """ Modifier to the displayed track position.

         float secondsPassed: are the number of seconds into the track.
         flat secondsTotal: are the total number of seconds in the track.
        """
        minutes = secondsPassed / 60
        seconds = float(secondsPassed) % float(60)
        self.__txtCtrlTimePosition.SetValue("%02d:%02.1f" % (minutes, seconds))
        minutes = secondsTotal / 60
        seconds = float(secondsTotal) % float(60)
        self.__txtCtrlTimeTotal.SetValue("%02d:%02.1f" % (minutes, seconds))
        if secondsTotal > 0:
            self.__sldrTrackPosition.SetRange(0, secondsTotal)
            self.__sldrTrackPosition.SetValue(secondsPassed)
        else:
            self.__sldrTrackPosition.SetValue(0)

    def isDisplayingPlay(self):
        """ Determines if the player is displaying "Play".

         return boolean: True if displaying "Play", False else-wise.
        """
        if self.__bttnPlay.GetLabel() == "Play":
            return True
        else:
            return False

    def togglePlay(self):
        """ Toggles between displaying "Play" and "Pause".

         Toggles to "Pause" if currently displaying "Play", and switches to
           "Play" when currently displaying "Pause".
        """
        if self.__bttnPlay.GetLabel() == "Play":
            self.__bttnPlay.SetLabel("Pause")
        else:
            self.__bttnPlay.SetLabel("Play")
