/* 
Copyright (C) 2010 Kevin Allen

This file is part of kacq.

kacq is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

kacq is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with kacq.  If not, see <http://www.gnu.org/licenses/>.

all the functions for the recording part of kacq 
*/
#include "main.h"

int recording_interface_init(struct recording_interface* rec)
{
  /* initialize the recording interface, by default it records all channels */
  int i;
  rec->is_recording=0;
  rec->number_of_channels_to_save=RECORDING_MAXIMUM_CHANNELS;
  
  // set the home directory as the default directory
  struct passwd *p;
  char *username=getenv("USER");
  p=getpwnam(username);
  rec->directory=strcat(p->pw_dir,"/");
   
  rec->rec_buffer_size=MAX_BUFFER_LENGTH*COMEDI_INTERFACE_MAX_DEVICES*FACTOR_RECORDING_BUFFER;
  if((rec->rec_buffer_data=malloc(sizeof(short int)*rec->rec_buffer_size))==NULL)
    {
      fprintf(stderr,"problem allocating memory for rec->rec_buffer_data\n");
      return -1;
    }
  rec->proportion_buffer_filled_before_save=RECORDING_PROPORTION_BUFFER_FILLED_BEFORE_SAVE;
  rec->max_samples_in_buffer=rec->rec_buffer_size/rec->number_of_channels_to_save;
  rec->new_samples_in_buffer=0;
  rec->file_size=0;
  rec->inter_recording_sleep_ms=40;
  rec->inter_recording_sleep_timespec=set_timespec_from_ms(rec->inter_recording_sleep_ms);
  rec->number_samples_saved=0;
  for (i=0;i<rec->number_of_channels_to_save;i++)
    {
      rec->channel_list[i]=i;
    }
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_init\n");
  fprintf(stderr,"rec_buffer_size: %d\n",rec->rec_buffer_size);
#endif
  return 0;  
}
int recording_interface_free(struct recording_interface* rec)
{
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_free\n");
#endif
  if(rec->rec_buffer_data!=NULL)
    {free(rec->rec_buffer_data);}
  return 0;
}


int recording_interface_open_file(struct recording_interface* rec)
{
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_open_file on %s\n",rec->file_name);
#endif
  rec->file=fopen(rec->file_name,"w");
  if (rec->file==NULL)
    {
      fprintf(stderr,"error opening %s in recording_interface_open_file()\n",rec->file_name);
      return -1;
    }
  return 0;
}
int recording_interface_get_data(struct recording_interface* rec, struct comedi_interface*com)
{
  /* this function takes new available samples from the comedi_interface and put them into the recording_interface buffer 
     if the buffer is more than half full, then save the data to disk
   */
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_get_data\n");
#endif
  int index_rec,index_acq,num_samples_to_transfer,i,j,num_samples_to_transfer_end, num_samples_to_transfer_start;


  // prevent other threads from changing comedi_interface_buffer
  pthread_mutex_lock( &mutex_comedi_interface_buffer );

  // check if the comedi device has new samples that are not in current recording  buffer
  if(com->number_samples_read > rec->number_samples_saved+rec->new_samples_in_buffer)
    {
      
      // calculate the number of samples to transfer
      if (com->number_samples_read-rec->number_samples_saved<rec->max_samples_in_buffer)
	{
	  // there is space for all the samples available from comedi_interface
	  num_samples_to_transfer=com->number_samples_read-(rec->number_samples_saved+rec->new_samples_in_buffer);
	}
      else
	{
	  // there is only space for part of samples available
	  num_samples_to_transfer=rec->max_samples_in_buffer-rec->new_samples_in_buffer;
	}
      
      // find the index in the recording buffer for the new samples coming from comedi_interface buffer
      index_rec=rec->new_samples_in_buffer*rec->number_of_channels_to_save;
      

      // if the comedi buffer has the data in chronological order and starts to sample 0
      if (com->number_samples_read<com->max_number_samples_in_buffer)
	{
	  index_acq=((rec->number_samples_saved+rec->new_samples_in_buffer)*com->number_channels);
	  // transfer the data
	  for(i=0;i<num_samples_to_transfer;i++)
	    {
	      for(j=0;j<rec->number_of_channels_to_save;j++)
		{
		  rec->rec_buffer_data[index_rec+j]=com->buffer_data[index_acq+rec->channel_list[j]];
		}
	      // move the rec_index and acq_index
	      index_rec+=rec->number_of_channels_to_save;
	      index_acq+=com->number_channels;
	    }
	}
      else // the comedi buffer has the data in a non-chronological order and first sample is not 0
	{
	  index_acq=(com->sample_no_to_add-num_samples_to_transfer)*com->number_channels;
	  // if larger or equal to 0, can get the data without without having to wrap the buffer
	  if(index_acq>=0)
	    {
	      // transfer the data
	      for(i=0;i<num_samples_to_transfer;i++)
		{
		  for(j=0;j<rec->number_of_channels_to_save;j++)
		    {
		      rec->rec_buffer_data[index_rec+j]=com->buffer_data[index_acq+rec->channel_list[j]];
		    }
		  // move the rec_index and acq_index
		  index_rec+=rec->number_of_channels_to_save;
		  index_acq+=com->number_channels;
		}
	    }
	  else // need to wrap the buffer
	    {
	      // get the data at the end of the buffer
	      num_samples_to_transfer_end=(0-index_acq)/com->number_channels;
	      index_acq=com->max_number_samples_in_buffer*com->number_channels+index_acq;
	      // transfer the data from the end of buffer
	      for(i=0;i<num_samples_to_transfer_end;i++)
		{
		  for(j=0;j<rec->number_of_channels_to_save;j++)
		    {
		      rec->rec_buffer_data[index_rec+j]=com->buffer_data[index_acq+rec->channel_list[j]];
		    }
		  // move the rec_index and acq_index
		  index_rec+=rec->number_of_channels_to_save;
		  index_acq+=com->number_channels;
		}
	      // get the data from the beginning of buffer
	      num_samples_to_transfer_start=num_samples_to_transfer-num_samples_to_transfer_end;
	      index_acq=0;
	      for(i=0;i<num_samples_to_transfer_start;i++)
		{
		  for(j=0;j<rec->number_of_channels_to_save;j++)
		    {
		      rec->rec_buffer_data[index_rec+j]=com->buffer_data[index_acq+rec->channel_list[j]];
		    }
		  // move the rec_index and acq_index
		  index_rec+=rec->number_of_channels_to_save;
		  index_acq+=com->number_channels;
		}
	    }
	}
      // update variable to know how much new data the rec buffer contains
      rec->new_samples_in_buffer+=num_samples_to_transfer;
    }
  // can now free the comedi buffer for other thread
  pthread_mutex_unlock( &mutex_comedi_interface_buffer );
#ifdef DEBUG_REC
  fprintf(stderr,"leaving recording_interface_get_data\n");
#endif

  return 0;
}

int recording_interface_save_data(struct recording_interface* rec)
{
  // save data when there is enough in buffer
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_save_data\n");
#endif
  int num_elements;
  num_elements=rec->new_samples_in_buffer*rec->number_of_channels_to_save;
  
  if (num_elements<=(int)(rec->rec_buffer_size*rec->proportion_buffer_filled_before_save))
    {
      return 0;
    }
  if(fwrite(rec->rec_buffer_data,sizeof(short int),num_elements, rec->file)!=num_elements)
    {
      fprintf(stderr,"problem saving data in recording_interface_save_data\n");
      fprintf(stderr,"recording should stop here\n");
      rec->is_recording=0;
      return -1;
    }
  rec->number_samples_saved+=rec->new_samples_in_buffer; // update count
#ifdef DEBUG_REC
  fprintf(stderr,"num_channels: %d, data point saved: %d , samples: %d, samples saved so far: %ld max_buff_s: %d\n",rec->number_of_channels_to_save,num_elements,rec->new_samples_in_buffer,rec->number_samples_saved,rec->max_samples_in_buffer);
#endif

  rec->new_samples_in_buffer=0;
  return 0;
}
int recording_interface_save_data_flush(struct recording_interface* rec)
{
  // save new samples in the file, no minimum number of samples, all is saved
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_save_data_flush\n");
#endif
  int num_elements;
  num_elements=rec->new_samples_in_buffer*rec->number_of_channels_to_save;
  
  if (num_elements<=0)
    {
      return 0;
    }
  if(fwrite(rec->rec_buffer_data,sizeof(short int),num_elements, rec->file)!=num_elements)
    {
      fprintf(stderr,"problem saving data in recording_interface_save_data\n");
      fprintf(stderr,"recording should stop here\n");
      rec->is_recording=0;
      return -1;
    }
  rec->number_samples_saved+=rec->new_samples_in_buffer; // update count
#ifdef DEBUG_REC
  fprintf(stderr,"num_channels: %d, data point saved: %d , samples: %d, samples saved so far: %ld max_buff_s: %d\n",rec->number_of_channels_to_save,num_elements,rec->new_samples_in_buffer,rec->number_samples_saved,rec->max_samples_in_buffer);
#endif
  rec->new_samples_in_buffer=0;
  return 0;
}

int recording_interface_close_file(struct recording_interface* rec)
{

#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_close_file\n");
#endif
  fclose(rec->file);
  return 0;
}

int recording_interface_check_file_size(struct recording_interface* rec)
{
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_check_file_size\n");
#endif
  
  // get the size of the recorded file
  fseek(rec->file,0,SEEK_END);
  rec->file_size=ftell(rec->file);
  //  fseek(rec->file,0,ios::beg);
  
  // check if the size has the predicted size
  rec->predicted_file_size=(sizeof(short int)*rec->number_of_channels_to_save*rec->number_samples_saved);
  if(rec->predicted_file_size!=rec->file_size)
    {
      fprintf(stderr,"recording_interface_check_file_size(), the size of the recorded file is not what it should be, problem opening file\n");
      fprintf(stderr,"size predicted: %ld, observed size: %ld\n",rec->file_size,rec->predicted_file_size);
      return -1;
    }
  else
    {
#ifdef DEBUG_REC
      fprintf(stderr,"size predicted: %ld, observed size: %ld\n",rec->file_size,rec->predicted_file_size);
#endif
    }
  return 0;
}


int recording_interface_clear_current_recording_variables(struct recording_interface* rec)
{
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_clear_current_recording_variables\n");
#endif

  // this reset some variables to 0. called at the end of a recording 
  rec->number_samples_saved=0;
  rec->new_samples_in_buffer=0;

#ifdef DEBUG_REC
  fprintf(stderr,"rec->number_samples_saved: %ld, rec->new_samples_in_buffer: %d\n",rec->number_samples_saved,rec->new_samples_in_buffer);
#endif
  return 0;
}

int recording_interface_get_channels_from_recording_channel_store(struct recording_interface* rec,struct comedi_interface* com, GtkListStore  *store)
{
  // function to update the channel list of the recording interface from
  // the GtkListStore that is modified by user in the preferences dialog
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_get_channels_from_recording_channel_store\n");
#endif

  GtkTreeIter iter;
  gboolean enabled;
  int channel_index;
  int i,j;
  gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);
  rec->number_of_channels_to_save=0;
  for (i=0; i < com->number_devices;i++)
    {
      for (j=0; j < com->dev[i].number_channels_analog_input; j++)
	{
	  gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, REC_COL_NO, &channel_index, REC_COL_SELECT, &enabled, -1);
	  if (enabled)
	    {
	      rec->channel_list[rec->number_of_channels_to_save]=channel_index;
	      rec->number_of_channels_to_save++;
	    }
	  gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter);
	}
    }
  
  if(rec->number_of_channels_to_save>RECORDING_MAXIMUM_CHANNELS)
    {
      fprintf(stderr,"rec->number_of_channels_to_save > RECORDING_MAXIMUM_CHANNELS in recording_interface_get_channels_from_recording_channel_store()\n");
      return -1;
    }

  // need to reset some variables that depends on the number of channels to record
  rec->max_samples_in_buffer=rec->rec_buffer_size/rec->number_of_channels_to_save;

#ifdef DEBUG_REC
  fprintf(stderr,"rec->number_of_channels_to_save: %d, rec->max_samples_in_buffer: %d\n", rec->number_of_channels_to_save,rec->max_samples_in_buffer);
#endif

  return 0;
}

int recording_interface_set_recording_time(struct recording_interface* rec, double recording_time_sec)
{
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_set_recording_time()\n");
#endif

  if(recording_time_sec < 0)
    {
      fprintf(stderr,"recording_time_sec < 0 in recording_interface_set_recording_time()\n");
      return 1;
    }
  if(rec->is_recording==1)
    {
      fprintf(stderr,"attempt to change recording time while recording\n");
      return 1;
    }
  rec->recording_time_sec=recording_time_sec;
#ifdef DEBUG_REC
  fprintf(stderr,"recording time set to %lf sec\n",recording_time_sec);
#endif

  return 0;
}

int recording_interface_set_recording_channels(struct recording_interface* rec, struct comedi_interface* com, int number_channels, int* channel_list)
{
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_set_recording_channels()\n");
#endif
  int i;
  if (number_channels <= 0 || number_channels > RECORDING_MAXIMUM_CHANNELS)
    {
      fprintf(stderr,"number_channels is smaller than 0 or larger than %i in recording_interface_set_recording_channels\n", RECORDING_MAXIMUM_CHANNELS);
      return 1;
    }
  for (i=0;i < number_channels; i++)
    {
      if(channel_list[i]<0 || channel_list[i]>=com->number_channels)
	{
	  fprintf(stderr,"a channel is out of range in recording_interface_set_recording_channels\n");
	  fprintf(stderr,"channel no is %d, but should be from 0 to %d\n",channel_list[i],com->number_channels);
	  return 1;
	}
    }
  rec->number_of_channels_to_save=number_channels;
  for (i=0;i < number_channels; i++)
    {
      rec->channel_list[i]=channel_list[i];
    }
#ifdef DEBUG_REC
  fprintf(stderr,"list of %d channels set in recording_interface_set_recording_channels()\n",number_channels);
#endif
  return 0;
}

int recording_interface_set_data_file_name(struct recording_interface* rec, char* data_file_name)
{
#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_set_data_file_name()\n");
#endif
  if (data_file_name==NULL)
    {
      fprintf(stderr,"data_file_name == NULL in recording_interface_set_data_file_name\n");
      return 1;
    }
  
  rec->file_name=g_strdup_printf("%s",data_file_name);
  
#ifdef DEBUG_REC
    fprintf(stderr,"recording data file name set to %s\n",rec->file_name);
#endif
  return 0;
}


void * recording(void * recording_inter)
{
  // function that is run by the recording_thread
#ifdef DEBUG_REC
  fprintf(stderr,"void * recording(void * recording_inter\n");
#endif
  
  struct recording_interface *rec;
  rec=(struct recording_interface *) recording_inter;
  void * pt=NULL;
  rec->recording_thread_running=1;
  rec->is_recording=1; // this will be set to 0 in a different thread when we want to stop the recording
  clock_gettime(CLOCK_REALTIME, &rec->start_recording_time_timespec);
  while(rec->is_recording==1)
    {
      if(recording_interface_get_data(rec, &comedi_inter)!=0)
	{
	  fprintf(stderr,"recording(): problem getting new data for recording, recording aborted\n");
	  rec->is_recording=0;
	  recording_interface_close_file(rec);
	  recording_interface_clear_current_recording_variables(rec);
	  rec->recording_thread_running=0;
	  return pt;
	}
      if(recording_interface_save_data(rec)!=0)
	{
	  fprintf(stderr,"recording(): problem saving data, recording aborted\n");
	  rec->is_recording=0;
	  recording_interface_close_file(rec);
	  recording_interface_clear_current_recording_variables(rec);
	  rec->recording_thread_running=0;
	  return pt;
	}
      // sleep a bit between getting data so that you get a decent amount each time
      clock_gettime(CLOCK_REALTIME, &rec->now_timespec);
      rec->duration_recording_timespec=diff(&rec->start_recording_time_timespec,&rec->now_timespec);
      nanosleep(&rec->inter_recording_sleep_timespec,&rec->req);
    }
  if(recording_interface_save_data_flush(rec)!=0)
    {
      fprintf(stderr,"recording(): problem with recording_interface_save_data_flush() in void * recording(void *recording_interface)\n");
    }
  
  
#ifdef DEBUG_REC
  fprintf(stderr,"exiting void * recording(void * recording_inter\n");
#endif
  rec->recording_thread_running=0;
  pthread_exit(NULL);
}

int recording_interface_start_recording(struct recording_interface* rec)
{

#ifdef DEBUG_REC
  fprintf(stderr,"recording_interface_start_recording\n");
#endif
  if (rec->is_recording==1)
    {
      fprintf(stderr,"recording already running in recording_interface_start_recording\n");
      return -1;
    }
  if (rec->recording_thread_running==1)
    {
      fprintf(stderr,"recording thread already running in recording_interface_start_recording\n");
      return -1;
    }
  recording_interface_clear_current_recording_variables(rec);

  if (recording_interface_open_file(rec)!=0)
    {
      fprintf(stderr,"recording(): unable to open file, in recording_interface_start_recording\n");
      return -1 ;
    }
  if((recording_thread_id=pthread_create(&recording_thread, NULL, recording,(void*)rec))==1)
    {
      fprintf(stderr,"error creating the recording thread, error_no:%d\n",
	      recording_thread_id);
      return -1;
    }
  return 0;
}
int recording_interface_stop_recording(struct recording_interface* rec)
{
  if (rec->is_recording==0)
    {
      fprintf(stderr,"rec->is_recording is already set to 0 in recording_interface_stop_recording\n");
    }
  
  if (rec->recording_thread_running==0)
    {
      fprintf(stderr,"rec->recording_thread_running is already set to 0 in recording_interface_stop_recording\n");
    }
  
  rec->is_recording=0; // will tell the recording thread to stop
  while(rec->recording_thread_running==1) // wait untill recording thread is over
    {
      nanosleep(&rec->inter_recording_sleep_timespec,&rec->req);
    }
  
  if(recording_interface_check_file_size(rec)!=0)
    {
      fprintf(stderr,"recording(): problem with save of the recorded file, in recording_interface_stop_recording\n");
    }

  recording_interface_close_file(rec);
  recording_interface_clear_current_recording_variables(rec);
  return 0;
}
