/* 
Copyright (C) 2010 Kevin Allen

This file is part of kacq.

kacq is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

kacq is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with kacq.  If not, see <http://www.gnu.org/licenses/>.


*/

#include "main.h"

int init_window()
{
  // function to complete the interface built with glade
  builder = gtk_builder_new ();
  char* glade_file_name;
  char* file_name = "kacq_sed.glade";
  glade_file_name=g_strdup_printf("%s/%s",DATADIR,file_name);
  if(gtk_builder_add_from_file (builder, glade_file_name, NULL)==0)
    {
      fprintf(stderr,"An error occurred reading kacq.glade\n" );
      return -1;
    }
  // get a reference for the widget we need to play with
  widgets.window = GTK_WIDGET (gtk_builder_get_object (builder, "window"));
  widgets.vbox1 = GTK_WIDGET (gtk_builder_get_object (builder, "vbox1"));
  widgets.about_dlg= GTK_WIDGET (gtk_builder_get_object (builder, "about_dialog"));
  widgets.preferences_dlg= GTK_WIDGET (gtk_builder_get_object (builder, "preferences_dialog"));
  widgets.toolbar= GTK_WIDGET (gtk_builder_get_object (builder, "toolbar"));
  widgets.dev1_name_label= GTK_WIDGET (gtk_builder_get_object (builder, "device_1_name_label"));
  widgets.dev2_name_label = GTK_WIDGET (gtk_builder_get_object (builder, "device_2_name_label"));
  widgets.dev1_driver_label= GTK_WIDGET (gtk_builder_get_object (builder, "device_1_driver_label"));
  widgets.dev2_driver_label= GTK_WIDGET (gtk_builder_get_object (builder, "device_2_driver_label"));
  widgets.num_devices_detected_label= GTK_WIDGET (gtk_builder_get_object (builder, "num_device_detected_label"));
  widgets.num_available_channels_label=GTK_WIDGET (gtk_builder_get_object (builder, "num_available_channels_label"));
  widgets.sampling_rate_spinbutton= GTK_WIDGET (gtk_builder_get_object (builder, "sampling_rate_spinbutton"));
  widgets.num_channels_device_1_label=GTK_WIDGET (gtk_builder_get_object (builder, "num_channels_device_1_label"));
  widgets.num_channels_device_2_label=GTK_WIDGET (gtk_builder_get_object (builder, "num_channels_device_2_label"));
  widgets.range_label=GTK_WIDGET (gtk_builder_get_object (builder, "range_label"));
  widgets.current_saving_directory_label2=GTK_WIDGET (gtk_builder_get_object (builder, "current_saving_directory_label2"));
  widgets.preferences_channel_vbox=GTK_WIDGET (gtk_builder_get_object (builder, "preferences_channel_vbox"));
  widgets.file_name_entry=GTK_WIDGET(gtk_builder_get_object (builder, "file_name_entry"));
  widgets.trial_spinbutton=GTK_WIDGET(gtk_builder_get_object (builder, "trial_spinbutton"));
  widgets.group_spinbutton=GTK_WIDGET(gtk_builder_get_object (builder, "group_spinbutton"));
  widgets.statusbar=GTK_WIDGET(gtk_builder_get_object (builder, "statusbar"));
  widgets.preferences_channel_vbox=GTK_WIDGET(gtk_builder_get_object (builder, "preferences_channel_vbox"));
  widgets.recording_channel_view=GTK_WIDGET(gtk_builder_get_object (builder, "rec_treeview"));
  widgets.oscilloscope_all_channels_view=GTK_WIDGET(gtk_builder_get_object (builder, "osc_all_channels_treeview"));
  widgets.sampling_rate_adjustment=GTK_ADJUSTMENT(gtk_builder_get_object(builder,"sampling_rate_adjustment"));
  widgets.osc_group_adjustment=GTK_ADJUSTMENT(gtk_builder_get_object(builder,"osc_group_adjustment"));
  widgets.trial_no_adjustment=GTK_ADJUSTMENT(gtk_builder_get_object(builder,"trial_no_adjustment"));
  widgets.drawing_area=GTK_WIDGET(gtk_builder_get_object (builder, "drawing_area"));
  widgets.time_decrease_toolbutton=GTK_WIDGET(gtk_builder_get_object (builder, "time_decrease_toolbutton"));
  widgets.time_increase_toolbutton=GTK_WIDGET(gtk_builder_get_object (builder, "time_increase_toolbutton"));
  widgets.gain_decrease_toolbutton=GTK_WIDGET(gtk_builder_get_object (builder, "gain_decrease_toolbutton"));
  widgets.gain_increase_toolbutton=GTK_WIDGET(gtk_builder_get_object (builder, "gain_increase_toolbutton"));

  // put a gtkdatabox in the vbox
  gtk_adjustment_configure(widgets.sampling_rate_adjustment,(gdouble)comedi_inter.sampling_rate,0,MAX_SAMPLING_RATE,1000,0,0);
  gtk_adjustment_configure(widgets.osc_group_adjustment,1,1,osc_inter.number_groups,1,0,0);
  gtk_adjustment_configure(widgets.trial_no_adjustment,1,1,1000,1,0,0);

  // so that comedi page of preferences dialog show the right labels
  set_device_labels();

  set_recording_labels();
  
  // need to set sampling rate adjustment and osc group adjustment
  
  // so that the channel page of the preferences dialog show the right labels
  build_recording_channel_tree_view();
  build_oscilloscope_all_channels_tree_view();
  
  gtk_builder_connect_signals (builder, NULL);          
  g_object_unref (G_OBJECT (builder));
  
  // show the main window
  gtk_widget_show (widgets.window);      
  return 0;
}
void create_and_fill_recording_model()
{
  GtkTreeIter    iter;
  int i, j, channel_index;
  widgets.recording_channel_store = gtk_list_store_new (REC_NUM_COLS, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN);
  channel_index=0;
  for (i=0; i < comedi_inter.number_devices;i++)
    {
      for (j=0; j < comedi_inter.dev[i].number_channels_analog_input; j++)
	{
	  // by default, all the channels are selected
	  gtk_list_store_append (widgets.recording_channel_store, &iter);
	  gtk_list_store_set (widgets.recording_channel_store, &iter,
			      REC_COL_NO, channel_index,
			      REC_COL_DEVICE, i,
			      REC_COL_SELECT, TRUE,
			      -1);
	  channel_index++; // increase the index for next channel
	}
    }
  return;
}
void cell_toggled_callback (GtkCellRendererToggle *cell,
			    gchar                 *path_string,
			    gpointer               user_data)
{
  // function to select and deselect channels in preferences dialog
  GtkTreeIter iter;
  gboolean enabled;
  GtkTreePath* path = gtk_tree_path_new_from_string(path_string);
  gtk_tree_model_get_iter(GTK_TREE_MODEL (widgets.recording_channel_store), &iter, path);
  gtk_tree_model_get(GTK_TREE_MODEL(widgets.recording_channel_store), &iter, REC_COL_SELECT, &enabled, -1);
  enabled = !enabled;
  gtk_list_store_set(widgets.recording_channel_store, &iter, REC_COL_SELECT, enabled, -1);
  return;
}

int build_recording_channel_tree_view()
{
  GtkCellRenderer     *renderer;
  GtkTreeSelection *selection;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(widgets.recording_channel_view));
  gtk_tree_selection_set_mode(selection,GTK_SELECTION_MULTIPLE);

  /* --- Column #1 --- */
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (widgets.recording_channel_view),
                                               -1,      
                                               "No",  
                                               renderer,
                                               "text", REC_COL_NO,
                                               NULL);
  /* --- Column #2 --- */
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (widgets.recording_channel_view),
                                               -1,      
                                               "Device",  
                                               renderer,
                                               "text", REC_COL_DEVICE,
                                               NULL);
  /* --- Column #3 --- */
  renderer=gtk_cell_renderer_toggle_new();
  g_object_set(renderer,"active", TRUE, NULL);
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (widgets.recording_channel_view),
					       -1,      
					       "Selected",  
					       renderer,
					       "active", REC_COL_SELECT,
                                                 NULL );
  g_signal_connect(renderer, "toggled", (GCallback) cell_toggled_callback, NULL);
  create_and_fill_recording_model ();
  gtk_tree_view_set_model(GTK_TREE_VIEW(widgets.recording_channel_view),GTK_TREE_MODEL(widgets.recording_channel_store));
  return 0;
}
int build_oscilloscope_all_channels_tree_view()
{
  GtkCellRenderer     *renderer;
  GtkTreeSelection *selection;
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(widgets.oscilloscope_all_channels_view));
  gtk_tree_selection_set_mode(selection,GTK_SELECTION_MULTIPLE);

  /* --- Column #1 --- */
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (widgets.oscilloscope_all_channels_view),
                                               -1,      
                                               "No",  
                                               renderer,
                                               "text", OSC_ALL_COL_NO,
                                               NULL);
  /* --- Column #2 --- */
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (widgets.oscilloscope_all_channels_view),
                                               -1,      
                                               "Device",  
                                               renderer,
                                               "text", OSC_ALL_COL_DEVICE,
                                               NULL);
  
  // g_signal_connect(renderer, "toggled", (GCallback) cell_toggled_callback, NULL);
  create_and_fill_oscilloscope_all_channels_model ();
  gtk_tree_view_set_model(GTK_TREE_VIEW(widgets.oscilloscope_all_channels_view),GTK_TREE_MODEL(widgets.oscilloscope_all_channels_store));
  return 0;
}

void create_and_fill_oscilloscope_all_channels_model()
{
  GtkTreeIter    iter;
  int i, j, channel_index;
  widgets.oscilloscope_all_channels_store=gtk_list_store_new (OSC_ALL_NUM_COLS, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_BOOLEAN);
  channel_index=0;
  for (i=0; i < comedi_inter.number_devices;i++)
    {
      for (j=0; j < comedi_inter.dev[i].number_channels_analog_input; j++)
	{
	  // by default, all the channels are selected
	  gtk_list_store_append (widgets.oscilloscope_all_channels_store, &iter);
	  gtk_list_store_set (widgets.oscilloscope_all_channels_store, &iter,
			      REC_COL_NO, channel_index,
			      REC_COL_DEVICE, i,
			      -1);
	  channel_index++; // increase the index for next channel
	}
    }
  return;
}




int set_device_labels()
{
  gchar *str;

  // sampling rate
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widgets.sampling_rate_spinbutton),comedi_inter.sampling_rate);
  // number channels
  str=g_strdup_printf("%d", comedi_inter.number_channels);
  gtk_label_set_text(GTK_LABEL(widgets.num_available_channels_label),str);


  if(comedi_inter.number_devices>0)
    {
      str=g_strdup_printf("%d", comedi_inter.number_devices);
      gtk_label_set_text(GTK_LABEL(widgets.num_devices_detected_label),str);
      gtk_label_set_text(GTK_LABEL(widgets.dev1_name_label),comedi_inter.dev[0].name);
      gtk_label_set_text(GTK_LABEL(widgets.dev1_driver_label),comedi_inter.dev[0].driver);
      str=g_strdup_printf("%d", comedi_inter.dev[0].number_channels_analog_input);
      gtk_label_set_text(GTK_LABEL(widgets.num_channels_device_1_label),str);
      
      str=g_strdup_printf("Range %d, from %lf to %lf Volt",comedi_inter.dev[0].range,comedi_inter.dev[0].range_input[comedi_inter.dev[0].range]->min,comedi_inter.dev[0].range_input[comedi_inter.dev[0].range]->max);
      gtk_label_set_text(GTK_LABEL(widgets.range_label),str);

    }
  if(comedi_inter.number_devices>1)
    {
      gtk_label_set_text(GTK_LABEL(widgets.dev2_name_label),comedi_inter.dev[1].name);
      gtk_label_set_text(GTK_LABEL(widgets.dev2_driver_label),comedi_inter.dev[1].driver);
      str=g_strdup_printf("%d", comedi_inter.dev[1].number_channels_analog_input);
      gtk_label_set_text(GTK_LABEL(widgets.num_channels_device_2_label),str);
    }
  else
    {
      gtk_label_set_text(GTK_LABEL(widgets.dev2_name_label),"None");
      gtk_label_set_text(GTK_LABEL(widgets.dev2_driver_label),"None");
    }
  g_free(str);
  return 0;
}

void set_recording_labels()
{
  gtk_label_set_text(GTK_LABEL(widgets.current_saving_directory_label2),recording_inter.directory);
  return;
}



// callback for the main window 
void on_window_destroy (GtkObject *object, gpointer user_data)
{
  // if a thread is running, kill it
  if(osc_inter.is_displaying==1)
    {
      osc_inter.is_displaying=0;
    }
  // if acquisition is running, kill it 
  if(comedi_inter.is_acquiring==1)
    {
      if(comedi_interface_stop_acquisition(&comedi_inter)==-1)
	{
	  fprintf(stderr,"problem stopping acquisistion in on_window_destroy\n");
	}
    }
  // if recording is running, kill it
  if (recording_inter.is_recording==1)
    {
      if(recording_interface_stop_recording(&recording_inter)==-1)
	{
	  fprintf(stderr,"problem stopping recording in on_window_destroy\n");
	}
    }
  gtk_main_quit();
} 

// callback for the about dialog
void on_about_dialog_delete_event(GtkObject *object, gpointer user_data)
{
  gtk_widget_hide(widgets.about_dlg);
}
// callback for the preferences dialog
void on_preferences_dialog_delete_event(GtkObject *object, gpointer user_data)
{
  gtk_widget_hide(widgets.preferences_dlg);
}

// callback for the menu item
void on_quit_menuitem_activate(GtkObject *object, gpointer user_data)
{
  gtk_main_quit();
} 
void on_about_menuitem_activate(GtkObject *object, gpointer user_data)
{
  int response;
  response=gtk_dialog_run(GTK_DIALOG(widgets.about_dlg));
  if ( response ==GTK_RESPONSE_CANCEL)
    {
      gtk_widget_hide(widgets.about_dlg);
    }
}
void on_preferences_menuitem_activate(GtkObject *object, gpointer user_data)
{
  int response;
  response=gtk_dialog_run(GTK_DIALOG(widgets.preferences_dlg));
  if ( response ==GTK_RESPONSE_CANCEL)
    {
      fprintf(stderr,"cancel preferences dlg\n");
      gtk_widget_hide(widgets.preferences_dlg);
    }
  if (response ==GTK_RESPONSE_OK)
    {
      if(comedi_inter.is_acquiring==0&&osc_inter.is_displaying==0&&recording_inter.is_recording==0)
	{
	  // reset the sampling rate only if there is no recording or oscilloscope in process
	  comedi_inter.sampling_rate=gtk_spin_button_get_value(GTK_SPIN_BUTTON(widgets.sampling_rate_spinbutton));
	  oscilloscope_interface_reset_current_variables(&osc_inter,&comedi_inter); // to adjust to new sampling rate
	  fprintf(stderr,"sampling rate: %d\n",comedi_inter.sampling_rate);
	  gtk_widget_hide(widgets.preferences_dlg);
	}
    }
}
void on_ok_preferences_button_activate(GtkObject *object, gpointer user_data)
{
  gtk_widget_hide(widgets.preferences_dlg);
  fprintf(stderr,"hide the preference dlg\n");
}

void on_rewind_toolbutton_clicked(GtkObject *object, gpointer user_data)
{
  oscilloscope_interface_show_previous_page(&osc_inter);
}

void on_play_toolbutton_toggled(GtkObject *object, gpointer user_data)
{
  
  // need to start the oscilloscope
  if(osc_inter.is_displaying==0)
    {

      // if acquisition not running, start it
      if(comedi_inter.is_acquiring==0)
	{
	  if(comedi_interface_start_acquisition(&comedi_inter)==-1)
	    {
	      fprintf(stderr,"could not start the acquisition in on_play_toolbutton_toggled()\n");
	      return;
	    }
	}
      if(oscilloscope_interface_reset_current_variables(&osc_inter, &comedi_inter)==-1)
	{
	  fprintf(stderr,"could not reset variables of oscilloscope_interface in on_play_toolbutton_toggled()\n");
	  return;
	}
      if(oscilloscope_interface_start_oscilloscope(&osc_inter)==-1)
	{
	  fprintf(stderr,"could not start oscilloscope in on_play_toolbutton_toggled()\n");
	  return;
	}
    }
  // need to stop the oscilloscope
  else
    {    
      if(oscilloscope_interface_stop_oscilloscope(&osc_inter)==-1)
	{
	  fprintf(stderr,"could not stop oscilloscope in on_play_toolbutton_toggled()\n");
	  return;
	}
      // if no recording in progress, stop acquisition
      if (recording_inter.is_recording==0)
	{
	  // start the acquisition
	  if(comedi_interface_stop_acquisition(&comedi_inter)==-1)
	    {
	      fprintf(stderr,"could not stop the acquisition in on_stop_toolbutton_clicked()\n");
	      return;
	    }
	}
    }
}
void on_forward_toolbutton_clicked(GtkObject *object, gpointer user_data)
{
  oscilloscope_interface_show_next_page(&osc_inter);
}
void on_record_toolbutton_toggled(GtkObject *object, gpointer user_data)
{
  int osc_flag;
  const gchar *str;
  gchar * str1;
  gchar * str2;
  gchar * str3;
  int index;
  // to know if we should restart oscilloscope at the end
  osc_flag=osc_inter.is_displaying;
  
  // if already recording, stop it
  if (recording_inter.is_recording==1)
    {
      if(recording_interface_stop_recording(&recording_inter)==-1)
	{
	  fprintf(stderr,"problem stoping recording thread in on_record_toolbutton_clicked\n");
	  return;
	}
      recording_inter.statusbar_context_id=gtk_statusbar_get_context_id(GTK_STATUSBAR(widgets.statusbar),"recording");
      gtk_statusbar_remove(GTK_STATUSBAR(widgets.statusbar),recording_inter.statusbar_context_id,recording_inter.statusbar_message_id);
        
      // increament the file index
      // get the file name from the gui
      index=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widgets.trial_spinbutton));
      index++;
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(widgets.trial_spinbutton),(gdouble)index);
      
      if (osc_inter.is_displaying==0)
	{
	  // if oscilloscope is not working, stop acquisition
	  if(comedi_interface_stop_acquisition(&comedi_inter)==-1)
	    {
	      fprintf(stderr,"problem stopping the acquisition thread in on_record_toolbutton_clicked\n");
	      return;
	    }
	}
      return ;
    }
  else
  // if recording is not already running, start the recording
    {
      // stop the oscilloscope if running
      if(osc_inter.is_displaying==1)
	{
	  if(oscilloscope_interface_stop_oscilloscope(&osc_inter)==-1)
	    {
	      fprintf(stderr,"problem stopping the oscilloscope thread in on_record_toolbutton_clicked\n");
	      return;
	    }
	}
      // stop the acquisition acquisition if running
      if(comedi_inter.is_acquiring==1)
	{
	  if(comedi_interface_stop_acquisition(&comedi_inter)==-1)
	    {
	      fprintf(stderr,"problem stopping the acquisition thread in on_record_toolbutton_clicked\n");
	      return;
	    }
	}
      // update the list of channels to record
      recording_interface_get_channels_from_recording_channel_store(&recording_inter,&comedi_inter,widgets.recording_channel_store);

      // get the file name from the gui
      str=gtk_entry_get_text(GTK_ENTRY(widgets.file_name_entry));
      str1=g_strdup_printf("%02d",gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widgets.trial_spinbutton)));
      str2=".dat";
      str3=g_strdup_printf("%s_%s%s",str,str1,str2);
      if(recording_interface_set_data_file_name(&recording_inter,g_strdup_printf("%s%s",recording_inter.directory,str3))==-1)
	{
	  fprintf(stderr,"problem setting the recording file name to %s in on_record_toolbutton_clicked\n",g_strdup_printf("%s%s",recording_inter.directory,str3));
	  return;
	}

      // check if the file already exist and warn the user if so
      struct stat st;
      if(stat(g_strdup_printf("%s%s",recording_inter.directory,str3),&st) == 0)
	{
	  // the file exist, start a dialog to get a confirmation before overwritting the file
	  //printf("%s is present\n",g_strdup_printf("%s%s",recording_inter.directory,str3));
	   GtkWidget *dialog, *label, *content_area;
	   gint result;

	   dialog = gtk_message_dialog_new (GTK_WINDOW(widgets.window),
					    GTK_DIALOG_DESTROY_WITH_PARENT,
					    GTK_MESSAGE_QUESTION,
					    GTK_BUTTONS_YES_NO,
					    "\n%s%s already exists.\nDo you want to overwrite it?",recording_inter.directory,str3);
	   gtk_widget_show_all (dialog);
	   result=gtk_dialog_run (GTK_DIALOG (dialog));
	   if(result==GTK_RESPONSE_NO)
	     {
	       // abort recording to avoid overwriting the file
	       gtk_widget_destroy (dialog);
	       return;
	     }
	   else
	     {
	       gtk_widget_destroy (dialog);
	     }
	}
      

      // set the statusbar to recording details
      str=g_strdup_printf("Recording from %d channels at %d Hz, saved in %s",recording_inter.number_of_channels_to_save, comedi_inter.sampling_rate, recording_inter.file_name);
      recording_inter.statusbar_context_id=gtk_statusbar_get_context_id(GTK_STATUSBAR(widgets.statusbar),"recording");
      recording_inter.statusbar_message_id=gtk_statusbar_push(GTK_STATUSBAR(widgets.statusbar),recording_inter.statusbar_context_id,str);
      

      if(recording_interface_start_recording(&recording_inter)==-1)
	{
	  fprintf(stderr,"problem starting the recording in on_record_toolbutton_clicked\n");
	  return;
	}

      if(osc_flag==1) // the oscilloscope was running, restart it
	{
	  // reset some variables in the osc_inter
	  if (oscilloscope_interface_reset_current_variables(&osc_inter, &comedi_inter)==-1)
	    {
	      fprintf(stderr,"could not reset variables of oscilloscope_interface\n");
	      return;
	    }
	  if(oscilloscope_interface_start_oscilloscope(&osc_inter)==-1)
	    {
	      fprintf(stderr,"could not start oscilloscope in on_record_toolbutton_clicked\n");
	      return;
	    }
	}
      if(comedi_interface_start_acquisition(&comedi_inter)==-1)
	{
	  fprintf(stderr,"error start the acquisition in on_record_toolbutton_clicked\n");
	  return;
	}
      
    }
}

void on_drawing_area_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
  // on expose, redraw the grid and the information regarding channels
  cairo_t *cr;
  cr = gdk_cairo_create (gtk_widget_get_window(widget));
  cairo_rectangle (cr, event->area.x,     event->area.y,
		   event->area.width, event->area.height);
  cairo_clip (cr);
  oscilloscope_interface_draw_grid(&osc_inter,widget,cr);
  cairo_destroy (cr);
  
}



void on_gain_increase_toolbutton_clicked(GtkObject *object, gpointer user_data)
{
  oscilloscope_interface_increase_global_gain(& osc_inter);
}
void on_gain_decrease_toolbutton_clicked(GtkObject *object, gpointer user_data)
{
  oscilloscope_interface_decrease_global_gain(& osc_inter);
}
void on_time_increase_toolbutton_clicked(GtkObject *object, gpointer user_data)
{
  oscilloscope_interface_increase_time_resolution(& osc_inter,&comedi_inter);
}
void on_time_decrease_toolbutton_clicked(GtkObject *object, gpointer user_data)
{
  oscilloscope_interface_decrease_time_resolution(& osc_inter,&comedi_inter);
}
void on_group_spinbutton_value_changed(GtkObject *object, gpointer user_data)
{
  osc_inter.gui_current_group=gtk_spin_button_get_value(GTK_SPIN_BUTTON(widgets.group_spinbutton))-1;
  oscilloscope_interface_update_group(&osc_inter,&comedi_inter);
}

struct timespec set_timespec_from_ms(double milisec)
{ // set the values in timespec structure
  struct timespec temp;
  time_t sec=(int)(milisec/1000);
  milisec=milisec-(sec*1000);
  temp.tv_sec=sec;
  temp.tv_nsec=milisec*1000000L;
  return temp;
}
struct timespec diff(struct timespec* start, struct timespec* end)
{
  // get the time difference between two times
  struct timespec temp;
  if ((end->tv_nsec-start->tv_nsec)<0) {
    temp.tv_sec = end->tv_sec-start->tv_sec-1;
    temp.tv_nsec = 1000000000+end->tv_nsec-start->tv_nsec;
  } 
  else {
    temp.tv_sec = end->tv_sec-start->tv_sec;
    temp.tv_nsec = end->tv_nsec-start->tv_nsec;
  }
  return temp;
}
int microsecond_from_timespec(struct timespec* duration)
{
  int ms;
  ms=duration->tv_nsec/1000;
  //  ms=ms+duration.tv_sec*1000;
  return ms;
}

void on_directory_button_clicked(GtkObject *object, gpointer user_data)
{
  GtkWidget *dialog;
  dialog = gtk_file_chooser_dialog_new ("Select a directory to save data files",
					GTK_WINDOW(widgets.window),
					GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					NULL);

  // select the current directory as a starting point
  gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog),
				recording_inter.directory);

  // show the dialog
 if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      // set the new directory
      recording_inter.directory=g_strdup_printf("%s/",gtk_file_chooser_get_filename((GTK_FILE_CHOOSER (dialog))));
      gtk_label_set_text(GTK_LABEL(widgets.current_saving_directory_label2),recording_inter.directory);
    }
  gtk_widget_destroy (dialog);
  return; 
}
void on_add_toolbutton_clicked(GtkObject *object, gpointer user_data)
{
  GtkTreeSelection  *selection;
  selection= gtk_tree_view_get_selection(GTK_TREE_VIEW(widgets.recording_channel_view));
  gtk_tree_selection_selected_foreach(selection, view_selected_add_foreach_func, NULL);

  return; 
}
void on_remove_toolbutton_clicked(GtkObject *object, gpointer user_data)
{
 GtkTreeSelection  *selection;
  selection=gtk_tree_view_get_selection(GTK_TREE_VIEW(widgets.recording_channel_view));
  gtk_tree_selection_selected_foreach(selection, view_selected_remove_foreach_func, NULL);

  return; 
}
void view_selected_add_foreach_func (GtkTreeModel  *model,
					 GtkTreePath   *path,
					 GtkTreeIter   *iter,
					 gpointer       userdata)
{
  gtk_list_store_set(GTK_LIST_STORE(model),iter,REC_COL_SELECT, TRUE, -1);
}
void view_selected_remove_foreach_func(GtkTreeModel  *model,
					   GtkTreePath   *path,
					   GtkTreeIter   *iter,
					   gpointer       userdata)
{
  gtk_list_store_set(GTK_LIST_STORE(model),iter,REC_COL_SELECT, FALSE, -1);
}

int read_configuration_file(char * file_name,struct comedi_interface* com, struct recording_interface* rec)
{
  /*
     function to set the comedi_interface and recording_interface from the configuration file given as option argument
     configuration file has the following format

     file_name
     comedi_sampling_rate
     recording_time_sec
     chan 1
     chan 2
     chan 3
     etc...
  */
  FILE* fp;
  size_t len = 0;
  ssize_t read;
  char data_file_name[255];
  int num_channels=0;
  int sampling_rate;
  double recording_sec;
  int channel_list[RECORDING_MAXIMUM_CHANNELS];
  int i,ret;

  fp = fopen(file_name, "r");
  if (fp == NULL)
    {
      fprintf(stderr,"problem opening %s in read_configuration_file\n",file_name);
      return -1;
    }
  
  if(fscanf(fp,"%s",&data_file_name)!=1)
    {
      fprintf(stderr,"problem reading the data file name in read_configuration_file\n");
      return -1;
    }
  if(fscanf(fp,"%i",&sampling_rate)!=1)
    {
      fprintf(stderr,"problem reading sampling rate in read_configuration_file\n");
      return -1;
    }
  if(fscanf(fp,"%lf",&recording_sec)!=1)
    {
      fprintf(stderr,"problem reading recording_sec in read_configuration_file\n");
      return -1;
    }
  while((ret=fscanf(fp,"%i",&channel_list[num_channels]))!=EOF&&num_channels<RECORDING_MAXIMUM_CHANNELS)
    {
      num_channels++;
    }
  if(ret!=EOF)
    {
      fprintf(stderr,"there were more channels in %s than the maximum allowed (%d) in read_configuration_file\n",file_name,RECORDING_MAXIMUM_CHANNELS);
      return -1;
    }

  // try to set the sampling rate
  if(comedi_interface_set_sampling_rate(com,sampling_rate)==-1)
    {
      fprintf(stderr,"unable to set sampling rate in read_configuration_file\n");
      return -1;
    }
  
  // set the name of the recording file
  
  if(recording_interface_set_data_file_name(rec,data_file_name)==-1)
    {
      fprintf(stderr,"unable to set data file name in read_configuration_file\n");
      return -1;
    }
  
  
  // try to set the recording time
  if(recording_interface_set_recording_time(rec,recording_sec)==-1)
    {
      fprintf(stderr,"unable to set recording time in read_configuration_file\n");
      return -1;
    }
  
  // try to set the list of recording channels
  if(recording_interface_set_recording_channels(rec,com,num_channels, channel_list)==-1)
    {
      fprintf(stderr,"unable to set the list of recording channels in read_configuration_file\n");
      return -1;
    }
  
  fclose(fp); 
  return 0;
}

int recording_terminal_mode(struct comedi_interface* com,struct recording_interface* rec)
{
  
  struct timespec elapsed_rec;
  struct timespec beginning_rec;
  struct timespec now;
  
  if(recording_interface_start_recording(rec)==-1)
    {
      fprintf(stderr,"unable to start recording in recording_terminal_mode()\n");
      return -1;
    }


  if(comedi_interface_start_acquisition(com)==-1)
    {
      fprintf(stderr,"unable to start acquisition in recording_terminal_mode()\n");
      return -1;
    }
  


  printf("recording %d channels for %lf sec at %d Hz, saved in %s\n",rec->number_of_channels_to_save,rec->recording_time_sec,com->sampling_rate,rec->file_name);

  // loop and sleep until the recording time is up
  clock_gettime(CLOCK_REALTIME, &beginning_rec);
  clock_gettime(CLOCK_REALTIME, &now);
  elapsed_rec=diff(&beginning_rec,&now);
  while(elapsed_rec.tv_sec < rec->recording_time_sec) // loop until the trial is over
    {
      // sleep to waist some time
      nanosleep(&rec->inter_recording_sleep_timespec,&rec->req);
      clock_gettime(CLOCK_REALTIME, &now);
      elapsed_rec=diff(&beginning_rec,&now);
    }

  if(recording_interface_stop_recording(rec)==-1)
    {
      fprintf(stderr,"problem with recording_interface_stop_recording in recording_terminal_mode()\n");
      return -1;
    }

  if(comedi_interface_stop_acquisition(com)==-1)
    {
      fprintf(stderr,"problem with comedi_interface_stop_acquisition in recording_terminal_mode()\n");
      return -1;
    }

  printf("recording completed sucessfully\n",rec->number_of_channels_to_save,rec->recording_time_sec);
  return 0;
}
