/***************************************************************
Copyright (C) 2010 Kevin Allen

This file is part of kacq.

kacq is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

kacq is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with kacq.  If not, see <http://www.gnu.org/licenses/>.



file with the functions to draw the signal as in an oscilloscope
***************************************************************/
#include "main.h"

int oscilloscope_interface_init(struct oscilloscope_interface* osc,struct comedi_interface* com)
{
  // return 
  int i;
  osc->is_displaying=0; // currently not running
  osc->is_drawing=0;
  osc->draw_only_mean=0;
  osc->number_channels=com->number_channels;
  osc->number_groups=OSCILLOSCOPE_GROUPS;
  osc->max_channels_per_group=MAX_CHANNELS_PER_GROUP;
  osc->buffer_size=MAX_BUFFER_LENGTH*COMEDI_INTERFACE_MAX_DEVICES*FACTOR_OSCILLATORY_BUFFER;

  if((osc->osc_buffer=malloc(sizeof(gfloat)*osc->buffer_size))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->osc_buffer\n");
      return -1;
    }
  
  osc->seconds_per_page=MIN_TIME_SEC_IN_OSCILLOSCOPE_PAGE; // set to lowest resolution by default
  osc->gui_seconds_per_page=OSCILLOSCOPE_DEFAULT_TIME_SEC_IN_PAGE;
  osc->samples_per_page=com->sampling_rate*osc->seconds_per_page;
  osc->page_size=osc->samples_per_page*osc->number_channels;
  osc->number_pages_in_buffer=osc->buffer_size/(osc->number_channels*osc->samples_per_page);
  osc->current_page=0;
  osc->pages_in_memory=0;
  osc->displayed_page=0;
  osc->pixels_per_data_point_to_draw=OSCILLOSCOPE_PIXELS_PER_DATA_POINT_TO_DRAW;
  osc->show_buffer_size=MAX_CHANNELS_PER_GROUP*MAX_SAMPLING_RATE*MAX_TIME_SEC_IN_OSCILLOSCOPE_PAGE; // size of the largest bunch of data put on screen at same time
  if((osc->show_buffer=malloc(sizeof(gfloat)*osc->show_buffer_size))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->show_buffer\n");
      return -1;
    }
  if (osc->samples_per_page>osc->buffer_size/osc->number_channels)
    {
      fprintf(stderr,"oscilloscope_interface_init: osc->samples_per_page>osc->buffer_size/osc->number_channels\n");
      return -1;
    }
  
  if((osc->x_axis_data=malloc(sizeof(gfloat)*MAX_SAMPLING_RATE*MAX_TIME_SEC_IN_OSCILLOSCOPE_PAGE))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->x_axis_data\n");
      return -1;
    }
  for (i=0; i<MAX_SAMPLING_RATE*MAX_TIME_SEC_IN_OSCILLOSCOPE_PAGE;i++)
    {
      osc->x_axis_data[i]=i;
    }
  osc->max_samples_in_buffer=osc->buffer_size/osc->number_channels;
  osc->new_samples_in_buffer=0; // undisplayed data
  osc->number_samples_displayed=0; // displayed data
  osc->inter_displaying_sleep_ms=20;
  osc->inter_displaying_sleep_timespec=set_timespec_from_ms(osc->inter_displaying_sleep_ms);
  osc->current_group=0;
  
  if((osc->number_channels_per_group=malloc(sizeof(int)*osc->number_groups))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->number_channels_per_group\n");
      return -1;
    }
  osc->global_gain=OSCILLOSCOPE_DEFAULT_GAIN;
  osc->gui_global_gain=OSCILLOSCOPE_DEFAULT_GAIN;
  osc->min_global_gain=OSCILLOSCOPE_MIN_GLOBAL_GAIN;
  osc->max_global_gain=OSCILLOSCOPE_MAX_GLOBAL_GAIN;
  osc->global_gain_factor=OSCILLOSCOPE_GLOBAL_GAIN_CHANGE_FACTOR;

  // allocate arrays of pointers for 2d arrays
  if((osc->channel_list_in_group=malloc(sizeof(int*)*osc->number_groups))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->channel_list_in_group\n");
      return -1;
    }
  if((osc->gain=malloc(sizeof(gfloat*)*osc->number_groups))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->gain\n");
      return -1;
    }
  if((osc->channel_red=malloc(sizeof(float*)*osc->number_groups))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->channel_color\n");
      return -1;	
    }
    if((osc->channel_green=malloc(sizeof(float*)*osc->number_groups))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->channel_color\n");
      return -1;	
    }

  if((osc->channel_blue=malloc(sizeof(float*)*osc->number_groups))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->channel_color\n");
      return -1;	
    }

  
  // allocate memory for the array related to each group
  for (i=0; i<osc->number_groups;i++)
    {
      if((osc->channel_list_in_group[i]=malloc(sizeof(int)*osc->max_channels_per_group))==NULL)
	{
	  fprintf(stderr,"problem allocating memory for osc->channel_list_in_group[%d]\n",i);
	  return -1;	
	}
      if((osc->gain[i]=malloc(sizeof(gfloat)*osc->max_channels_per_group))==NULL)
	{
	  fprintf(stderr,"problem allocating memory for osc->gain[%d]\n",i);
	  return -1;	
	}
      if((osc->channel_red[i]=malloc(sizeof(float)*osc->max_channels_per_group))==NULL)
	{
	  fprintf(stderr,"problem allocating memory for osc->channel_color[%d]\n",i);
	  return -1;	
	}
      if((osc->channel_green[i]=malloc(sizeof(float)*osc->max_channels_per_group))==NULL)
	{
	  fprintf(stderr,"problem allocating memory for osc->channel_color[%d]\n",i);
	  return -1;	
	}
      if((osc->channel_blue[i]=malloc(sizeof(float)*osc->max_channels_per_group))==NULL)
	{
	  fprintf(stderr,"problem allocating memory for osc->channel_color[%d]\n",i);
	  return -1;	
	}
    }

  if((osc->pointer_draw_data=malloc(sizeof(gfloat *)*osc->max_channels_per_group))==NULL) // array of gfloat pointers
    {
      fprintf(stderr,"problem allocating memory for osc->pointer_draw_data\n");
      return -1;	
    }
  oscilloscope_interface_set_default_group_values(osc);
  
  if((osc->y_min_for_pixel_x=malloc(sizeof(gfloat)*OSCILLOSCOPE_MAXIMUM_X_PIXEL_FOR_DRAWING_AREA))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->y_min_for_pixel_x\n");
      return -1;	
    }
  if((osc->y_max_for_pixel_x=malloc(sizeof(gfloat)*OSCILLOSCOPE_MAXIMUM_X_PIXEL_FOR_DRAWING_AREA))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->y_min_for_pixel_x\n");
      return -1;	
    }

  if((osc->mean_for_pixel_x=malloc(sizeof(gfloat)*OSCILLOSCOPE_MAXIMUM_X_PIXEL_FOR_DRAWING_AREA))==NULL)
    {
      fprintf(stderr,"problem allocating memory for osc->mean_for_pixel_x\n");
      return -1;	
    }


#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_init\n");
  fprintf(stderr,"osc buffer_size: %d, number pages: %d, number samples per page: %d, groups: %d, max chan group: %d\n",	  osc->buffer_size,osc->number_pages_in_buffer,osc->samples_per_page, osc->number_groups, osc->max_channels_per_group);
#endif

  return 0;
}
int oscilloscope_interface_free(struct oscilloscope_interface* osc)
{
#ifdef DEBUG_OSC
  fprintf(stderr,"entering oscilloscope_interface_free\n");
#endif
  int i;
  for (i=0; i<osc->number_groups;i++)
    {
      free(osc->channel_list_in_group[i]);
      free(osc->gain[i]);
      free(osc->channel_red[i]);
      free(osc->channel_green[i]);
      free(osc->channel_blue[i]);
    }
  free(osc->channel_list_in_group);
  free(osc->gain);
  free(osc->channel_red);
  free(osc->channel_green);
  free(osc->channel_blue);
  
  free(osc->osc_buffer);
  free(osc->show_buffer);
  free(osc->x_axis_data);
  free(osc->number_channels_per_group);
  free(osc->pointer_draw_data);
  free(osc->y_min_for_pixel_x);
  free(osc->y_max_for_pixel_x);
  free(osc->mean_for_pixel_x);
  return 0;
}

int oscilloscope_interface_reset_current_variables(struct oscilloscope_interface* osc,struct comedi_interface* com)
{
  osc->samples_per_page=com->sampling_rate*osc->seconds_per_page;
  osc->number_channels=com->number_channels;
  osc->page_size=osc->samples_per_page*osc->number_channels;
  osc->number_pages_in_buffer=osc->buffer_size/(osc->number_channels*osc->samples_per_page);
  osc->max_samples_in_buffer=osc->buffer_size/osc->number_channels;
  osc->current_page=0;
  osc->new_samples_in_buffer=0; // undisplayed data


  if(com->is_acquiring==1)
    {
      osc->number_samples_displayed=com->number_samples_read;
    }
  else
    {
      osc->number_samples_displayed=0; // displayed data 
    }
  osc->pages_in_memory=0;
  osc->displayed_page=0;
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_reset_current_variables\n");
  fprintf(stderr,"samples_per_page: %d, number_channels: %d, page_size: %d, number_pages_in_buffer: %d, max_samples_in_buffer: %d, current_page %d, new_samples_in_buffer: %d, number_samples_displayed: %ld\n",
	  osc->samples_per_page, osc->number_channels, osc->page_size, osc->number_pages_in_buffer, osc->max_samples_in_buffer, osc->current_page, osc->new_samples_in_buffer, osc->number_samples_displayed);
#endif

  return 0;
}

int oscilloscope_interface_update_time_gain(struct oscilloscope_interface* osc, struct comedi_interface* com)
{
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_update_time_gain\n");
#endif
  // this function applies changes made by the user to the oscilloscope
  // make sure that the changes are applied at the right time i.e. not when graphs are being made.
  if (osc->seconds_per_page!=osc->gui_seconds_per_page||osc->global_gain!=osc->gui_global_gain)
    {
      osc->global_gain=osc->gui_global_gain;
      osc->seconds_per_page=osc->gui_seconds_per_page;
      osc->samples_per_page=com->sampling_rate*osc->seconds_per_page;
      osc->page_size=osc->samples_per_page*osc->number_channels;
      osc->number_pages_in_buffer=osc->buffer_size/(osc->number_channels*osc->samples_per_page);
      osc->current_page=0;
      osc->new_samples_in_buffer=0;
      osc->number_samples_displayed=com->number_samples_read;
      osc->pages_in_memory=0;
      
      // need to adjust the timer

    }
  return 0;
}

int oscilloscope_interface_update_group(struct oscilloscope_interface* osc,struct comedi_interface* com)
{
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_update_group\n");
#endif
  if(osc->gui_current_group!=osc->current_group)
    {
      osc->current_group=osc->gui_current_group;
    }
  if(osc->is_displaying==0)
    { 
      oscilloscope_interface_refresh_display(osc);
    }
  return 0;
}

int oscilloscope_interface_draw_grid(struct oscilloscope_interface* osc, GtkWidget *widget, cairo_t *cr)
{

  
  int width, height,x_margin,i;
  double vertical_channel_space,horizontal_channel_space;
  
  // get the size of the drawing area
  gdk_drawable_get_size(gtk_widget_get_window(widget),&width, &height);
  x_margin=30;
      
  // get the vertical space allocated for each channel in the current group
  vertical_channel_space=(double)height/osc->number_channels_per_group[osc->current_group];
  horizontal_channel_space=width-x_margin;
      
  /* Set surface to opaque color (r, g, b) */
  cairo_set_source_rgb (cr, 0.9, 0.9, 0.9);
  cairo_paint (cr);
      
  // write the channel number on the left side of the drawing area
  cairo_select_font_face (cr, "serif", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL  );
  cairo_set_font_size (cr, 12.0);
  cairo_set_source_rgb (cr, 0, 0, 0);
  for (i=0;i<osc->number_channels_per_group[osc->current_group];i++)
    {
      cairo_move_to(cr, 5, (i*vertical_channel_space)+(vertical_channel_space/2));
      cairo_show_text (cr, g_strdup_printf("%d",osc->channel_list_in_group[osc->current_group][i]));
    }
      
  // write the number of sec per page
  cairo_move_to(cr,x_margin+(horizontal_channel_space/10*9.2),15);
  cairo_show_text (cr, g_strdup_printf("%.1lf ms",osc->seconds_per_page*1000));
      
  // draw an horizontal tic between each channel at the right and left of the oscilloscope
  cairo_set_source_rgb (cr, 0, 0, 0);
  cairo_set_line_width (cr, 1);
  for (i=1;i<osc->number_channels_per_group[osc->current_group];i++)
    {
      // left side
      cairo_move_to(cr, 0, i*vertical_channel_space);
      cairo_rel_line_to (cr, 5, 0);
      // right side
      cairo_move_to(cr, width-5, i*vertical_channel_space);
      cairo_rel_line_to (cr, 5, 0);
    }
  cairo_stroke (cr);
      
  // draw 9 vertical tics that will separate the oscilloscope region in 10 equal parts, at top and bottom
  for (i=0; i < 11; i++)
    {
      cairo_move_to (cr, x_margin+(i*horizontal_channel_space/10), 0);
      cairo_rel_line_to (cr, 0, 5);
      cairo_move_to (cr, x_margin+(i*horizontal_channel_space/10), height);
      cairo_rel_line_to (cr, 0, -5);
    }
  cairo_stroke (cr);
  return 0;
}

int oscilloscope_interface_fill_show_buffer(struct oscilloscope_interface* osc,int page)
{
  // function to fill the show_buffer
  int i,j;
  // move data used to make the display in the show buffer after applying gain, reversing polarity, etc
  for (i=0;i<osc->number_channels_per_group[osc->current_group];i++)
    {
      // point to the first channel of that page
      osc->pointer_draw_data[i]=osc->osc_buffer+(page*osc->samples_per_page*osc->number_channels)+(osc->channel_list_in_group[osc->current_group][i]*osc->samples_per_page);
      
      // copy in the show buffer, and apply gain and offset, reverse so that more negative goes down
      for (j=0;j<osc->samples_per_page;j++)
	{
	  osc->show_buffer[(osc->samples_per_page*i)+j]=0-(osc->pointer_draw_data[i][j]*osc->gain[osc->current_group][i]*osc->global_gain);
	}
    }
  return 0;
}

int oscilloscope_interface_show_data(struct oscilloscope_interface* osc,int page)
{
  
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_show_data, page %d\n",page);
#endif
  
  // to get an idea of drawing time
  struct timespec beginning_drawing,end_drawing,drawing_duration,data_crunch_end,data_crunch_duration,  rec; 
  int i,j,k;
  // do all the drawing here
  cairo_t * cr;
  cairo_t * buffer_cr;
  cairo_surface_t *buffer_surface;
  cairo_surface_t *drawable_surface;
  int width_start, height_start;
  int vertical_channel_space;
  int horizontal_channel_space;
  int x_margin;
  int start_index,end_index; // to get the min and max y value for each x coordinate from 0 to horizontal_channel_space
  double data_points_per_x_pixel;
  double pixels_per_data_point;
  int x_pixels_to_draw;
  clock_gettime(CLOCK_REALTIME, &beginning_drawing);
  if(osc->number_channels_per_group[osc->current_group]<1)
    {
      fprintf(stderr,"no channel to display in group %d\n",osc->current_group);
      return;
    }
  
  oscilloscope_interface_fill_show_buffer(osc,page);
   
  
  x_margin=30; // space on the left side to write channel number

  gdk_drawable_get_size(gtk_widget_get_window(widgets.drawing_area),&width_start, &height_start);
  if(width_start>OSCILLOSCOPE_MAXIMUM_X_PIXEL_FOR_DRAWING_AREA)
    {
      fprintf(stderr,"the width of the display is larger than OSCILLOSCOPE_MAXIMUM_X_PIXEL_FOR_DRAWING_AREA in oscilloscope_interface_show_data\n");
      return;
    }
 
  // get a cairo context to draw on drawing_area
  cr = gdk_cairo_create(gtk_widget_get_window(widgets.drawing_area));
  drawable_surface = cairo_get_target (cr);
  // we will draw in a buffer surface and then copy to drawing_area, for smoother oscilloscope
  buffer_surface= cairo_surface_create_similar(drawable_surface,
					       CAIRO_CONTENT_COLOR_ALPHA,
					       width_start,
					       height_start);
  buffer_cr=cairo_create(buffer_surface);
  
  // get the vertical space allocated for each channel in the current group
  vertical_channel_space=height_start/osc->number_channels_per_group[osc->current_group];
  horizontal_channel_space=width_start-x_margin;
  x_pixels_to_draw=horizontal_channel_space/osc->pixels_per_data_point_to_draw; // to reduce drawing time, and x resolution.
  data_points_per_x_pixel=(double)osc->samples_per_page/x_pixels_to_draw;
  pixels_per_data_point=(double)x_pixels_to_draw/osc->samples_per_page;

  // draw the grid and channel information
  oscilloscope_interface_draw_grid(osc,widgets.drawing_area,buffer_cr);
  // for each channel
  
  if(data_points_per_x_pixel>=1) // more than one data point per pixel in the screen
    {
      for (i=0;i<osc->number_channels_per_group[osc->current_group];i++)
	{
	  cairo_set_source_rgb (buffer_cr, osc->channel_red[osc->current_group][i],osc->channel_green[osc->current_group][i],osc->channel_blue[osc->current_group][i]);
	  // get the minimum y and maximum y and mean y for each x coordinate on the screen
	  for (j=0;j<x_pixels_to_draw;j++)
	    {
	      start_index=(int)round(j*data_points_per_x_pixel);
	      end_index=(int)round(start_index+data_points_per_x_pixel);
	      osc->y_min_for_pixel_x[j]=osc->show_buffer[(osc->samples_per_page*i)+start_index];
	      osc->y_max_for_pixel_x[j]=osc->show_buffer[(osc->samples_per_page*i)+start_index];
	      osc->mean_for_pixel_x[j]=0;
	      for (k=start_index;k<end_index;k++)
		{
		  if(osc->y_max_for_pixel_x[j]<osc->show_buffer[(osc->samples_per_page*i)+k])
		    {
		      osc->y_max_for_pixel_x[j]=osc->show_buffer[(osc->samples_per_page*i)+k];
		    }
		  if(osc->y_min_for_pixel_x[j]>osc->show_buffer[(osc->samples_per_page*i)+k])
		    {
		      osc->y_min_for_pixel_x[j]=osc->show_buffer[(osc->samples_per_page*i)+k];
		    }
		  osc->mean_for_pixel_x[j]+=osc->show_buffer[(osc->samples_per_page*i)+k];
		}
	      osc->mean_for_pixel_x[j]=osc->mean_for_pixel_x[j]/(end_index-start_index);
	    }
	  /*to draw the mean */
	  for (j=1;j<x_pixels_to_draw;j++)
	    {
	      if (j==0) 
	      	{
		  cairo_move_to(buffer_cr, x_margin+j*osc->pixels_per_data_point_to_draw,(int)((vertical_channel_space*i+vertical_channel_space/2)+(osc->mean_for_pixel_x[j])));
	      	}
	      else
	      	{
		  cairo_line_to(buffer_cr, x_margin+j*osc->pixels_per_data_point_to_draw,(int)((vertical_channel_space*i+vertical_channel_space/2)+(osc->mean_for_pixel_x[j])));
	      	}
	    }
	  if(osc->draw_only_mean==0)
	    {
	      /* draw from max to min */
	      for (j=1;j<x_pixels_to_draw;j++)
		{
		  cairo_move_to(buffer_cr, x_margin+j*osc->pixels_per_data_point_to_draw,(int)((vertical_channel_space*i+vertical_channel_space/2)+(osc->y_max_for_pixel_x[j])));
		  cairo_line_to(buffer_cr, x_margin+j*osc->pixels_per_data_point_to_draw,(int)((vertical_channel_space*i+vertical_channel_space/2)+(osc->y_min_for_pixel_x[j])));
		}
	    }
	  cairo_stroke (buffer_cr);
	}
    }
  else // less than a data point per pixel in the screen
    {
      
      for (i=0;i<osc->number_channels_per_group[osc->current_group];i++)
	{
	  cairo_set_source_rgb (buffer_cr, osc->channel_red[osc->current_group][i],osc->channel_green[osc->current_group][i],osc->channel_blue[osc->current_group][i]);
	  // get the minimum y and maximum y and mean y for each x coordinate on the screen
	  for (j=0;j<osc->samples_per_page;j++)
	    {
	      if (j==0) 
	      	{
		  cairo_move_to(buffer_cr, x_margin+j*pixels_per_data_point,(int)((vertical_channel_space*i+vertical_channel_space/2)+osc->show_buffer[(osc->samples_per_page*i)+j]));
	      	}
	      else
	      	{
		  cairo_line_to(buffer_cr, x_margin+j*pixels_per_data_point,(int)((vertical_channel_space*i+vertical_channel_space/2)+osc->show_buffer[(osc->samples_per_page*i)+j]));
	      	}
	    }
	  cairo_stroke (buffer_cr);
	}
    }
  
  // copy the buffer surface to the drawable widget
  cairo_set_source_surface (cr,buffer_surface,0,0);
  cairo_paint (cr);
  
  // delete the context
  cairo_destroy(cr);
  cairo_destroy(buffer_cr);
  cairo_surface_destroy(buffer_surface);
  clock_gettime(CLOCK_REALTIME, &end_drawing);
  drawing_duration=diff(&beginning_drawing,&end_drawing);
  
   
  // adjust the number of pixels drawn on the screen depending on the time it took to draw the data
  // if slower than the time in display, reduce number of pixels drawn
  
  
  if (drawing_duration.tv_sec*1000+drawing_duration.tv_nsec/1000000.0>osc->seconds_per_page*1000*1.20)
    {

      if (osc->draw_only_mean!=1)
	{
	  fprintf(stderr,"Drawing time for %d pixels: %.3lf(ms), set osc->draw_only_mean=1\n",drawing_duration.tv_sec*1000+drawing_duration.tv_nsec/1000000.0,x_pixels_to_draw);
	  osc->draw_only_mean=1;
	  
	  //      fprintf(stderr,"Drawing time for %d pixels: %.3lf(ms)\n",drawing_duration.tv_sec*1000+drawing_duration.tv_nsec/1000000.0,x_pixels_to_draw);
	  // if(osc->pixels_per_data_point_to_draw<4)
	  //	{
	  //  osc->pixels_per_data_point_to_draw=osc->pixels_per_data_point_to_draw*2;
	  //  fprintf(stderr,"osc->pixels_per_data_point_to_draw:%d\n",osc->pixels_per_data_point_to_draw);
	  //	}
	}
    }
  
  if (drawing_duration.tv_sec*1000+drawing_duration.tv_nsec/1000000.0<(osc->seconds_per_page*1000)*.50)
    {
      if(osc->draw_only_mean!=0)
	{
	  fprintf(stderr,"Drawing time for %d pixels: %.3lf(ms), set osc->draw_only_mean=0\n",drawing_duration.tv_sec*1000+drawing_duration.tv_nsec/1000000.0,x_pixels_to_draw);
	  osc->draw_only_mean=0;
	  //      if(osc->pixels_per_data_point_to_draw>=2)
	  //	{
	  //  osc->pixels_per_data_point_to_draw=osc->pixels_per_data_point_to_draw/2;
	  //  fprintf(stderr,"osc->pixels_per_data_point_to_draw:%d\n",osc->pixels_per_data_point_to_draw);
	  //	}
	}
    }
  
  
  // set which page is being displayed
  osc->displayed_page=page;
  return 0;
}

int oscilloscope_interface_refresh_display(struct oscilloscope_interface* osc)
{
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_refresh_display\n");
#endif
  oscilloscope_interface_show_data(osc,osc->displayed_page);
  return 0;
}
int oscilloscope_interface_get_data(struct oscilloscope_interface* osc, struct comedi_interface* com)
{
  
  /* this function takes new available samples from the comedi_interface and 
     put them into the oscilloscope_interface buffer 
  */
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_get_data\n");
#endif

  int index_osc,index_acq,num_samples_to_transfer,i,j,num_samples_to_transfer_end, num_samples_to_transfer_start,diff;
  // prevent other threads from changing comedi_interface_buffer
  pthread_mutex_lock( &mutex_comedi_interface_buffer );

#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_get_data, within mutex\n");
#endif
  // check how many samples are available, never take more than what is needed to fill a page
  // this maximum of samples_to_transfer make sure we don't wrap around the buffer, which
  // makes the transfer operation more complicated. Other advantage is that the thread is back to show the data sooner
  
  // check if the comedi device has new samples that are not in current recording  buffer
  if(com->number_samples_read>osc->new_samples_in_buffer+osc->number_samples_displayed)
    {
      // by default, assume that it reads all what is in the acq buffer
      num_samples_to_transfer=com->number_samples_read-(osc->number_samples_displayed+osc->new_samples_in_buffer);
      osc->acq_last_sample_no_needed=com->sample_no_to_add;

      // check if the data needed are still in acq buffer
      if(num_samples_to_transfer>com->max_number_samples_in_buffer)
	{
	  fprintf(stderr,"oscilloscope_interface_get_data would need %d samples, but there are only %d in acq buffer\n",num_samples_to_transfer,com->max_number_samples_in_buffer);
	  fprintf(stderr,"time between calls to oscilloscope_interface_get_data is probably to long\n");
	  pthread_mutex_unlock( &mutex_comedi_interface_buffer );
	  return -1;
	}
      
      // if more data in acq buffer to fill one osc page, get only what needed to fill one osc page
      if (num_samples_to_transfer+osc->new_samples_in_buffer>osc->samples_per_page)
	{
	  // diff is the number of samples that we don't want to transfer from the end of the acq buffer, it is a negative value
	  // when we need to transfer fewer samples that what is available in acq buffer
	  diff=osc->samples_per_page-(num_samples_to_transfer+osc->new_samples_in_buffer);
	  num_samples_to_transfer=osc->samples_per_page-osc->new_samples_in_buffer;

	  // need to get the data relative to the last sample no needed, not last data loaded in acq buffer
	  osc->acq_last_sample_no_needed=com->sample_no_to_add+diff;
	  if(osc->acq_last_sample_no_needed<0)
	    {
	      osc->acq_last_sample_no_needed=com->max_number_samples_in_buffer+osc->acq_last_sample_no_needed; 
	    }
	}
#ifdef DEBUG_OSC
      fprintf(stderr,"new_samples_in_buffer:%d, number_samples_to_transfer: %d new+transfer: %d\n",osc->new_samples_in_buffer,num_samples_to_transfer,osc->new_samples_in_buffer+num_samples_to_transfer);
      fprintf(stderr,"osc->acq_last_sample_no_needed:%d, com->sample_no_to_add: %d diff:%d\n",osc->acq_last_sample_no_needed,com->sample_no_to_add, diff);
#endif
      
      // get a pointer to the beginning of the current oscilloscope page
      osc->page_pointer=osc->osc_buffer+(osc->current_page*osc->samples_per_page*osc->number_channels);

      // index in the acq buffer to start reading
      index_acq=(osc->acq_last_sample_no_needed-num_samples_to_transfer)*com->number_channels;

      // if index larger or equal to 0, can get the data without without having to wrap the buffer
      if(index_acq>=0)
	{
	  for(i=0;i<num_samples_to_transfer;i++)
	    {
	      for(j=0;j<osc->number_channels;j++)
		{
		  index_osc=(j*osc->samples_per_page)+osc->new_samples_in_buffer+i;
		  if(index_osc>osc->buffer_size)
		    {
		      fprintf(stderr,"oscilloscope_interface_get_data, no wrapping: index_osc>osc-buffer_size\n");
		      fprintf(stderr,"%d %d\n",index_osc,osc->buffer_size);
		      pthread_mutex_unlock( &mutex_comedi_interface_buffer );
		      return -1;
		    }
		  osc->page_pointer[index_osc]=com->buffer_data[index_acq+j];
		}
	      // move the rec_index and acq_index
	      index_acq+=com->number_channels;
	    }
	}
      else // need to wrap the buffer
	{
	  // get the data at the end of the buffer
	  num_samples_to_transfer_end=(0-index_acq)/com->number_channels;
	  index_acq=com->max_number_samples_in_buffer*com->number_channels+index_acq; // index_acq is negative
	  // transfer the data from the end of buffer
	  for(i=0;i<num_samples_to_transfer_end;i++)
	    {
	      for(j=0;j<osc->number_channels;j++)
		{
		  index_osc=(j*osc->samples_per_page)+osc->new_samples_in_buffer+i;
		  if(index_osc>osc->buffer_size)
		    {
		      fprintf(stderr,"oscilloscope_interface_get_data, wrapping end: index_osc>osc-buffer_size\n");
		      fprintf(stderr,"%d %d\n",index_osc,osc->buffer_size);
		      pthread_mutex_unlock( &mutex_comedi_interface_buffer );
		      return -1;
		    }
		  osc->page_pointer[index_osc]=com->buffer_data[index_acq+j];
		}
	      // move the acq_index to the next sample
	      index_acq+=com->number_channels;
	    }
	  // get the data from the beginning of buffer
	  num_samples_to_transfer_start=num_samples_to_transfer-num_samples_to_transfer_end;
	  index_acq=0;
	  for(i=0;i<num_samples_to_transfer_start;i++)
	    {
	      for(j=0;j<osc->number_channels;j++)
		{
		  index_osc=(j*osc->samples_per_page)+osc->new_samples_in_buffer+i+num_samples_to_transfer_end;
		  if(index_osc>osc->buffer_size)
		    {
		      fprintf(stderr,"oscilloscope_interface_get_data wrapping start: index_osc>osc-buffer_size\n");
		      fprintf(stderr,"%d %d\n",index_osc,osc->buffer_size);
		       pthread_mutex_unlock( &mutex_comedi_interface_buffer );
		      return -1;
		    }
		  osc->page_pointer[index_osc]=com->buffer_data[index_acq+j];
		}
	      // move the acq_index
	      index_acq+=com->number_channels;
	    }
	}
      // update variable to know how much new data the rec buffer contains
      osc->new_samples_in_buffer=osc->new_samples_in_buffer+num_samples_to_transfer;
    }
  // can now free the comedi buffer for other thread
  pthread_mutex_unlock( &mutex_comedi_interface_buffer );
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_get_data, outside mutex\n");
#endif

  return 0;
}

int oscilloscope_interface_show_new_data(struct oscilloscope_interface* osc)
{
  // function to show newly acquired data in the oscilloscope
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_show_new_data\n");
#endif
  
  // check if the new page is full of data
  if (osc->new_samples_in_buffer==osc->samples_per_page)
    {
#ifdef DEBUG_OSC
      fprintf(stderr,"osc->current_group: %d, osc->current_page: %d osc->pages_in_memory: %d\n", osc->current_group, osc->current_page, osc->pages_in_memory);
#endif
      // show the new page of data
      oscilloscope_interface_show_data(osc,osc->current_page);
      
      osc->number_samples_displayed+=osc->samples_per_page;
      // reset the number of new samples in buffer
      osc->new_samples_in_buffer=0;
      // move to the next page
      if(osc->current_page<osc->number_pages_in_buffer-1)
	{ osc->current_page++;}
      else
	{ osc->current_page=0;}
      
      if(osc->pages_in_memory<osc->number_pages_in_buffer-1)
	{osc->pages_in_memory++;}
    }
  return 0;
}

int oscilloscope_interface_increase_time_resolution(struct oscilloscope_interface* osc, struct comedi_interface* com)
{
  if(osc->gui_seconds_per_page/TIME_SEC_IN_OSCILLOSCOPE_PAGE_CHANGE_FACTOR>=MIN_TIME_SEC_IN_OSCILLOSCOPE_PAGE)
    {
      osc->gui_seconds_per_page=osc->seconds_per_page/TIME_SEC_IN_OSCILLOSCOPE_PAGE_CHANGE_FACTOR;
#ifdef DEBUG_OSC
      fprintf(stderr,"oscilloscope_interface_increase_time_resolution: gui_seconds_per_page: %lf\n",osc->gui_seconds_per_page);
#endif
    }
  return 0;
}
int oscilloscope_interface_decrease_time_resolution(struct oscilloscope_interface* osc, struct comedi_interface* com)
{
  if(osc->gui_seconds_per_page*TIME_SEC_IN_OSCILLOSCOPE_PAGE_CHANGE_FACTOR<=MAX_TIME_SEC_IN_OSCILLOSCOPE_PAGE)
    {
      osc->gui_seconds_per_page=osc->seconds_per_page*TIME_SEC_IN_OSCILLOSCOPE_PAGE_CHANGE_FACTOR;
#ifdef DEBUG_OSC
      fprintf(stderr,"oscilloscope_interface_decrease_time_resolution: gui_seconds_per_page: %lf\n",osc->gui_seconds_per_page);
#endif
    }
  return 0;
}
int oscilloscope_interface_decrease_global_gain(struct oscilloscope_interface* osc)
{
  if(osc->gui_global_gain/osc->global_gain_factor>=osc->min_global_gain)
    {
      osc->gui_global_gain=osc->gui_global_gain/osc->global_gain_factor;
    }
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_decrease_global_ gain to %lf\n",osc->gui_global_gain);
#endif

  return 0;
}
int oscilloscope_interface_increase_global_gain(struct oscilloscope_interface* osc)
{
  if(osc->gui_global_gain*osc->global_gain_factor<=osc->max_global_gain)
    {
      osc->gui_global_gain=osc->gui_global_gain*osc->global_gain_factor;
    }
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_increase_global_ gain to %lf\n",osc->gui_global_gain);
#endif
  return 0;
}

int oscilloscope_interface_show_previous_page(struct oscilloscope_interface* osc)
{
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_show_previous_page, in_buffer: %d, displayed_page:%d, current_page: %d, pages_in_memory: %d\n", osc->number_pages_in_buffer,osc->displayed_page,osc->current_page,osc->pages_in_memory);
#endif
  int page;
  // by default, show the page currently on displayed
  page=osc->displayed_page;
  
  if(osc->displayed_page<osc->current_page)
    {
      if(osc->displayed_page>0&&osc->current_page-osc->displayed_page< osc->pages_in_memory )
	{
	  page=osc->displayed_page-1;
	}
      if(osc->displayed_page==0)
	{
	  if(osc->number_pages_in_buffer-1>osc->current_page && osc->pages_in_memory>osc->current_page)
	    {
	      page=osc->number_pages_in_buffer-1;
	    }
	}
    }                                                     
  if(osc->displayed_page-1>osc->current_page && osc->current_page+(osc->number_pages_in_buffer-osc->displayed_page)<osc->pages_in_memory)
    {
      page=osc->displayed_page-1;
    }
  fprintf(stderr,"show page %d\n", page);
  oscilloscope_interface_show_data(osc,page);

  return 0;
}
int oscilloscope_interface_show_next_page(struct oscilloscope_interface* osc)
{
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_show_next_page\n");
#endif
  // by default, show the page currently on displayed
  int page=osc->displayed_page;
  if(osc->displayed_page+1<osc->current_page)
    {
      page=osc->displayed_page+1;
    }
  if(osc->displayed_page>osc->current_page)
    {
      if (osc->displayed_page+2<osc->number_pages_in_buffer)
	page=osc->displayed_page+1;
      else
	page=0;
    }
  fprintf(stderr,"show page %d\n", page);
  oscilloscope_interface_show_data(osc,page);
   
  return 0;
}

int oscilloscope_interface_set_default_group_values(struct oscilloscope_interface* osc)
{
  // function to set default values for group display
  // which channels are shown where, using which color,  etc.
#ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_set_default_group_values\n");
#endif
  
  int i,j,chan,k,index_color;
  chan=0;
  index_color=0;
  for (i=0; i< osc->number_groups;i++)
    {
      if(i<osc->number_channels/DEFAULT_CHANNELS_PER_GROUP)
	{
	  osc->number_channels_per_group[i]=DEFAULT_CHANNELS_PER_GROUP;
	}
      else if(i==osc->number_channels/DEFAULT_CHANNELS_PER_GROUP)
	{
	  osc->number_channels_per_group[i]=osc->number_channels%DEFAULT_CHANNELS_PER_GROUP;	  
	}
      else
	{
	  osc->number_channels_per_group[i]=0;
	}

      for (j=0;j<osc->number_channels_per_group[i];j++)
	{
	  // set value for each channel of the group
	  osc->channel_list_in_group[i][j]=chan;
	  osc->gain[i][j]=OSCILLOSCOPE_DEFAULT_GAIN;
	  k=(index_color/4)%8;
	  switch (k)
	    {
	    case 0 :
	      {
		osc->channel_red[i][j]=0;
		osc->channel_green[i][j]=0;
		osc->channel_blue[i][j]=0;
		break;
	      }
	    case 1 :
	      {
		osc->channel_red[i][j]=0;
		osc->channel_green[i][j]=0;
		osc->channel_blue[i][j]=0.5;
		break;
	      }
	    case 2 :
	      {
		osc->channel_red[i][j]=0;
		osc->channel_green[i][j]=0.5;
		osc->channel_blue[i][j]=0;
		break;
	      }
	    case 3 :
	      {
		osc->channel_red[i][j]=0.5;
		osc->channel_green[i][j]=0;
		osc->channel_blue[i][j]=0;
		break;
	      }
	    case 4 :
	      {
		osc->channel_red[i][j]=1;
		osc->channel_green[i][j]=0;
		osc->channel_blue[i][j]=0;
		break;
	      }
	    case 5 :
	      {
		osc->channel_red[i][j]=0;
		osc->channel_green[i][j]=1;
		osc->channel_blue[i][j]=0;
		break;
	      }
	    case 6 :
	      {
		osc->channel_red[i][j]=0;
		osc->channel_green[i][j]=0;
		osc->channel_blue[i][j]=1;
		break;
	      }
	    case 7 :
	      {
		osc->channel_red[i][j]=0.5;
		osc->channel_green[i][j]=0.5;
		osc->channel_blue[i][j]=0.0;
		break;
	      }

	    default:
	      {
		osc->channel_red[i][j]=0;
		osc->channel_green[i][j]=0;
		osc->channel_blue[i][j]=0;
		break;
	      }
	    }
	  chan++;
	  index_color++;
	}
    }
  return 0;
}

static gboolean oscilloscope_interface_timer_update()
{
  if(osc_inter.is_displaying==0)
    { // this will be the last tic.
      return FALSE;
    }
  
  if(osc_inter.is_drawing==1)
    {
      return TRUE;
    }
  else
    {
      osc_inter.is_drawing=1;
      if(oscilloscope_interface_update_time_gain(&osc_inter,&comedi_inter)==-1)
	{
	  osc_inter.is_drawing=0;
	  return FALSE;
	}
      if(oscilloscope_interface_get_data(&osc_inter,&comedi_inter)==-1)
	{
	  
	  if(oscilloscope_interface_reset_current_variables(&osc_inter, &comedi_inter)==-1)
	    {
	      fprintf(stderr,"drawing in oscilloscope is probably too slow in oscilloscope_interface_timer_update()\n");
	      osc_inter.is_drawing=0;
	      return TRUE;
	    }
	}
      oscilloscope_interface_show_new_data(&osc_inter);
      osc_inter.is_drawing=0;
      return TRUE;
    }
}

int oscilloscope_interface_start_oscilloscope(struct oscilloscope_interface* osc)
{
  #ifdef DEBUG_OSC
  fprintf(stderr,"oscilloscope_interface_start_oscilloscope\n");
#endif
  if (osc->is_displaying==1)
    {
      fprintf(stderr,"oscilloscope already running in oscilloscope_interface_start_oscilloscope()\n");
      return -1;
    }
  //start a timer that will call oscilloscope_interface_timer_update();
  osc->is_displaying=1;
  g_timeout_add((MIN_TIME_SEC_IN_OSCILLOSCOPE_PAGE*1000)/4,(GSourceFunc) oscilloscope_interface_timer_update, (gpointer)widgets.drawing_area);
  return 0;
}


int oscilloscope_interface_stop_oscilloscope(struct oscilloscope_interface* osc)
{
#ifdef DEBUG_ACQ
  fprintf(stderr,"oscilloscope_interface_stop_oscilloscope\n");
#endif
  if(osc->is_displaying==0)
    {
      fprintf(stderr,"osc->is_displaying is already set to 0 in oscilloscope_interface_stop_oscilloscope\n");
    }

  osc->is_displaying=0;
  return 0;
}
int oscilloscope_interface_update_channel_list_in_group_from_liststore(struct oscilloscope_interface* osc,GtkListStore *oscilloscope_selected_channels_store, int group )
{
#ifdef DEBUG_ACQ
  fprintf(stderr,"oscilloscope_interface_update_channel_list_in_group_from_liststore\n");
#endif
  int channel_no; 
  GtkTreeIter iter;
  int rows_in_model=0;
  // check if the number of rows in model is valid
  if(gtk_tree_model_get_iter_first(GTK_TREE_MODEL(widgets.oscilloscope_selected_channels_store),&iter))
    {
      rows_in_model++;
    }
  while(gtk_tree_model_iter_next(GTK_TREE_MODEL(widgets.oscilloscope_selected_channels_store),&iter))
    {
      rows_in_model++;
    }
  if(rows_in_model>osc->max_channels_per_group)
    {
      fprintf(stderr,"oscilloscope_interface_update_channel_list_in_group_from_liststore, too many rows in model: %d, max: %d\n",rows_in_model,osc->max_channels_per_group);
      return 1;
    }
  if(group<0)
    {
      fprintf(stderr,"oscilloscope_interface_update_channel_list_in_group_from_liststore, group is smaller than 0: %d\n",group);
      return 1;
    }
  if(group>=osc->number_groups)
    {
      fprintf(stderr,"oscilloscope_interface_update_channel_list_in_group_from_liststore, group (%d) is larger than osc->number_group: %d\n",group,osc->number_groups);
      return 1;
    }

  osc->number_channels_per_group[group]=0;
  if(gtk_tree_model_get_iter_first(GTK_TREE_MODEL(widgets.oscilloscope_selected_channels_store),&iter))
    {
      gtk_tree_model_get(GTK_TREE_MODEL(widgets.oscilloscope_selected_channels_store), &iter,OSC_ALL_COL_NO,&channel_no,-1);
      //g_print ("%d is at index %d of group %d\n", channel_no,osc->number_channels_per_group[group],group);
      if(channel_no>=osc->number_channels)
	{
	  fprintf(stderr,"oscilloscope_interface_update_channel_list_in_group_from_liststore, channel_no from model (%d) is larger than osc->number_channels: %d\n",channel_no,osc->number_channels);
	  return 1;
	}
      if(channel_no<0)
	{
	  fprintf(stderr,"oscilloscope_interface_update_channel_list_in_group_from_liststore, channel_no from model (%d) is less than 0\n",channel_no);
	  return 1;
	}
      osc->channel_list_in_group[group][osc->number_channels_per_group[group]]=channel_no;
      osc->number_channels_per_group[group]++;
      while(gtk_tree_model_iter_next(widgets.oscilloscope_selected_channels_store,&iter))
	{
	  gtk_tree_model_get(GTK_TREE_MODEL(widgets.oscilloscope_selected_channels_store), &iter,OSC_ALL_COL_NO,&channel_no,-1);
	  //g_print ("while loop, %d is at index %d of group %d\n", channel_no,osc->number_channels_per_group[group],group);
	  if(channel_no>=osc->number_channels)
	    {
	      fprintf(stderr,"oscilloscope_interface_update_channel_list_in_group_from_liststore, channel_no from model (%d) is larger than osc->number_channels: %d\n",channel_no,osc->number_channels);
	      return 1;
	    }
	  if(channel_no<0)
	    {
	      fprintf(stderr,"oscilloscope_interface_update_channel_list_in_group_from_liststore, channel_no from model (%d) is less than 0\n",channel_no);
	      return 1;
	    }
	  osc->channel_list_in_group[group][osc->number_channels_per_group[group]]=channel_no;
	  osc->number_channels_per_group[group]++;
	}
    }
  return 0;
}
