/*
  tree.c, written by Rhett "Jonzy" Jones

  Jonzy's Universal Gopher Hierarchy Excavation And Display.
  Excavates through gopher menus and displays the hierarchy
  of the menus encountered

  Copyright (C) 1993, 1994 University of Utah Computer Center.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program (look for the file called COPYING);
  if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/* Description:	Contains routines dealing with sockets and communicating
 *		over the internet.
 */

#include "stdinc.h"
#include "utils.h"

void PostContactHostError (int error, char *hostStr, char *portStr);
char *GetHostName (char *hostDomainFromConf_h);
void IP2Hostname (int sockfd, char *host, char *ip);
int SetUpReadNwriter (int theSocket);
void CloseReadNwriter (void);
int ContactHost (char *theHost, int thePort);
int SendBuffer (char *buffer, int bufferSize);
int SendString (char *s);
char *GetString (FILE *ptr);
static void TooMuchTime4Read (int sig);
int ListenerEstablished (int port);
int ProcessRequest (int sockfd);

#define READTIMEOUT	(5 * 60)
#define MAXLINE		512
#define RECVSIZE	4096
#define BUFFERSIZE	2048
static jmp_buf env;

/* These are defined in "jugtailConf.c". */
extern int debug;
extern FILE *wtPtr, *rdPtr, *rptPtr;
extern char buffer[BUFFERSIZE];

extern char *hostname,		/* Declared in "searchCmnds.c". */
 *gtstrerr, *readerr;

extern char *errorhost;		/* Declared in "searchCmnds.c". */

extern void PostPositions ();	/* Defined in "searchCmnds.c". */
extern void LogMessage ();	/* Defined in "searchCmnds.c". */
extern void MemoryCaution ();	/* Defined in "searchCmnds.c". */


/*****************************************************************************
 * PostContactHostError simply prints the result of any error encountered
 * with the call to ContactHost().
 ****************************************************************************/
void
PostContactHostError (int error, char *hostStr, char *portStr)
     /* error: The error to report.
	hostStr: The host we tried to connect to.
	portStr: The port we used. */
{
  switch (error)
    {
    case -1:
      fprintf (rptPtr,
	       "error: SetUpReadNwriter could not open socket file for writing.\n");
      break;
    case -2:
      fprintf (rptPtr,
	       "error: SetUpReadNwriter could not open socket file for reading.\n");
      break;
    case -3:
      fprintf (rptPtr, "error: ContactHost could not get the socket.\n");
      break;
    case -4:
      fprintf (rptPtr, "error: ContactHost found unknown host [%s].\n",
	       hostStr);
      break;
    case -5:
      fprintf (rptPtr,
	       "error: ContactHost could not connect to [%s] via port [%s].\n",
	       hostStr, portStr);
      break;
    default:
      fprintf (rptPtr, "error: ContactHost returned unknown error.\n");
      break;
    }

}				/* PostContactHostError */

/*****************************************************************************
 * GetHostName returns the name of the machine we are running on.
 * If HOSTDOMAIN is nil we conclude that the system call to hostname
 * returns our Fully Qualified Domain Name, otherwise HOSTDOMAIN
 * was set to append to what hostname returns to get the
 * Fully Qualified Domain Name.  The value returned is acquired in
 * malloc space.
 ****************************************************************************/
char *
GetHostName (char *hostDomainFromConf_h)
     /* hostDomainFromConf_h: HOSTDOMAIN in jugtail.h */
{
  char theHostName[256],	/* Who we think we are. */
   *ourName;			/* The string we return. */
  struct hostent *hostEntry;	/* Information about the host. */

  if (hostname)			/* Free up the memory. */
    {
      free (hostname);
      hostname = (char *) 0;
    }

  theHostName[0] = '\0';
  if (gethostname (theHostName, 256))	/* Yipes. */
    return ((char *) 0);
  if ((hostEntry = gethostbyname (theHostName))
      && strlen (hostEntry->h_name) > strlen (theHostName))
    strncpy (theHostName, hostEntry->h_name, 256);
  else
    strcat (theHostName, hostDomainFromConf_h);
  if ((ourName = malloc (strlen (theHostName) + 1)))
    strcpy (ourName, theHostName);
  return (ourName);

}				/* GetHostName */

/*****************************************************************************
 * IP2Hostname places the name of the host, as acquired from the IP
 * number, into 'host'.  This way we can log the name of the host instead
 * of the IP number.
 ****************************************************************************/
void
IP2Hostname (int sockfd, char *host, char *ip)
     /* sockfd: Socket file descriptor.
	host: The name of the host we are talking to.
	ip: The hosts IP number. */
{
  struct sockaddr_in sin;	/* Info about the socket. */
  socklen_t len;		/* The size of 'sin'. */
  struct hostent *hostEntry;	/* Information about the host. */

  len = sizeof (sin);
  getpeername (sockfd, &sin, &len);
  strcpy (ip, inet_ntoa (sin.sin_addr));
  strcpy (host, inet_ntoa (sin.sin_addr));

  hostEntry = gethostbyaddr (&sin.sin_addr, sizeof (sin.sin_addr.s_addr),
			     AF_INET);

  if (hostEntry)
    strcpy (host, hostEntry->h_name);

}				/* IP2Hostname */

/*****************************************************************************
 * SetUpReadNwriter returns 0 if we can set up read and write pointers
 * to the socket 'theSocket'.  Otherwise the value returned is type of
 * error encountered, where -1 = couldn't setup for writing, and
 * -2 = couldn't setup for reading
 ****************************************************************************/
int
SetUpReadNwriter (int theSocket)
{
  /* See if we can open a file for writing. */
  if (!(wtPtr = fdopen (theSocket, "w")))
    return (-1);

  /* See if we can open a file for reading. */
  if (!(rdPtr = fdopen (theSocket, "r")))
    return (-2);

  return (0);

}				/* SetUpReadNwriter */

/*****************************************************************************
 * CloseReadNwriter simply closes 'rdPtr' and 'wtPtr'.
 ****************************************************************************/
void
CloseReadNwriter ()
{
  fclose (wtPtr);
  fclose (rdPtr);

}				/* CloseReadNwriter */

/*****************************************************************************
 * ContactHost returns 0 if we were able to get a connection to 'theHost'
 * out port 'thePort', and open the files 'wrPtr' and 'rdPtr' for reading and
 * and writing information to and from 'theHost' out 'thePort'.  Otherwise
 * this routine returns a value representing the type of error encountered.
 * Returned values are:
 *   0 -> no error
 *  -1 -> SetUpReadNwriter could not open socket file for writing
 *  -2 -> SetUpReadNwriter could not open socket file for reading
 *  -3 -> ContactHost could not get the socket
 *  -4 -> ContactHost found unknown host 'theHost'
 *  -5 -> ContactHost could not connect 'theHost' via port 'thePort'
 ****************************************************************************/
int
ContactHost (char *theHost, int thePort)
     /* theHost: The host to connect with.
	thePort: The port to use. */
{
  int theSocket,		/* The socket. */
    error;			/* The error if encountered. */
  struct sockaddr_in scktRec;	/* The socket address info. */
  struct hostent *theHostEntry;	/* The host entry. */

  if (debug)
    fprintf (rptPtr, "ContactHost attempting a connection to: %s %d\n",
	     theHost, thePort);

  /* See if we can get a socket. */
  if ((theSocket = socket (PF_INET, SOCK_STREAM, 0)) < 0)
    return (-3);

  /* Initialize our socket address information. */
  scktRec.sin_family = AF_INET;
  scktRec.sin_port = htons (thePort);
  if ((theHostEntry = gethostbyname (theHost)))
    memcpy (&scktRec.sin_addr.s_addr, theHostEntry->h_addr, 4);
  else
    return (-4);

  /* See if we can get a connection. */
  if (connect (theSocket, (struct sockaddr *) &scktRec, sizeof (scktRec)) < 0)
    return (-5);

  if ((error = SetUpReadNwriter (theSocket)))
    return (error);

  if (debug)
    fprintf (rptPtr, "ContactHost has established connection\n");

  return (0);

}				/* ContactHost */

/*****************************************************************************
 * SendBuffer returns the number of bytes written to 'wtPtr', but first writes
 * the contents of 'buffer' to 'wtPtr' which is the machine we are talking to.
 ****************************************************************************/
int
SendBuffer (char *buffer, int bufferSize)
     /* buffer: The buffer we are sending to the machine.
	bufferSize: The number of bytes in 'buffer'. */
{
  int result;			/* The result of writting to the socket. */

  result = fwrite (buffer, 1, bufferSize, wtPtr);
  fflush (wtPtr);
  return (result);

}				/* SendBuffer */

/*****************************************************************************
 * SendString returns true no matter what, but first writes the string 's' to
 * 'wtPtr' which is the machine we are talking to.
 ****************************************************************************/
int
SendString (char *s)
     /* s: The string we are sending to the machine. */
{
  fputs (s, wtPtr);
  fflush (wtPtr);
  return (1);

}				/* SendString */

/*****************************************************************************
 * GetString returns a line of text from either the socket or file we are
 * reading from, which is pointed to by the 'ptr'.  The string will contain at
 * most 'BUFFERSIZE' characters.  For more information on the contents of the
 * string returned by this routine consult the man page on 'fgets'.
 ****************************************************************************/
char *
GetString (FILE *ptr)
     /* ptr: The file or socket we are reading from. */
{
  return (fgets (buffer, BUFFERSIZE, ptr));

}				/* GetString */

/*****************************************************************************
 * TooMuchTime4Read gets called only when the time period has elapsed when
 * waiting to read from our input..
 ****************************************************************************/
static void
TooMuchTime4Read (int sig)
{
  if (debug)
    fprintf (rptPtr, "In TooMuchTime4Read\n");

  longjmp (env, 1);

}				/* TooMuchTime4Read */

/*****************************************************************************
 * ListenerEstablished returns -1 if we could NOT: create a socket, set up the socket
 * such that the address will be reused and will stick around for the client,
 * we can bind the socket to the port 'port', and we can set up to listen for
 * up to 5 connections out the port, a socket listening out port 'port'.
 * Otherwise we were able to do all of the above so we return the socket.
 ****************************************************************************/
int
ListenerEstablished (int port)
{
  struct sockaddr_in sin;	/* The socket information. */
  struct linger so_linger;	/* The SO_LINGER info. */
  int so_reuseaddr,		/* The SO_REUSEADDR info. */
    sckt,			/* The socket to return. */
    error;			/* Did we get an error? */

  if (debug)
    fprintf (rptPtr, "In ListenerEstablished\n");

  /* Initialize the variables. */
  so_reuseaddr = 1;
  error = so_linger.l_onoff = so_linger.l_linger = 0;
  memset (&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = htonl (INADDR_ANY);
  sin.sin_port = htons (port);

  /* Create a socket binding the same local address and make sure the other will still be around. */
  if ((sckt = socket (AF_INET, SOCK_STREAM, 0)) >= 0)
    if (setsockopt
	(sckt, SOL_SOCKET, SO_REUSEADDR, &so_reuseaddr,
	 sizeof (so_reuseaddr)) >= 0)
      if (setsockopt
	  (sckt, SOL_SOCKET, SO_LINGER, &so_linger, sizeof (so_linger)) >= 0)
	if (bind (sckt, (struct sockaddr *) &sin, sizeof (sin)) >= 0)
	  if (listen (sckt, 5) >= 0)
	    return (sckt);
	  else
	    error = fprintf (rptPtr, "ListenerEstablished: can't listen 5\n");
	else
	  error = fprintf (rptPtr, "ListenerEstablished: can't bind\n");
      else
	error =
	  fprintf (rptPtr,
		   "ListenerEstablished: can't setsockopt SO_LINGER\n");
    else
      error =
	fprintf (rptPtr,
		 "ListenerEstablished: can't setsockopt REUSEADDR!\n");
  else
    error = fprintf (rptPtr, "ListenerEstablished: can't socket\n");

  return (-error);

}				/* ListenerEstablished */

/*****************************************************************************
 * ProcessRequest gets and processes a request from a port.
 ****************************************************************************/
int
ProcessRequest (int sockfd)
     /* sockfd: The socket file descriptor. */
{
  char *inputline,		/* The request to process. */
    bufr[1056];			/* Buffer for error messages. */
  int length;			/* Length of the command line */

  if (SetUpReadNwriter (sockfd))
    {
      LogMessage (-1,
		  "error: ProcessRequest could not setup read and writters");
      if (debug)
	fprintf (rptPtr,
		 "error: ProcessRequest could not setup read and writters\n");
      return (0);
    }

  /* Set things up so we don't wait for ever waiting to get a request. */
  signal (SIGALRM, TooMuchTime4Read);
  alarm (READTIMEOUT);

  if (setjmp (env))
    {
      MemoryCaution (strlen (gtstrerr) + strlen (errorhost), 1056,
		     "error: ProcessRequest attempted to overwrite memory");
      sprintf (bufr, "%s%s", gtstrerr, errorhost);
      SendString (bufr);
      exit (-1);
    }

  inputline = GetString (rdPtr);
  length = strlen (inputline);

  /* We got our request to deactivate the alarm. */
  alarm (0);
  signal (SIGALRM, SIG_IGN);

  if (length <= 0)
    {
      MemoryCaution (strlen (readerr) + strlen (errorhost), 1056,
		     "error: ProcessRequest attempted to overwrite memory");
      sprintf (bufr, "%s%s", readerr, errorhost);
      SendString (bufr);
      close (sockfd);
      return (0);
    }

  RemoveCRLF (inputline);

  if (debug)
    fprintf (rptPtr, "ProcessRequest processing [%s]\n", inputline);

  PostPositions (sockfd, inputline);
  SendString (".\r\n");
  CloseReadNwriter ();
  return (1);

}				/* ProcessRequest */
