#!/bin/csh -f

# jugtail.sample.csh written by Rhett "Jonzy" Jones 
#
# Jonzy's Universal Gopher Hierarchy Excavation And Display.
# Excavates through gopher menus and displays the hierarchy
# of the menus encountered
#
# Copyright (C) 1993, 1994 University of Utah Computer Center.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program (look for the file called COPYING);
# if not, write to the Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

# Description:	Builds or rebuilds the jugtail database and either
#		starts jugtail as a server or tells jugtail to reread
#		in the database depending on the existance of the
#		file /usr/local/etc/jugtail.pid, and sends mail stating
#		the database has been rebuilt with listings of any
#		problems and some statistical information.
#		This script is intended to be run via cron.
#
# Usage:	jugtail.sample.csh
#
# Notes:	You will need to make the appropriate changes
#		for your particular environment and needs.
#		Inparticular you need to set the following values:
#			DEBUG		1 = debugging
#			PS		Your local ps command
#			JUGTAIL		The jugtail program
#			JUGTAILPID	The jugtail.pid file
#			DATANAME	Name of the database
#			REPORTNAME	Name of the report file
#			STATS		Name of the rebuild history file
#			LOG		The search engine log file.
#			DATADIR		The data directory
#			TEMPDIR		Temporary directory for building
#			MAILTO		Who gets the mail
#			MAILSUBJECT	The email subject
#
#		And then edit the COLLECT, BUILD, and SEARCH blocks
#		#########################################################
#		#  COLLECT # COLLECT # COLLECT # COLLECT # COLLECT # COLL
#		#
#		$JUGTAIL ... Your jugtail -b command
#		#########################################################
#
#		#########################################################
#		# BUILD # BUILD # BUILD # BUILD # BUILD # BUILD # BUILD #
#		#
#		$JUGTAIL ... Your jugtail -B command
#		#########################################################
#
#		#########################################################
#		# SEARCH # SEARCH # SEARCH # SEARCH # SEARCH # SEARCH # S
#		#
#		#	$JUGTAIL ... Your jugtail -S command
#		#########################################################
#
#		And then comment out or remove the lines begining with
#			echo
#			exit
#		There are (2) two of these
#
#		You might want to rename this file to something like:
#		jugtail.csh
#
# Bugs:		If jugtail is not running at the time this script is
#		started and there exists a process with the same id
#		as that recorded in the /usr/local/etc/jugtail.pid file
#		well, ..., huh, ..., ah-oh.  Oops.
#
#########################################################################

# Set to 1 if you want debugging.
set DEBUG	= 0

# This is your local `ps` command
set PS		= "ps -aux"

# This is the location of your jugtail program.
set JUGTAIL	= "/usr/local/etc/jugtail"

# The jugtail.pid file with pathway
set JUGTAILPID	= "/usr/local/etc/jugtail.pid"

# This is the name of your jugtail database.
set DATANAME	= "data"

# This is the name of your jugtail rebuild report.
set REPORTNAME	= "rebuild.log"

# This is the name of your jugtail record of rebuilds.
set STATS	= "/jugtail.rebuild.stats"

# This is the name of the jugtail search engine log.
set LOG		= "/cwis/info/gopher/gopherLog/jugtail.log"

# This is where your jugtail database resides.
set DATADIR	= "/jugdata"

# This is the location for rebuilding the jugtail databases.
set TEMPDIR	= "/newdat"

# The rfc 822 email address of who gets the report.
set MAILTO	= "yourUserid@your.site"

# The subject for the email message.
set MAILSUBJECT	= "jugtail rebuild info"


#########################################################################
# Comment the next 2 lines out after you have made the required changes
# for this script to funtion in your environment.
echo You need to read and edit this script for it to work.
exit
#########################################################################

#########################################################################
#   Edit the COLLECT BUILD and SEARCH blocks below to your particular   #
#               environment specifications and needs.                   #
#########################################################################

# Lets make sure jugtail is even compiled and in place.
if (!( -x "$JUGTAIL")) then
  echo $JUGTAIL does not exist or permissions are wrong.
  exit
endif

# This step rebuilds the database.
if ($DEBUG == 1) then
  echo rebuilding the database
endif
#########################################################################
#  COLLECT # COLLECT # COLLECT # COLLECT # COLLECT # COLLECT # COLLECT  #
#
# This is the jugtail call to collect your data
# My call looks like
# $JUGTAIL -Vb "$TEMPDIR/$DATANAME" -l "$TEMPDIR/$REPORTNAME" -X "ftp:*" .utah.edu gopher.utah.edu
# Make sure you comment out the next 2 lines once you set your call up.
echo You need to adjust the jugtail COLLECT call
exit
$JUGTAIL -Vb "$TEMPDIR/$DATANAME" -l "$TEMPDIR/$REPORTNAME" -X "ftp:*" .your.domain gopher.your.domain
#########################################################################
if ($DEBUG == 1 && $status != 0) then
  echo error: jugtail -b failed
  exit
endif


# This step rebuilds the tables.
if ($DEBUG == 1) then
  echo rebuilding the tables
endif
#########################################################################
# BUILD # BUILD # BUILD # BUILD # BUILD # BUILD # BUILD # BUILD # BUILD #
#
# This is the jugtail call to Build your data tables
$JUGTAIL -Bml "$TEMPDIR/$REPORTNAME" "$TEMPDIR/$DATANAME"
#########################################################################
if ($DEBUG == 1 && $status != 0) then
  echo error: jugtail -Bml failed
  exit
endif


# This step moves the databases into place.
if ($DEBUG == 1) then
  echo doing mv "$TEMPDIR/$DATANAME*" "$DATADIR/"
endif
/bin/mv $TEMPDIR/$DATANAME* "$DATADIR/"
if ($DEBUG == 1 && $status != 0) then
  echo error: /bin/mv failed
  exit
endif


# Append the report to the statistical file for history of jugtail rebuilds.
if ($DEBUG == 1) then
  echo cating to $STATS
endif
/bin/cat "$TEMPDIR/$REPORTNAME" >> "$STATS"
if ($DEBUG == 1 && $status != 0) then
  echo error: /bin/cat failed
  exit
endif

# Either start jugtail from scratch or kill HUP the sucker.
# See if the jugtail.pid file exists.
if ( -f $JUGTAILPID ) then	# Yep it does.

  if ($DEBUG == 1) then
    echo the jugtail.pid file exists
  endif

  # This is a kludge to see if jugtail is currently running
  # with the process as specified on the first line of the
  # jugtail.pid file.

  $PS > /tmp/jtmp1
  if ($DEBUG == 1) then
    echo greping for the process
    ls /tmp/jtmp1
  endif

  /bin/grep `/usr/ucb/head -1 $JUGTAILPID` /tmp/jtmp1 > /tmp/jtmp2

  # Now if /tmp/jtmp2 is 0 bytes jugtail is not running
  # or if it is jugtail did not create the file
  # /usr/local/etc/jugtail.pid.  If jugtail is not
  # running and there exists another process with the
  # pid as recoreded in the jugtail.pid, well ah-oh.

  # Check file size of /tmp/jtmp2.
  if ($DEBUG == 1) then
    echo checking if the jugtail process is running
  endif
  if ( -z /tmp/jtmp2 ) then	# No process running with the jugtail-pid.

    if ($DEBUG == 1) then
      echo jugtail is NOT running so starting jugtail per jugtail.pid
    endif

    # Make note that jugtail is dead for some reason.
    echo jugtail is NOT running so starting jugtail per jugtail.pid >> "$TEMPDIR/$REPORTNAME"

    # Restart jugtail like the last time it was started.
    eval `/usr/ucb/tail -1 $JUGTAILPID`

  else	# jugtail running or another process with the recorded jugtail-pid is.

    # Check if we can SIGHUP the sucker
    $PS | grep $JUGTAIL | grep `whoami` >& /dev/null
    if ($status == 1) then
      if ($DEBUG == 1) then
        echo error: cannot SIGHUP jugtail - does not look like your the owner
      endif
      echo not owner cannot SIGHUP jugtail  >> "$TEMPDIR/$REPORTNAME"
    else
      if ($DEBUG == 1) then
        echo jugtail is running so sending it SIGHUP
      endif

      kill -HUP `/usr/ucb/head -1 $JUGTAILPID`
    endif

  endif	# Checking file size of /tmp/jtmp2.

  # Get rid of the temp files for our kludge.
  /bin/rm -f /tmp/jtmp1 /tmp/jtmp2
else	# No jugtail.pid file

  # Make note that jugtail was started from scratch.
  if ($DEBUG == 1) then
    echo starting jugtail from scratch
  endif
  echo starting jugtail from scratch >> "$TEMPDIR/$REPORTNAME"
#########################################################################
# SEARCH # SEARCH # SEARCH # SEARCH # SEARCH # SEARCH # SEARCH # SEARCH #
#
# This is the call to start jugead as a search engine from scratch
$JUGTAIL -Sl "$LOG" "$DATADIR/$DATANAME"
#########################################################################

endif	# Existance of the jugtail.pid file.

# Send off a note stating what happened when rebuilding jugtail.

# Mail the results to the person concerned.
if ($DEBUG == 1) then
  echo mailing results to $MAILTO
endif
/usr/ucb/mail -s "$MAILSUBJECT" "$MAILTO" < "$TEMPDIR/$REPORTNAME"
if ($DEBUG == 1 && $status != 0) then
  echo error: /usr/ucb/mail failed
  exit
endif

# Now deleted the old report so it doesn't grow and get mailed.
if ($DEBUG == 1) then
  echo cleaning up
endif
/bin/rm -f "$TEMPDIR/$REPORTNAME"
if ($DEBUG == 1 && $status != 0) then
  echo error: /bin/rm failed
  exit
endif
