/*
  jugtail.c, written by Rhett "Jonzy" Jones 

  Jonzy's Universal Gopher Hierarchy Excavation And Display.
  Excavates through gopher menus and displays the hierarchy
  of the menus encountered

  Copyright (C) 1993, 1994 University of Utah Computer Center.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program (look for the file called COPYING);
  if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "stdinc.h"

#include "dirTree.h"
#include "tree.h"
#include "utils.h"
#include "sockets.h"
#include "jugtail.conf.h"

#include "jugtail.h"


void HandleIndention (int indent);
short Host2Search (char *s);
List *CreateNode (char *sStr, char *hStr, char *pStr);
List *InsertNode (List *tail, List *node);
List *RemoveNode (List *tail);
short BeenHereB4 (char *sStr, char *hStr, char *pStr, List *head);
short CantGet2Host (List *nogoList, char *sStr, char *Str, char *pStr);
short DoWeTraverse (char type, char *sStr, char *hStr, char *pStr, int indent);
void PrintItem (char *dStr, int indent);
char *EmptyStringCheck (char *sStr);
char *HostField2Lower (char *line);
static void TooMuchTime4Read (int sig);
int LooksGood (char **rsltLine, FILE *rdPtr);
int Unlink (char *path);
void ProcessMenu (char *selStr, char *hostStr, char *portStr, int indent);
void PostTime2Process (void);
int ValidVariableCheck ();
void SpankDuplicateEntries (char *f1, char *f2);
void PrintJugtailConf ();
int InitializeTheWorld ();
char *GetCommandLine (int argc, char *argv[]);
static void ControlCAbort (int sig);

/* #define DOPATHS */

#define READTIMEOUT	(5 * 60)
static jmp_buf env;

extern void DoSearch ();	/* Defined in "searchCmnds.c". */
extern void MakeHashTables ();	/* Defined in "searchCmnds.c". */
extern char *GetString ();	/* Defined in "tree.c". */

/* These routines are defined in "path.c". */
extern int PopPath ();
extern int PrintPath ();
extern int PushPath ();

/* These routines are defined in "jugtailConf.c". */
extern char *MakeStr ();
extern int MakeInt ();
extern int ReadConfFile ();
extern int ReadLine ();

/* These routines are defined in "search.c". */
extern int CreateWordsTree ();

FILE *wtPtr,			/* File pointer for writing to 'theHost'. */
 *rdPtr,			/* File pointer for reading from 'theHost'. */
 *rptPtr;			/* Pointer to the logFile file. */
char buffer[BUFFERSIZE],	/* The output from the machine. */
 *searchHosts[MAXHOSTS],	/* Hosts to search. */
 *initialHost,			/* The initial host we connected to. */
 *selStr,			/* The selector string. */
 *hostStr,			/* The host string. */
 *portStr,			/* The port string. */
 *fileName,			/* Name of the file to open or create. */
 *userName,			/* Name of the user to use with -S. */
 *veronica;			/* Name of the veronica dataFile. */
int numSearchHosts,		/* The number of hosts to search. */
  debug,			/* Are we debugging? */
  info4dirsOnly,		/* Do we print host info for directories only? */
  info4allItems,		/* Do we print host info for all the items? */
  listHosts,			/* Do we print only the hosts accessed? */
  listHostsNPorts,		/* Do we print only the hosts and ports accessed? */
  buildDataFile,		/* Are we building the dataFile? */
  menuFlag,			/* Do we display the menus? */
  printLineNumbers,		/* Do we print line numbers? */
  printDTree,			/* Do we print the dirTree with the number of directories served? */
  printDTreeDirs,		/* Do we print the dirTree along with the directories served? */
  time2process;			/* Do we calculate the time for a certain run? */
short buildIndex,		/* Do we build the index table? */
  doSearch;			/* Are we doing the search stuff? */
List *nogos = (List *) NULL,	/* Information about who to not traverse. */
  *head = (List *) NULL,		/* The head of the current menu path list. */
  *tail = (List *) NULL,		/* The tail of the current menu path list. */
  *nogoHead = (List *) NULL,	/* Head of the can't connect to host and port list. */
  *nogoTail = (List *) NULL;	/* Tail of the can't connect to host and port list. */
long seenDirB4;			/* Have we seen this before? */
time_t startTime;		/* The time a run was started, for use with 'time2process'. */

/* The root of the veronica tree, which is a list of gopher servers we need	*
 * to try and grab the veronica.ctl file from to acquire the "Disallow" paths.	*/
TreeType *vRoot = (TreeType *) NULL;

extern long lineNumber;		/* Defined in "getargs.c". */

extern char *vctlHost,		/* Defined in "path.c". */
 *vctlPort;
extern void PrintTheList ();

extern void CreateVeronicaFile ();	/* Defined in "path.c". */

/*****************************************************************************
 * HandleIndention prints 'indent' number of indentions to stdout.  This
 * routine is used to keep directory contents lined up horizontaly.
 ****************************************************************************/
void
HandleIndention (int indent)
  /* indent: The level of indention. */
{
  int numIndents;	/* The number of indentions to add to path. */

  /* Indent the data 'indent' number of indentions. */
  for (numIndents = 0; numIndents < indent; numIndents++)
    fprintf (stdout, "%s", INDENTSTR);

}				/* HandleIndention */

/*****************************************************************************
 * Host2Search returns true if the host 's' is a member of 'searchHosts',
 * otherwise it returns false.
 ****************************************************************************/
short
Host2Search (char *s)
  /* s: Is this in 'searchHosts'? */
{
  int i;		/* A loop counter. */

  for (i = 0; i < numSearchHosts; i++)
    if (StrRcmp (searchHosts[i], s))
      return (1);

  return (0);

}				/* Host2Search */

/*****************************************************************************
 * CreateNode returns a newly created and initialized node for the list.
 ****************************************************************************/
List *
CreateNode (char *sStr, char* hStr, char *pStr)
     /* sStr: The selector string.
        hStr: The host string.
	pStr: The port string.     */
{
  List *node;			/* The node we are returning. */

  if ((node = malloc (sizeof (List))))
    {
      if ((node->info.sStr = malloc (strlen (sStr) + 1)))
	{
	  strcpy (node->info.sStr, sStr);
	  if ((node->info.hStr = malloc (strlen (hStr) + 1)))
	    {
	      strcpy (node->info.hStr, hStr);
	      
	      if ((node->info.pStr = malloc (strlen (pStr) + 1)))
		{
		  strcpy (node->info.pStr, pStr);
		  node->next = node->last = (List *) NULL;
		  return (node);
		}
	      else
		{
		  free (node->info.hStr);
		  node->info.hStr = (char *) NULL;
		  free (node->info.sStr);
		  node->info.sStr = (char *) NULL;
		  free (node);
		  node = (List *) NULL;
		}
	    }
	  else
	    {
	      free (node->info.sStr);
	      node->info.sStr = (char *) NULL;
	      free (node);
	      node = (List *) NULL;
	    }
	}
    }
  else
    {
      free (node);
      node = (List *) NULL;
    }
  
  fprintf (rptPtr, "error: CreateNode could not get memory for the node.\n");
  return (node);

}				/* CreateNode */

/*****************************************************************************
 * InsertNode inserts the node 'node' on the end of the list 'list'.
 ****************************************************************************/
List *
InsertNode (List *tail, List *node)
     /* tail: The tail of the list we are inserting into.
        node: The node we are inserting. */
{
  if (!node)
    return (tail);

  tail->next = node;
  node->last = tail;
  return (node);

}				/* InsertNode */

/*****************************************************************************
 * RemoveNode removes and frees up the memory occupied by the last item
 * in the list, which is pointed to by 'tail'.
 ****************************************************************************/
List *
RemoveNode (List *tail)
     /* tail: The tail of the list we are inserting into. */
{
  List *node;			/* The node to remove. */

  node = tail;
  tail = tail->last;
  if (tail)
    tail->next = (List *) NULL;
  free (node->info.sStr);
  node->info.sStr = (char *) NULL;
  free (node->info.hStr);
  node->info.hStr = (char *) NULL;
  free (node->info.pStr);
  node->info.pStr = (char *) NULL;
  free (node);
  node = (List *) NULL;
  return (tail);

}				/* RemoveNode */

/*****************************************************************************
 * BeenHereB4 returns true if 'sStr', 'hStr', and 'pStr' are in the list
 * pointed to by 'list'.  This is done so we will not travese a menu
 * we have already looking at.  Otherwise it returns false because we
 * are not looking at nor have been in this part of the tree.
 ****************************************************************************/
short
BeenHereB4 (char *sStr, char *hStr, char *pStr, List *list)
{
  while (list)			/* See if we have been here before. */
    {
      if (!strcmp (list->info.sStr, sStr))
	if (!strcmp (list->info.hStr, hStr))
	  if (!strcmp (list->info.pStr, pStr))
	    return (1);
      list = list->next;
    }
  return (0);

}				/* BeenHereB4 */

/*****************************************************************************
 * CantGet2Host returns true if we could not connect to 'hStr' out port 'pstr'
 * previously.  Otherwise it returns false.  This routine calls BeenHereB4()
 * and was written for readability only.
 ****************************************************************************/
short
CantGet2Host (List *nogoList, char *sStr, char *hStr, char *pStr)
     /* nogoList: List of hosts we can't connect to. 
	sStr: The selector string.
	hStr: The host to connect to.
	pStr: The port to use.                      */
{
  return (BeenHereB4 (sStr, hStr, pStr, nogoList));

}				/* CantGet2Host */

/*****************************************************************************
 * DoWeTraverse returns true if we can look at this directory in the menu.
 * Otherwise it returns false.  In otherwords return true if we have not
 * looked at this directory yet, and the current entry is a directory,
 * and the directory is from one of the hosts contained in 'hosts2search'.
 ****************************************************************************/
short
DoWeTraverse (char type, char *sStr, char *hStr, char *pStr, int indent)
     /* type: What type of entry is this?
        sStr: The selector string.
	hStr: The host string.
	pStr: The port string.
	indent: The level to indent if need be. */
{
  List *t;			/* A temporary list of nogos. */
  char *asterik;		/* Where is the asterik? */
  size_t numChars;		/* Number of characters to the asterik. */

  for (t = nogos; t; t = t->next)
    if (t->info.sStr[0])
      {
	if ((asterik = strchr (t->info.sStr, '*')))
	  numChars = (size_t) (asterik - t->info.sStr);
	else
	  numChars = strlen (t->info.sStr);
	if (!strncmp (t->info.sStr, sStr, numChars))
	  return (0);
      }
    else if (t->info.hStr)
      {
	if (!strcmp (pStr, t->info.pStr) &&
	    (!strcmp (hStr, t->info.hStr) || !strcmp ("*", t->info.hStr)))
	  return (0);
      }

  WaterTree (sStr, hStr, pStr);

  if (type == A_DIRECTORY && Host2Search (hStr))
    {
      if (!(seenDirB4 = InDirTree (dirRoot, PSTR)))
	{
	  dirRoot = BuildDirTree (dirRoot);
	  return (1);
	}
      else if (menuFlag)
	{
	  HandleIndention (indent + 2 * printLineNumbers);
	  fprintf (stdout, "<< see line %ld >>\n", seenDirB4),
	    ++lineNumber;
	}
    }
  return (0);

}				/* DoWeTraverse */

/*****************************************************************************
 * PrintItem prints the item with the proper number of indentions, as well
 * as prints what type of item it is just like the unix gopher from Minnesota.
 ****************************************************************************/
void
PrintItem (char *dStr, int indent)
     /* dStr: The display string.
	indent: The level of indention. */
{
  if (printLineNumbers)		/* User wants the line numbers printed. */
    fprintf (stdout, "%7ld ", lineNumber + 1);

  HandleIndention (indent);

  /* Print the menu title like a gopher menu. */
  fprintf (stdout, "%s", dStr + 1);
  switch (dStr[0])
    {
    case A_DIRECTORY:
      fprintf (stdout, "/");
      break;
    case A_CSO:
      fprintf (stdout, " <CSO>");
      break;
    case A_TN3270:
      fprintf (stdout, " <3270>");
      break;
    case A_TELNET:
      fprintf (stdout, " <TEL>");
      break;
    case A_INDEX:
      fprintf (stdout, " <?>");
      break;
    case A_SOUND:
      fprintf (stdout, " <)");
      break;
    case A_FILE:
      fprintf (stdout, ".");
      break;
    case A_PCBIN:
      fprintf (stdout, " <PC Bin>");
      break;
    case A_UNIXBIN:
      fprintf (stdout, " <Bin>");
      break;
    case A_IMAGE:
    case A_GIF:
      fprintf (stdout, " <Picture>");
      break;
    case A_MACHEX:
      fprintf (stdout, " <HQX>");
      break;
    default:
      fprintf (stdout, " <~%c~>", dStr[0]);
      break;
    }

}				/* PrintItem */

/*****************************************************************************
 * EmptyStringCheck checks if the string 'sStr' is either "/", "1/", or "1",
 * and if it is we return the emptystring "".  Otherwise 'sStr' gets returned
 * unchanged.  The reason for this routine is to prevent a recursive menu
 * loop from happening.  The gopher protocol says you must send the empty
 * string to acquire the menu.  Well it looks like a unix server handles
 * "", "/", "1/", and "1" all the same.  Hmm.
 ****************************************************************************/
char *
EmptyStringCheck (char *sStr)
     /* sStr: The string we are testing. */
{

  if (!strcmp (sStr, "/") || !strcmp (sStr, "1/") || !strcmp (sStr, "1"))
    return (EMPTYSTRING);
  return (sStr);

}				/* EmptyStringCheck */

/*****************************************************************************
 * HostField2Lower returns the string passed in with the host field converted
 * to lowercase.
 ****************************************************************************/
char *
HostField2Lower (char *line)
     /* line: The line we are looking at. */
{
  char *s;			/* A position in 'line'. */


  for (s = strchr (strchr (line, '\t') + 1, '\t') + 1; *s && *s != '\t'; s++)
    *s = tolower (*s);

  return (line);

}				/* HostField2Lower */

/*****************************************************************************
 * TooMuchTime4Read gets called only when the time period has elapsed when
 * waiting to read from our input..
 ****************************************************************************/
static void
TooMuchTime4Read (int sig)
{
  if (debug)
    fprintf (rptPtr, "In TooMuchTime4Read\n");

  longjmp (env, 1);

}				/* TooMuchTime4Read */

/*****************************************************************************
 * LooksGood returns true if we received a valid line from 'rdPtr', otherwise
 * it returns false.  A valid line has the form:
 *	dStr\tsStr\thStr\tpStr[\tmore]\r\n
 * a invalid line, or we are done, has the form:
 *	.\r\n
 * or any line with no tabs, etc.
 * This routine was written because the PC Gopher server does not follow
 * gopher protocol, and sends the string:
 *	Error: Could not find requested file\r\n.\r\n
 * when it should look something like:
 *	Error: could not find requested file\t\terror.host\t-1\r\n.\r\n
 ****************************************************************************/
int
LooksGood (char **rsltLine, FILE *rdPtr)
     /* rsltLine: The line we are getting from 'rdPtr'. 
	rdPtr: The file or socket we are reading from.  */
{
  char *s;			/* A position in 'rsltLine'. */
  int numTabs;			/* How many tabs are there? */


  /* Set things up so we don't wait for ever waiting to read. */
  signal (SIGALRM, TooMuchTime4Read);
  alarm (READTIMEOUT);
  if (setjmp (env))
    {
      if (debug)
	fprintf (rptPtr, "Too much time waiting to read\n");
      return (0);
    }

  /* Get the line of data from the server and make sure we got something. */
  if (!(*rsltLine = GetString (rdPtr)))
    return (0);

  /* We got our request so deactivate the alarm. */
  alarm (0);
  signal (SIGALRM, SIG_IGN);

  /* Check if we are done. */
  if (strstr (*rsltLine, ".\r"))
    return (0);

  /* See if we have at least 3 tabs. */
  for (s = *rsltLine, numTabs = 0; *s; s++)
    if (*s == '\t')
      numTabs++;
  if (numTabs > 2)
    return (1);

  /* We got something that is totaly out to lunch. */
  return (0);

}				/* LooksGood */

/*****************************************************************************
 * Unlink removes the directory entry names by 'path'.  This basicly does
 * same thing "man 2 unlink" is supposed to do, except this routine first
 * checks if 'path' exists prior to calling unlink() to prevent a core
 * dump if 'path' does not exist.  If the file 'path' does not exist this
 * routine returns 0, otherwise it returns the result of unlink().
 ****************************************************************************/
int
Unlink (char *path)
     /* path: Pathway to the file to remove. */
{
  FILE *fp;			/* Pointer to the file to remove. */

  if ((fp = fopen (path, "r")))
    {
      fclose (fp);
      return (unlink (path));
    }

  return (0);

}				/* Unlink */

/*****************************************************************************
 * ProcessMenu processes the menu entity which should be a directory.  If
 * the entity is not a directory ... well you know what they say, "garabage
 * in ... garbage out."
 * This code is ugly, but oh well such is life.
 ****************************************************************************/
void
ProcessMenu (char *selStr, char *hostStr, char *portStr, int indent)
     /* selStr: Selection string to send.
	hostStr: The host to contact.
	portStr: The port to use.
	indent: The level of indentation. */
{
  char **data,			/* The data acquired from the host via the file. */
   *rsltLine,			/* The resultant line of data. */
   *dStr,			/* The display field. */
   *sStr,			/* The selector field. */
   *hStr,			/* The host field. */
   *pStr,			/* The port field. */
   *gPlusStr;			/* The gopher+ field. */
  FILE *fPtr;			/* Pointer to the menu file. */
  int numItems,			/* Number of items in the menu. */
    error = 0,			/* Did we encounter an error? */
    i;				/* A loop counter. */

  if (!head)
    head = tail = CreateNode (selStr, hostStr, OnlyDigits (portStr));
  else
    tail =
      InsertNode (tail, CreateNode (selStr, hostStr, OnlyDigits (portStr)));

  if (!dirRoot && !indent)
    {
      PushPath ("Gopher root", selStr, hostStr, portStr);
      WaterTree (selStr, hostStr, portStr);
      indent++;
      lineNumber++;
      if (menuFlag)		/* User wants the menu displayed. */
	{
	  if (printLineNumbers)	/* User wants line numbers too. */
	    fprintf (stdout, "%7ld ", lineNumber);
	  fprintf (stdout, "Gopher root is [%s] port = %s", hostStr, portStr);
	  if (selStr[0])
	    fprintf (stdout, " branch %s", selStr);
	  fprintf (stdout, "\n");
	}
      dirRoot = BuildDirTree (dirRoot);
    }

  if (!(error = ContactHost (hostStr, Str2Int (portStr))))
    {
      /* Write the information to a temporary file. */
      if ((fPtr = fopen (tmpfilename, "w")))
	{
	  SendString (selStr);
	  SendString ("\r\n");
	  for (numItems = 0; LooksGood (&rsltLine, rdPtr); numItems++)
	    fprintf (fPtr, "%s",
		     (buildDataFile) ? HostField2Lower (rsltLine) : rsltLine);
	  CloseReadNwriter ();
	  fclose (fPtr);
	  if (buildDataFile)
	    DoSystemCall (Mysprint (catcommand, tmpfilename, fileName));
	  if (debug)
	    fprintf (rptPtr,
		     "Connections are now closed with %d item%c found\n",
		     numItems, numItems > 1 ? 's' : '\0');
	}
      else
	error =
	  fprintf (rptPtr, "error: ProcessMenu cannot create [%s]\n",
		   tmpfilename);

      if (veronica)
	{
	  char s[1024];
	  if ((strlen (hostStr) + strlen (portStr) + 2) < 1024)
	    {
	      sprintf (s, "%s\t%s", hostStr, portStr);
	      BuildTree (&vRoot, s, (long) -1);
	    }
	  else			/* Better safe than sorry. */
	    {
	      fprintf (rptPtr, "error: attempted to overwrite memory\n");
	      exit (1);
	    }
	}

      /* Read the temporary file into a dynamic array. */
      if (!error
	  && (data = malloc (numItems * sizeof (char *))))
	{
	  if ((fPtr = fopen (tmpfilename, "r")))
	    {
	      for (i = 0; i < numItems && !error; i++)
		if ((data[i] = 
		     malloc (strlen (rsltLine = GetString (fPtr)) + 1)))
		  strcpy (data[i], rsltLine);
		else
		  {
		    error =
		      fprintf (rptPtr,
			       "error: ProcessMenu cannot get memory for the %dth element\n", i);
		    for (; i; i--)
		      free (data[i]);
		    free (data);
		  }
	      fclose (fPtr);
	      if (Unlink (tmpfilename))
		fprintf (rptPtr, "error: %d could delete [%s]\n",
			 errno, tmpfilename);
	    }
	  else
	    error =
	      fprintf (rptPtr, "error: ProcessMenu cannot read [%s]\n",
		       tmpfilename);
	}
      else if (!error && numItems)
	error =
	  fprintf (rptPtr,
		   "error: ProcessMenu cannot get memory for the %d data items\n", numItems);

      if (!error)
	{
	  for (i = 0; i < numItems; i++)
	    {
	      dStr = MyStrTok (data[i], '\t');
	      sStr = EmptyStringCheck (MyStrTok ((char *) NULL, '\t'));
	      hStr = StrToLower (MyStrTok ((char *) NULL, '\t'));

	      /* Make sure we are gopher+ savvy. */
	      pStr = OnlyDigits (gPlusStr = MyStrTok ((char *) NULL, '\0'));
	      gPlusStr = pStr + strlen (pStr) + 1;

	      /* Make sure we were able to parse the line correctly. */
	      if (!hStr[0] || !pStr[0])	/* Yipes!  Skip this item. */
		continue;

	      /* Now make sure we don't try to traverse a gopher+ ASK block directory. */
	      if (dStr[0] == A_DIRECTORY && strchr (gPlusStr, '?'))
		{
		  if (debug)
		    fprintf (rptPtr,
			     "Encountered an ASK block directory: SKIPPING\n");
		  continue;
		}
	      
	      if (debug)
		{
		  fprintf (rptPtr, "\tdStr = [%s]\n", dStr);
		  fprintf (rptPtr, "\tsStr = [%s]\n", sStr);
		  fprintf (rptPtr, "\thStr = [%s]\n", hStr);
		  fprintf (rptPtr, "\tpStr = [%s]\n", pStr);
		}

	      if (listHosts)
		{
		  char s[1024];
		  if ((strlen (hStr) + strlen (pStr) + 2) < 1024)
		    {
		      sprintf (s, "%s\t%s", hStr, pStr);
		      BuildTree (&root, s, (long) -1);
		    }
		  else		/* Better safe than sorry. */
		    {
		      fprintf (rptPtr,
			       "error: attempted to overwrite memory\n");
		      exit (1);
		    }
		}
	      
	      if (menuFlag && Host2Search (hStr))
		{
		  PrintItem (dStr, indent);
		  if (info4allItems
		      || (info4dirsOnly && dStr[0] == A_DIRECTORY))
		    fprintf (stdout, "\t%s %s", hStr, pStr);
		  fprintf (stdout, "\n"), ++lineNumber;
		}

	      if (DoWeTraverse (dStr[0], sStr, hStr, pStr, indent + 1))
		{
		  PushPath (dStr + 1, sStr, hStr, pStr);
		  if (!CantGet2Host (nogoHead, EMPTYSTRING, hStr, pStr))
		    ProcessMenu (sStr, hStr, pStr, indent + 1);
		  else
		    {
		      fprintf (rptPtr,
			       "warning: ProcessMenu could not previously connect to %s %s\n",
			       hStr, pStr);
		      PrintPath ();
		    }
		  PopPath ();
		}
	    }
	  /* Free up the memory we acquired. */
	  for (i = 0; i < numItems; i++)
	    free (data[i]);
	  free (data);
	}
    }
  else				/* Keep track of hosts we can't connect to. */
    {
      PostContactHostError (error, hostStr, portStr);
      PrintPath ();

      if (!nogoHead)
	nogoHead = nogoTail = CreateNode (EMPTYSTRING, hostStr, portStr);
      else
	nogoTail =
	  InsertNode (nogoTail, CreateNode (EMPTYSTRING, hostStr, portStr));
    }

  if (head != tail)
    tail = RemoveNode (tail);

}				/* ProcessMenu */

/*****************************************************************************
 * PostTime2Process simply prints the time required for a given run.  Where
 * the run could be building the datafile, building the tables, or starting
 * up as a search engine.
 ****************************************************************************/
void
PostTime2Process (void)
{
  time_t theTime,		/* The current time. */
    timeUsed;			/* The time in seconds required for a given run. */
  int hours,			/* The hours for the process. */
    minutes,			/* The minutes for the process. */
    seconds;			/* The seconds for the process. */

  time (&theTime);
  timeUsed = theTime - startTime;
  if (doSearch)
    fprintf (rptPtr, "The time required to load the index table took ");
  else if (buildIndex)
    fprintf (rptPtr, "The time required to build the index table took ");
  else
    fprintf (rptPtr, "The time required to process the menus of %s took ",
	     initialHost);
  hours = (int) (timeUsed / 3600);
  timeUsed -= hours * 3600;
  minutes = (int) (timeUsed / 60);
  timeUsed -= minutes * 60;
  seconds = (int) timeUsed;
  fprintf (rptPtr, "%d:%d:%d\n", hours, minutes, seconds);

}				/* PostTime2Process */

/*****************************************************************************
 * ValidVariableCheck returns true if the variables are valid and false
 * otherwise.  This routine is a redunant valididy checker and gets called
 * via ReadConfFile().
 ****************************************************************************/
int
ValidVariableCheck (void)
{
  int error = 0;		/* Did we get an error? */
  
  if (!hostname)
    if (!(hostname = GetHostName (HOSTDOMAIN)))
      {
	fprintf (rptPtr, "error: cannot get name of this machine\n");
	fprintf (rptPtr, "       HOSTDOMAIN = \"%s\" in jugtail.h may be wrong.\n",
			HOSTDOMAIN);
	error = 1;
      }
  if (!jugtailhelp)
    jugtailhelp = MakeStr (jugtailhelp, JUGTAILHELP);
  if (!errorhost)
    errorhost = MakeStr (errorhost, ERRORHOST);
  if ((strlen (USAGERROR) + strlen (jugtailhelp)) > 1000 ||
      (strlen (VERSION) + strlen (jugtailhelp)) > 975)
    {
      fprintf (rptPtr, "error: too many characters in [%s], [%s], or [%s]\n",
	       jugtailhelp, USAGERROR, VERSION);
      error = 1;
    }

  return (!error);

}				/* ValidVariableCheck */

/*****************************************************************************
 * SpankDuplicateEntries is a quick and dirty hack to ensure no duplicate
 * entries exist in the data file.  The system call to 'sort' works great
 * except duplicate entries can exist if one is a gopher0 item and the other
 * a gopher+ item.  So, what this routine does is determine if we have 2
 * entries pointing to the same gopher0 and gopher+ item, and if so keeps
 * the gopher+ item.  Basicly what I am doing here is grabbing two lines
 * from the data file, and parsing each line into the display part 'dStrN'
 * and the rest 'restN', where 'restN' is the selector string, host string,
 * port string and optional gopher+ items.  A duplicate item has the
 * selector, host, and port which are the same, and when this routine gets
 * called the only thing different will be the gopher+ stuff at the end of
 * the line.  Then I move the second line into the first line and do the
 * comparision stuff all over again.
 * 
 * I should mention that by the time this routine is called the data file
 * will have been sorted such that if there exists a gopher0 and gopher+
 * items pointing to the same thing, the first line is the gopher0 item and
 * the second line is the gopher+ item.
 *
 * This routine does NOT make the most effecient use of memory, due to the
 * copying of 'line2' into 'line1', and the duplicated call to MyStrTok() to
 * re-parse the line.  Oh well!  At least the sucker works.
 *
 * This routine removes the duplicate entries from file 'f1' and stores the
 * results in file 'f2'.
 ****************************************************************************/
void
SpankDuplicateEntries (char *f1, char *f2)
     /* f1: Temporary file name.
        f2: Name of the data file. */
{
  FILE *fin,			/* Pointer to the data file. */
    *fout;			/* Pointer to the temp file. */
  char line1[BUFFERSIZE],	/* The first line we're looking at. */
    line2[BUFFERSIZE],		/* The second line we looking at. */
    *dStr1, *rest1,		/* The parsed first line. */
    *dStr2, *rest2;		/* The parsed second line. */
  int done = 0;			/* Are we done yet? */

  if ((fin = fopen (f1, "r")))
    {
      if ((fout = fopen (f2, "w")))
	{
	  if (ReadLine (fin, line1, BUFFERSIZE) != EOF)
	    {
	      /* Parse the first line. */
	      dStr1 = MyStrTok (line1, '\t');
	      rest1 = MyStrTok ((char *) NULL, '\0');

	      while (!done && (ReadLine (fin, line2, BUFFERSIZE) != EOF))
		{
		  /* Parse the second line. */
		  dStr2 = MyStrTok (line2, '\t');
		  rest2 = MyStrTok ((char *) NULL, '\0');

		  if (!strncmp (rest1, rest2, strlen (rest1) - 2))
		    {
		      fprintf (fout, "%s\t%s\n", dStr2, rest2);
		      done = ReadLine (fin, line1, BUFFERSIZE) == EOF;
		    }
		  else
		    {
		      fprintf (fout, "%s\t%s\n", dStr1, rest1);
		      sprintf (line1, "%s\t%s", dStr2, rest2);
		    }

		  /* Re-parse the first line.  Yuck! */
		  dStr1 = MyStrTok (line1, '\t');
		  rest1 = MyStrTok ((char *) NULL, '\0');
		}
	      fprintf (fout, "%s\t%s\n", dStr1, rest1);
	    }
	  fclose (fout);
	}
      else
	fprintf (rptPtr, "error: could not open [%s] for writing.\n", f2);
      fclose (fin);
    }
  else
    fprintf (rptPtr, "error: could not open [%s] for reading.\n", f1);

}				/* SpankDuplicateEntries */

/*****************************************************************************
 * PrintJugtailConf prints the jugtail configuration variables if debug is
 * true.
 ****************************************************************************/
void
PrintJugtailConf ()
{
  if (debug)
    {
      fprintf (rptPtr, "Using [%s] as hostname\n",
	       hostname ? hostname : "NULL");
      fprintf (rptPtr, "Using [%s] as jugtailhelp\n",
	       jugtailhelp ? jugtailhelp : "NULL");
      fprintf (rptPtr, "Using [%s] as errorhost\n",
	       errorhost ? errorhost : "NULL");
      fprintf (rptPtr, "Using [%d] as port2use\n", port2use);
      fprintf (rptPtr, "Using [%d] as maxprocs\n", maxprocs);
      fprintf (rptPtr, "Using [%s] as delimiters\n",
	       delimiters ? delimiters : "NULL");
      fprintf (rptPtr, "Using [%d] as maxitems2return\n", maxitems2return);
      fprintf (rptPtr, "Using [%s] as defaultboolop\n",
	       defaultboolop ? defaultboolop : "NULL");
      fprintf (rptPtr, "Using [%s] as jugtailhelptitl\n",
	       jugtailhelptitl ? jugtailhelptitl : "NULL");
      fprintf (rptPtr, "Using [%s] as usagerror\n",
	       usagerror ? usagerror : "NULL");
      fprintf (rptPtr, "Using [%s] as wildcarderr\n",
	       wildcarderr ? wildcarderr : "NULL");
      fprintf (rptPtr, "Using [%s] as tomnyprcserr\n",
	       tomnyprcserr ? tomnyprcserr : "NULL");
      fprintf (rptPtr, "Using [%s] as noforkerr\n",
	       noforkerr ? noforkerr : "NULL");
      fprintf (rptPtr, "Using [%s] as gtstrerr\n",
	       gtstrerr ? gtstrerr : "NULL");
      fprintf (rptPtr, "Using [%s] as readerr\n",
	       readerr ? readerr : "NULL");
      fprintf (rptPtr, "Using [%s] as tmpfilename\n",
	       tmpfilename ? tmpfilename : "NULL");
      fprintf (rptPtr, "Using [%s] as catcommand\n",
	       catcommand ? catcommand : "NULL");
      fprintf (rptPtr, "Using [%s] as sortcommand\n",
	       sortcommand ? sortcommand : "NULL");
      fprintf (rptPtr, "Using [%s] as touchcommand\n",
	       touchcommand ? touchcommand : "NULL");
    }

}				/* PrintJugtailConf */

/*****************************************************************************
 * InitializeTheWorld initializes the global variables to either 0, NULL, or
 * the respective default value and always returns true.
 ****************************************************************************/
int
InitializeTheWorld (void)
{
  rptPtr = stderr;

  initialHost = selStr = hostStr = portStr =
    fileName = userName = logFile = veronica = (char *) NULL;

  numSearchHosts = debug = info4dirsOnly = info4allItems = listHosts =
    listHostsNPorts = buildDataFile = printLineNumbers = printDTree =
    printDTreeDirs = time2process = buildIndex = doSearch = 0;

  menuFlag = 1;

  selStr = EMPTYSTRING;
  portStr = DEFAULTPORT;

  head = tail = nogoHead = nogoTail = (List *) NULL;

  hostname = MakeStr (hostname, HOSTDOMAIN);
  jugtailhelp = MakeStr (jugtailhelp, JUGTAILHELP);
  errorhost = MakeStr (errorhost, ERRORHOST);
  port2use = PORT2USE;
  maxprocs = MAXPROCS;
  delimiters = MakeStr (delimiters, DELIMITERS);
  maxitems2return = MAXITEMS2RETURN;
  defaultboolop = MakeStr (defaultboolop, DEFAULTBOOLOP);
  jugtailhelptitl = MakeStr (jugtailhelptitl, JUGTAILHELPTITL);
  usagerror = MakeStr (usagerror, USAGERROR);
  wildcarderr = MakeStr (wildcarderr, WILDCARDERR);
  tomnyprcserr = MakeStr (tomnyprcserr, TOMNYPRCSERR);
  noforkerr = MakeStr (noforkerr, NOFORKERR);
  gtstrerr = MakeStr (gtstrerr, GTSTRERR);
  readerr = MakeStr (readerr, READERR);
  tmpfilename = MakeStr (tmpfilename, TMPFILENAME);
  catcommand = MakeStr (catcommand, CATCOMMAND);
  sortcommand = MakeStr (sortcommand, SORTCOMMAND);
  touchcommand = MakeStr (touchcommand, TOUCHCOMMAND);

  return (1);

}				/* InitializeTheWorld */

/*****************************************************************************
 * GetCommandLine returns a string which makes up the command line.
 * This routine is used so we write a copy of the command line to the
 * jugtail.pid file which allows an easy use for SIGHUP and restarting
 * jugtail.  The file jugtail.pid is only used when jugtail is started
 * as a search engine with the -S flag.
 ****************************************************************************/
char *
GetCommandLine (int argc, char *argv[])
{
  int i;			/* A loop counter. */
  size_t len;			/* Length of the string to create. */
  char *str;			/* The string to return. */

  for (i = len = 0; i < argc; i++)
    len += strlen (argv[i]) + 1;

  if ((str = malloc (len)))
    {
      for (i = 0; i < len; i++)	/* Null the whole string. */
	str[i] = '\0';

      for (i = 0; i < argc; i++)	/* Make up the string. */
	{
	  strcat (str, argv[i]);
	  if (i < (argc - 1))
	    strcat (str, " ");
	}
      
      return (str);		/* Return the sucker. */
    }
  else				/* What the hell? */
    {
      fprintf (rptPtr, "error: GetCommandLine could not get memory\n");
      return ((char *) NULL);
    }

}				/* GetCommandLine */

/*****************************************************************************
 * ControlCAbort does some cleaning up and is only called if Control-C
 * is encountered and we are not debugging.
 ****************************************************************************/
static void
ControlCAbort (int sig)
{
  if (debug)
    fprintf (rptPtr,
	     "\nControl-C encountered: debug is on, not cleaning up.\n");
  else
    {
      fprintf (rptPtr, "\nControl-C encountered: cleaning up and aborting.\n");

      /* Make sure the temporary file is history. */
      if (Unlink (tmpfilename))
	fprintf (rptPtr, "error: %d could delete [%s]\n", errno, tmpfilename);
    }
  exit (1);
  
}				/* ControlCAbort */

/*****************************************************************************
 * main is the heart of this program and is simply the traffic director
 * calling the appropriate routines the user wants jugtail to do.
 ****************************************************************************/
int
main (int argc, char *argv[])
{
  char *cmdLine;		/* The entire command line. */
  time_t clock;			/* The current time. */

  if (!InitializeTheWorld ())
    {
      fprintf (rptPtr, "error: could not initialize the environment\n");
      if (debug)
	fprintf (rptPtr, "could not initialize the environment\n");
      exit (1);
    }

  cmdLine = GetCommandLine (argc, argv);

  if (!GetArguments (argc, argv, &fileName, &logFile))	/* Get out of Dodge. */
    return (-1);

  /* Handle the jugtailConf.conf configuration file if it exists. */
  if (initialHost)
    vctlHost = initialHost;	/* To support the "Disallow:" directive. */
  else
    vctlHost = EMPTYSTRING;	/* Must be running as a search engine. */
  vctlPort = portStr;		/* To support the "Disallow:" directive. */
  if (debug)
    {
      PrintJugtailConf ();
      fprintf (rptPtr, "vctlHost = [%s], vctlPort = [%s]\n", vctlHost,
	       vctlPort);
    }
  ReadConfFile (JUGTAILCONF);
  if (debug)
    PrintTheList ();

  if (doSearch)			/* Running with the -S flag. */
    DoSearch (cmdLine, fileName, logFile);	/* Never returns. */

  /* Set things up so we clean up if Control-C is encountered. */
  signal (SIGINT, ControlCAbort);

  if (time2process)		/* Running with the -t flag. */
    time (&startTime);

  if (logFile)			/* Open the logFile file for appending. */
    {
      if (!(rptPtr = fopen (logFile, "a")))
	fprintf (rptPtr = stderr,
		 "warning: could not open [%s] writing to stderr\n", logFile);
      time (&clock);
      fprintf (rptPtr,
	       "------------------------------------------------------\n");
      fprintf (rptPtr, "%s%s\n", ctime (&clock), cmdLine);
    }

  if (buildIndex)		/* Build the index tables and bail out. */
    {
      if (CreateWordsTree (fileName))
	{
	  MakeHashTables (fileName, root);
	  if (time2process)
	    PostTime2Process ();
	  return (0);
	}
      else
	return (-1);
    }
  if (cmdLine)			/* Don't need this anymore. */
    free (cmdLine);

  if (buildDataFile)		/* Make sure we start with a fresh data file. */
    if (Unlink (tmpfilename))
      fprintf (rptPtr, "error: %d could delete [%s]\n", errno, tmpfilename);

  ProcessMenu (selStr, hostStr, portStr, FIRSTMENU);

  if (buildDataFile)		/* Make sure we remove duplicate entries. */
    {
      if (!logFile)
	fprintf (rptPtr, "Removing any duplicates from [%s].\n", fileName);
      DoSystemCall (Mysprint (sortcommand, tmpfilename, fileName));
      SpankDuplicateEntries (tmpfilename, fileName);
      if (Unlink (tmpfilename))
	fprintf (rptPtr, "error: %d could delete [%s]\n", errno, tmpfilename);
    }

  if (listHosts || listHostsNPorts)	/* Print the hosts tree. */
    {
      fprintf (rptPtr, "Hosts %saccessed via %s\n",
	       (listHostsNPorts) ? "and ports " : "", hostStr);
      PrintTree (root, listHosts + listHostsNPorts);
    }

  if (printDTree)		/* Print the directory tree. */
    {
      fprintf (rptPtr, "Directory tree with the %s\n",
	       (printDTreeDirs) ? "directories being served" :
	       "number of directories served");
      PrintDirTree (dirRoot, PSTR);
    }

  if (nogoHead)			/* Print the hosts we couldn't connect with. */
    {
      fprintf (rptPtr, "Could not connect to the following host(s):\n");
      while (nogoHead)
	{
	  fprintf (rptPtr, "    %s %s\n", nogoHead->info.hStr,
		   nogoHead->info.pStr);
	  nogoHead = nogoHead->next;
	}
    }

  if (buildDataFile)		/* Report on the number of items we have. */
    {
      if (!logFile)
	{
	  fprintf (rptPtr,
		   "[%s] now contains the information with duplicates removed\n", fileName);
	  fprintf (rptPtr,
		   "and points to %ld different gopher menu items\n",
		   NumberOfLines (fileName));
	}
      else
	{
	  time (&clock);
	  fprintf (rptPtr,
		   "jugtail points to %ld gopher menu items and was rebuilt %s",
		   NumberOfLines (fileName), ctime (&clock));
	}
    }

  if (veronica)			/* Build the data file for veronica. */
    CreateVeronicaFile (vRoot);

  if (logFile)			/* All finished with the log. */
    fclose (rptPtr);

  /* Make sure the temporary file is history. */
  if (Unlink (tmpfilename))
    fprintf (rptPtr, "error: %d could delete [%s]\n", errno, tmpfilename);

  if (time2process)		/* Finished with the -t flag. */
    PostTime2Process ();

  return (0);

}				/* main */
