/*
    getargs.c, written by Rhett "Jonzy" Jones 

    Jonzy's Universal Gopher Hierarchy Excavation And Display.
    Excavates through gopher menus and displays the hierarchy
    of the menus encountered

    Copyright (C) 1993, 1994 University of Utah Computer Center.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program (look for the file called COPYING);
    if not, write to the Free Software Foundation, Inc.,
        51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
	                                                                 */

/* Description:	Gets the various arguments from the command line for use
 *		with jugtail.
 */

#include "stdinc.h"
#include "jugtail.h"

static void ShowArguments (void);
static short UsageError (char *progName);
static short PrintVersion (char *progName);
static short GetParamsFromStdin (void);
static int GetArg (char ***argv, char **argument);
int GetArguments (int argc, char *argv[], char **fileName, char **logFile);

#define VERONICAXTENTION	"veronica"

extern List *CreateNode ();	/* Defined in "jugtailConf.c". */
extern char *logFile;		/* Defined in "searchCmnds.c". */

/*****************************************************************************
 * ShowArguments simply displays the status of the arguments passed in, and
 * is solely for debugging.
 ****************************************************************************/
static void
ShowArguments (void)
{
  int i;			/* A loop counter. */
  List *t;			/* Pointer into the list of nogos. */

  fprintf (stderr, "flags are [%c%c%c%c%c%c%c%c%c%c%c%c%c]",
	   printDTree ? 'a' : '\0',
	   printDTreeDirs ? 'A' : '\0',
	   buildIndex ? 'B' : '\0',
	   debug ? 'd' : '\0',
	   listHosts ? 'h' : '\0',
	   listHostsNPorts ? 'H' : '\0',
	   info4dirsOnly ? 'i' : '\0',
	   info4allItems ? 'I' : '\0',
	   maxprocs ? 'M' : '\0',
	   menuFlag ? 'm' : '\0',
	   printLineNumbers ? 'n' : '\0',
	   doSearch ? 'S' : '\0', time2process ? 't' : '\0');
  if (buildDataFile)
    {
      fprintf (stderr, " -%cb [%s]", veronica ? 'V' : '\0', fileName);
      if (strcmp (portStr, DEFAULTPORT))
	fprintf (stderr, " -p [%s]", portStr);
    }
  else if (doSearch)
    if (port2use != 3000)	/* ??? 3000 -> PORT2USE ??? */
      fprintf (stderr, " -p [%d]", port2use);

  if (selStr[0])
    fprintf (stderr, " -s [%s]", selStr);

  if (nogos) {
    for (t = nogos; t; t = t->next)
      if (t->info.sStr[0])
	fprintf (stderr, " -X [%s]", t->info.sStr);
      else
	fprintf (stderr, " -x [%s] [%s]", t->info.hStr, t->info.pStr);
  }
  if (logFile)
    fprintf (stderr, " -l [%s]", logFile);
  if (maxprocs)
    fprintf (stderr, " -M [%d]", maxprocs);
  if (buildIndex || doSearch)
    fprintf (stderr, " dataFile=[%s]", fileName);
  if (numSearchHosts)
    {
      fprintf (stderr, " hosts2search=(");
      for (i = 0; i < numSearchHosts; i++)
	fprintf (stderr, " [%s]", searchHosts[i]);
      fprintf (stderr, " )");
    }
  if (userName)
    fprintf (stderr, " -u [%s]", userName);
  if (!buildIndex && !doSearch && hostStr)
    fprintf (stderr, " host=[%s]", hostStr);
  fprintf (stderr, "\n");
  
}				/* ShowArguments */

/*****************************************************************************
 * UsageError prints the usage of the program and returns false so we can
 * terminate.
 ****************************************************************************/
static short
UsageError (char *progName)
     /* progName: Name of the program. */
{
  int numSpaces;		/* Number of spaces to print or the second
				 * part of the main usage line. */

  numSpaces = strlen ("usage: ") + strlen (progName) + 1;
  fprintf (stderr,
	   "usage: %s [-?aAdhHiImntv] [-[V]b dataFile] [-l log]"
	   " [-s selStr] [-p port#]\n", progName);
  fprintf (stderr,
	   "%*c[[-x hostNot2search port#]...] [[-X selStrNot2Search]...]\n",
	   numSpaces, ' ');
  fprintf (stderr, "%*c[- | hosts2search...] host\n", numSpaces, ' ');
  fprintf (stderr, "usage: %s -B[?dmtv] [-l log] dataFile\n",
	   progName);
  fprintf (stderr,
	   "usage: %s -S[?dtv] [-l log] [-M maxProcs] [-p port#]"
	   " [-u username] dataFile\n", progName);
  return (0);

}				/* UsageError */

/*****************************************************************************
 * PrintVersion prints the version of the program and returns false so we
 * can terminate.
 ****************************************************************************/
static short
PrintVersion (char *progName)
     /* progName: The name of this program. */
{
  fprintf (stderr, "This version of %s is %s\n", progName, PACKAGE_VERSION);
  return (0);

}				/* PrintVersion */

/*****************************************************************************
 * GetParamsFromStdin gets words from stdin until end-of-file is
 * encounterd.  This routine also takes care of backspaces.
 ****************************************************************************/
static short
GetParamsFromStdin (void)
{
  int c;			/* A character from stdin. */
  int i;			/* A loop counter. */
  short done = 0;		/* Are we done yet? */
  char *s;			/* The string we are building. */

  if (isatty (0))
    {
      fprintf (stderr, "Please enter the hosts you would like to traverse\n");
      fprintf (stderr, "separated by whitespace.  When you are finished\n");
      fprintf (stderr, "enter ^D [control-d].\n  --> ");
    }

  for (i = 0; (c = fgetc (stdin)) && !done && numSearchHosts < MAXHOSTS; i++)
    if (isspace (buffer[i] = (char) c) || c == EOF)
      {
	if (c == EOF)
	  done = 1;
	buffer[i] = '\0';

	if (i)
	  {
	    if ((s = malloc (i)))
	      searchHosts[numSearchHosts++] = StrToLower (strcpy (s, buffer));
	    else
	      {
		fprintf (stderr, "error: could not get memory for [%s]\n",
			 buffer);
		exit (-2);
	      }
	  }
	i = -1;
      }
    else if (c == '\b')		/* Take care of backspace. */
      {
	i -= 2;
	if (i < -1)
	  i = -1;
      }

  if (numSearchHosts >= MAXHOSTS)
    fprintf (stderr,
	     "warning: only the first %d hosts2search are being used.\n",
	     MAXHOSTS);

  return (1);
  
}				/* GetParamsFromStdin */

/*****************************************************************************
 * GetArg returns true if the current argument is exhausted, as in empty.
 * Otherwise this routine returns false.  If returning true we assign the
 * next argument in the list to 'argument', and adjust the list such that
 * it is pointing to the second to last character in 'argument'.
 ****************************************************************************/
static int
GetArg (char ***argv, char **argument)
     /* argv: The list of arguments. 
        argument: The argument to assign to. */
{
  if (!(*++**argv))		/* We're at the end of the list. */
    {
      *argument = *++*argv;	/* Get the next argument. */
      while (***argv && *(**argv + 1))
	++ ** argv;
      return (1);
    }
  else
    return (0);

}				/* GetArg */

/*****************************************************************************
 * GetArguments returns true if we were able to initialize variables from
 * argc and argv.  Otherwise this routine calls UsageError() which exits, if
 * something was wrong with the way the arguments were passed in.
 ****************************************************************************/
int
GetArguments (int argc, char *argv[], char **fileName, char **logFile)
     /* fileName: Name of the file to open or create.
        logFile: Name of the file to log to if searching. */
{
  char *programName,		/* The name of the program. */
    *s,				/* A temporary string. */
    *exceptionHost, *exceptionPort, *thePort;	/* The port to use. */
  int i,			/* Loop counter. */
    vFlag = 0;			/* Are we doing a veronica file? */

  thePort = (char *) NULL;

  for (programName = *argv, argc--, argv++; *argv; argv++, argc--)
    if (**argv == '-')
      if (!*++*argv)
	GetParamsFromStdin ();
      else
	for (; **argv; ++*argv)
	  switch (**argv)
	    {
	    case '?':
	      return (UsageError (programName));
	    case 'A':
	      printDTree = printDTreeDirs = 1;
	      break;
	    case 'a':
	      printDTree = 1;
	      break;
	    case 'B':
	      buildIndex = 1;
	      break;
	    case 'b':
	      buildDataFile = 1;
	      if (argc > 1 && GetArg (&argv, fileName))
		argc--;
	      else
		return (UsageError (programName));
	      break;
	    case 'd':
	      debug = 1;
	      break;
	    case 'H':
	      listHosts = listHostsNPorts = 1;
	      break;
	    case 'h':
	      listHosts = 1;
	      break;
	    case 'I':
	      info4allItems = 1;
	      break;
	    case 'i':
	      info4dirsOnly = 1;
	      break;
	    case 'l':
	      if (argc > 1 && GetArg (&argv, logFile))
		argc--;
	      else
		return (UsageError (programName));
	      break;
	    case 'M':
	      if (argc > 2 && GetArg (&argv, &s))
		{
		  argc--;
		  maxprocs = 0;
		  for (; *s; s++)
		    if (!isdigit (*s))
		      return (UsageError (programName));
		    else
		      maxprocs = maxprocs * 10 + *s - '0';
		  if (!maxprocs)
		    {
		      fprintf (stderr,
			       "error: maxprocs must be greater than 0\n");
		      return (UsageError (programName));
		    }
		}
	    case 'm':
	      menuFlag = !menuFlag;
	      break;
	    case 'n':
	      printLineNumbers = 1;
	      break;
	    case 'p':
	      if (argc > 1 && GetArg (&argv, &thePort))
		{
		  argc--;
		  for (s = thePort; *s; s++)	/* Validate the port. */
		    if (!isdigit (*s))
		      return (UsageError (programName));
		}
	      else
		return (UsageError (programName));
	      break;
	    case 'S':
	      doSearch = 1;
	      break;
	    case 's':
	      if (argc > 1 && GetArg (&argv, &selStr))
		argc--;
	      else
		return (UsageError (programName));
	      break;
	    case 't':
	      time2process = 1;
	      break;
	    case 'V':
	      vFlag = 1;
	      break;
	    case 'v':
	      return (PrintVersion (programName));
	    case 'X':
	      if (argc > 1 && GetArg (&argv, &s))
		{
		  argc--;
		  if (!nogos)
		    nogos = CreateNode (s, EMPTYSTRING, EMPTYSTRING);
		  else
		    {
		      List *t = CreateNode (s, EMPTYSTRING, EMPTYSTRING);
		      t->next = nogos;
		      nogos = t;
		    }
		}
	      else
		return (UsageError (programName));
	      break;
	    case 'x':
	      if (argc > 2 && GetArg (&argv, &s))
		{
		  argc -= 2;
		  exceptionHost = StrToLower (s);
		  if (GetArg (&argv, &exceptionPort))
		    for (s = exceptionPort; *s; s++)	/* Validate the port. */
		      {
			if (!isdigit (*s))
			  return (UsageError (programName));
		      }
		  else
		    return (UsageError (programName));
		  if (!nogos)
		    nogos =
		      CreateNode (EMPTYSTRING, exceptionHost, exceptionPort);
		  else
		    {
		      List *t =
			CreateNode (EMPTYSTRING, exceptionHost,
				    exceptionPort);
		      t->next = nogos;
		      nogos = t;
		    }
		}
	      else
		return (UsageError (programName));
	      break;
	    case 'u':
	      if (argc > 1 && GetArg (&argv, &userName))
		argc--;
	      else
		return (UsageError (programName));
	      break;
	    default:
	      return (UsageError (programName));
	    }
    else if (argc == 1)
      if (buildIndex || doSearch)
	*fileName = *argv;
      else
	initialHost = hostStr = *argv;
    else if (numSearchHosts < MAXHOSTS)
      searchHosts[numSearchHosts++] = StrToLower (*argv);
    else
      fprintf (stderr, "warning: Too many hosts to search.  Ignoring %s\n",
	       *argv);
  
  if (buildDataFile)		/* Tweek the menu flag. */
    menuFlag = !menuFlag;
  
  if (thePort)			/* User wants to use a different port. */
    {
      if (!doSearch && !buildIndex)
	portStr = thePort;
      else if (doSearch)
	port2use = atoi (thePort);
      else if (buildIndex)
	return (UsageError (programName));
    }
  /* If no hosts to search just search the host. */
  if (!numSearchHosts && !buildIndex && !doSearch)
    searchHosts[numSearchHosts++] = hostStr;

  if (!*fileName && !hostStr)	/* Make sure we got the last argument. */
    return (UsageError (programName));

  /* Make sure the proper flags have been used correctly:                         *
   * jugtail [-?aAdhHiImnoOtv] [-[V]b dataFile] [-l log] [-s selStr] [-p port#]   *
   *         [-x hostNot2search port#] [-X selStrNot2Search]                      *
   *         [- | hosts2search...] host                                           *
   * jugtail -B[?dtv] [-l log] dataFile                                           *
   * jugtail -S[?dtv] [-l log] [-M maxProcs] [-p port#] [-u username] dataFile    */
  if ((buildIndex && doSearch) || (!doSearch && userName)
      || (buildIndex && thePort))
    return (UsageError (programName));
  if ((buildIndex || doSearch) && (nogos ||
				   selStr[0] ||
				   printDTree ||
				   printDTreeDirs ||
				   listHosts ||
				   listHostsNPorts ||
				   info4allItems || info4dirsOnly ||
/*					 !menuFlag ||			*/
				   printLineNumbers || numSearchHosts))
    return (UsageError (programName));

  if (vFlag || doSearch)
    {
      if (vFlag && !buildDataFile)
	return (UsageError (programName));
      else if ((veronica =
		malloc (strlen (*fileName) + strlen ("veronica") + 2)))
	{
	  strcpy (veronica, *fileName);
	  strcat (veronica, ".");
	  strcat (veronica, VERONICAXTENTION);
	}
      else
	{
	  fprintf (stderr, "error: could not get memory [%s.veronica] name\n",
		   *fileName);
	  return (0);
	}
    }
  if (debug)
    {
      ShowArguments ();
      if (*logFile)		/* Turn off writting to the logFile file. */
	{
	  fprintf (stderr, "-l %s turned off for debugging\n", *logFile);
	  *logFile = (char *) NULL;
	}
    }

  /* Now make sure that none of the searchHosts exceed 255 characters. */
  for (i = 0; i < numSearchHosts; i++)
    if (strlen (searchHosts[i]) > 255)
      {
	fprintf (stderr, "error: [%s] has too many characters\n",
		 searchHosts[i]);
	return (0);
      }

  return (1);

}				/* GetArguments */
