/*
	This file is part of jrisk.

	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 16-feb-2005 by Alessio Treglia
 * Copyright  2005 Alessio Treglia
 */

import java.io.IOException;

/**
 * Implementa un help di programma. Stampa l'elenco dei comandi
 * oppure l'aiuto per il singolo comando.
 * 
 * @author Alessio Treglia
 * @version 1.0.1
 */
public class ComandoAiuto extends Comando {
    /** Nome del comando */
    private final String nomeComando = "aiuto";
    /** Contiene il nome del comando per il quale viene richiesto l'aiuto */
    private String aiutoComando;
    
    /**
     * Costruisce un comando di aiuto per l'utente.
     * 
     * @param aiutoComando comando per il quale si chiede l'aiuto
     */
    public ComandoAiuto(String aiutoComando) {
        this.aiutoComando = aiutoComando;
    }
    /* (non-Javadoc)
     * @see Comando#getNomeComando()
     */
    public String getNomeComando() { return this.nomeComando; }

    /* (non-Javadoc)
     * @see Comando#esegui(Giocatore, Tabellone)
     */
    public boolean esegui(Giocatore giocatore, Tabellone planisfero)
            throws IOException {
        if(this.aiutoComando == null)
            stampaAiutoGenerico(giocatore);
        else {
            ElencoComandi elenco = new ElencoComandi();
            if(elenco.comandoValido(aiutoComando))
                stampaAiutoPerComando(giocatore, aiutoComando);
            else
                new ComandoNonValido().esegui(giocatore,planisfero);
        }
        return false;
    }
    /**
     * Stampa l'elenco dei comandi validi e l'URL della
     * casa editrice di Risiko.
     * 
     * @param g <code>Giocatore</code> che ha invocato il comando
     */
    private void stampaAiutoGenerico(Giocatore g) {
        g.schermo.stampa("I comandi accettati dalla console sono: ");
        g.schermo.stampaln(ElencoComandi.elencoTuttiComandi());
        g.schermo.stampaln("Le regole del gioco le potete trovare su " +
                "http://www.hasbro.com/risk/rules.cfm");
    }
    
    private void stampaAiutoPerComando(Giocatore g, String str) {
        if(str.equals("attacca"))
            g.schermo.stampaln(aiutoPerAttacca());
        if(str.equals("sposta"))
            g.schermo.stampaln(aiutoPerSposta());
        if(str.equals("esci"))
            g.schermo.stampaln(aiutoPerEsci());
        if(str.equals("stampa"))
            g.schermo.stampaln(aiutoPerStampa());
        if(str.equals("passa"))
            g.schermo.stampaln(aiutoPerPassa());
        if(str.equals("aiuto"))
            g.schermo.stampaln(aiutoPerAiuto());
        if(str.equals("versione"))
            g.schermo.stampaln(aiutoPerVersione());
    }
    /**
     * Stampa l'help del comando attacca
     * 
     * @return una <code>String</code> contenente la descrizione e l'uso del comando attacca
     */
    private String aiutoPerAttacca() {
        return ("Attacca un territorio avversario da un territorio confinante.\n" +
            "Uso: attacca \"territorio_base\" \"territorio_avversario\"");
    }
    /**
     * Stampa l'help del comando sposta
     * 
     * @return una <code>String</code> contenente la descrizione e l'uso del comando sposta
     */
    private String aiutoPerSposta() {
        return ("Sposta un numero valido di proprie armate da un territorio a un altro confinante.\n" +
            "Uso: sposta \"territorio_partenza\" \"territorio_arrivo\"");
    }
    /**
     * Stampa l'help del comando di uscita dal gioco
     * 
     * @return una <code>String</code> contenente la descrizione e l'uso del comando esci
     */
    private String aiutoPerEsci() {
        return ("Esci da jRisk");
    }
    /**
     * Stampa l'help del comando passa turno
     * 
     * @return una <code>String</code> contenente la descrizione e l'uso del comando passa
     */
    private String aiutoPerPassa() {
        return ("Passa il turno");
    }
    /**
     * Stampa l'help del comando stampa
     * 
     * @return una <code>String</code> contenente la descrizione e l'uso del comando stampa
     */
    private String aiutoPerStampa() {
        return ("Stampa le informazioni su un giocatore, su un territorio oppure l'intero tabellone.\n" +
                "Uso: stampa [tabellone|\"nome_territorio\"|\"nome_giocatore\"]");
    }
    /**
     * Stampa l'help del comando aiuto
     * 
     * @return una <code>String</code> contenente la descrizione e l'uso del comando aiuto
     */
    private String aiutoPerAiuto() {
        return ("Stampa questa schermata di aiuto.\n" + 
                "Uso: aiuto [comando]");
    }
    
    /**
     * Stampa la versione del software
     * 
     * @return una <code>String</code> contenente la descrizione del comando versione
     */
    private String aiutoPerVersione() {
        return ("Stampa la versione corrente di jrisk");
    }
}
