#!/usr/bin/perl
#
# $Id: s.bdfsetdepth.pl 1.6 03/09/15 13:51:28-04:00 cloos@lugabout.jhcloos.org $
#
# Change the depth of a bdf font
#
# Copyright © 2003 James H. Cloos, Jr. <cloos@jhcloos.com>
#
# You may distribute under the terms of either the
# GNU General Public License or the Artistic License,
# as specified in the README file.

use Getopt::Long;
use Pod::Usage qw(pod2usage);

$NL = "\n";
$in_depth = 0;
$out_depth = 8;
$in_bitmap = 0;
@ok_depths = (1, 2, 4, 8, 16);

sub doVersion() {
    print STDERR "bdfsetdepth 1.0", $NL, $NL,
        "Copyright (C) 2003 James H. Cloos, Jr. <cloos\@jhcloos.com>", $NL,
        "This is free software; see the source for copying conditions.  There is NO", $NL,
        "warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.", $NL;
    exit;
}

GetOptions("depth=i" => \$out_depth,
	   "version" => sub { doVersion() },
	   "man"     => sub { pod2usage(-exitstatus => 0, -verbose => 2) },
	   "help"    => sub { pod2usage(0) }) or pod2usage(1);

grep(/$out_depth/, @ok_depths) or pod2usage(-exitstatus => 2, -msg => "Error: Depth must be one of [1, 2, 4, 8, 16]\n");

while (<>) {
    chomp;
    $line = $_;

    $in_bitmap = 0 unless /^[0-9A-Fa-f]+$/;

    if ($in_bitmap) {
	my @bits = ();
	my @chunks = ();
	my @out = ();
	my $j = 0;
	@bits = split('', unpack("B*", pack("H*", $line)));
	while (@bits) {
	    my $i;
	    foreach (1..$in_depth) {
		$i .= shift(@bits);
	    }
	    push @chunks, $i;
	}
	foreach $j (@chunks) {
	    # if out > in repeat j out/in times
	    # if out < in take out most significant bits of j
	    if ($out_depth < $in_depth ) {
		push @out, substr($j, 0, $out_depth);
	    } else {
		foreach (1..($out_depth/$in_depth)) {
		    push @out, $j;
		}
	    }
	}
	
	$line = uc(unpack("H*", pack("B*", join('', @out))));
    }
    
    if (/^SIZE/) {
	@line = split ' ', $line;
	$in_depth = $line[4] ? $line[4] : 1;
	$line[4] = $out_depth;
	$line = join ' ', @line;
    }

    $in_bitmap = 1 if /^BITMAP/;
    print $line, $NL;
}

__END__

=head1 NAME

bdfsetdepth - set depth of a bdf font

=head1 SYNOPSIS

bdfsetdepth [options] [file]

 Options:
    -d, --depth=DEPTH  Make depth DEPTH (1, 2, 4, 8 or 16)     
        --help         Brief help message
        --man          Full documentation
        --version      Output version information and exit

=head1 OPTIONS

=over 8

=item B<-d>, B<--depth>

Set the new depth for the font.  Available depths are 1, 2, 4, 8 or 16 bits.

=item B<--help>

Print a brief help message and exit.

=item B<--man>

Print the manual page and exit.

=item B<--version>

Print the version info and exit.

=back

=head1 DESCRIPTION

B<bdfsetdepth> sets the depth of the BDF font either specified on the
command line of passed on STDIN.

=cut
