/*
 vim: filetype=c:expandtab:shiftwidth=4:tabstop=8:softtabstop=4:fileencoding=utf-8:textwidth=99 :
*/
/*
    libzint - the open source barcode library
    Copyright (C) 2009 Robin Stuart <robin@zint.org.uk>
    Copyright (C) 2015 iwrite authors

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
*/
/* copied from png.c */
#include <locale.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "common.h"
#include <cairo.h>

extern void to_latin1(unsigned char source[], unsigned char preprocessed[]);

static void
cairo_hex_to_colour(
    double *const red,
    double *const green,
    double *const blue,
    char const i_hex_str[6])
{
    char value[3];

    value[0] = i_hex_str[0];
    value[1] = i_hex_str[1];
    value[2] = 0;
    (*red) = strtol(value, 0, 16) / 255.0;

    value[0] = i_hex_str[1];
    value[1] = i_hex_str[2];
    (*green) = strtol(value, 0, 16) / 255.0;

    value[0] = i_hex_str[3];
    value[1] = i_hex_str[4];
    (*blue) = strtol(value, 0, 16) / 255.0;

    return;
}

static void
cairo_draw_string(cairo_t *cr,
                  char const input_string[],
                  int const xposn,
                  int const yposn,
                  int const smalltext,
                  int const image_width, int const image_height)
{

    cairo_save(cr);
    cairo_move_to(cr, xposn, yposn);
    cairo_set_font_size(cr, (smalltext) ? 8.0 : 11.0);
    cairo_select_font_face(
        cr, "Serif", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL);
    cairo_show_text(cr, input_string);
    cairo_restore(cr);

    return;
}

static void
cairo_draw_bar(cairo_t *cr,
               int const xpos,
               int const xlen,
               int const ypos,
               int const ylen,
               int const image_width,
               int const image_height)
{
    int png_ypos = image_height - ypos - ylen;

    cairo_rectangle(cr, xpos, png_ypos, xlen, ylen);
    cairo_fill(cr);

    return;
}

static int
cairo_plot_maxi(struct zint_symbol *symbol, cairo_t *cr)
{
    return ZINT_ERROR_ENCODING_PROBLEM;
}

static int
cairo_plot_default(struct zint_symbol *symbol, cairo_t *cr)
{
    int textdone, main_width, comp_offset, large_bar_count;
    char textpart[10], addon[6];
    float addon_text_posn, preset_height, large_bar_height;
    int i, r, textoffset, yoffset, xoffset, latch, image_width, image_height;
    int addon_latch = 0, smalltext = 0;
    int this_row, block_width, plot_height, plot_yposn, textpos;
    float row_height, row_posn;
    int error_number = 0;
    int default_text_posn;
    int next_yposn;
    unsigned char *local_text;
    unsigned local_text_length;
    cairo_text_extents_t extents;

    local_text_length = strlen((char *) symbol->text);
    local_text = (unsigned char *) malloc(1 + local_text_length);
    memset(local_text, 0, 1 + local_text_length);

    if (symbol->show_hrt != 0) {
        /* see ticket #11 */
        /* to_latin1(symbol->text, local_text); */
        memcpy(local_text, symbol->text, local_text_length);
    } 

    textdone = 0;
    main_width = symbol->width;
    strcpy(addon, "");
    comp_offset = 0;
    addon_text_posn = 0.0;
    row_height = 0;
    if (symbol->output_options & SMALL_TEXT) {
        smalltext = 1;
    }

    if (symbol->height == 0) {
        symbol->height = 50;
    }

    large_bar_count = 0;
    preset_height = 0.0;
    for (i = 0; i < symbol->rows; i++) {
        preset_height += symbol->row_height[i];
        if (symbol->row_height[i] == 0) {
            large_bar_count++;
        }
    }

    if (large_bar_count == 0) {
        symbol->height = preset_height;
        large_bar_height = 10;
    } else {
        large_bar_height = (symbol->height - preset_height) / large_bar_count;
    }

    while (!(module_is_set(symbol, symbol->rows - 1, comp_offset))) {
        comp_offset++;
    }

    /* Certain symbols need whitespace otherwise characters get chopped off the sides */
    if ((((symbol->symbology == BARCODE_EANX) && (symbol->rows == 1))
            || (symbol->symbology == BARCODE_EANX_CC))
            || (symbol->symbology == BARCODE_ISBNX)) {
        switch (local_text_length) {
        case 13:        /* EAN 13 */
        case 16:
        case 19:
            if (symbol->whitespace_width == 0) {
                symbol->whitespace_width = 10;
            }
            main_width = 96 + comp_offset;
            break;
        default:
            main_width = 68 + comp_offset;
        }
    }

    if (((symbol->symbology == BARCODE_UPCA) && (symbol->rows == 1))
            || (symbol->symbology == BARCODE_UPCA_CC)) {
        if (symbol->whitespace_width == 0) {
            symbol->whitespace_width = 10;
            main_width = 96 + comp_offset;
        }
    }

    if (((symbol->symbology == BARCODE_UPCE) && (symbol->rows == 1))
            || (symbol->symbology == BARCODE_UPCE_CC)) {
        if (symbol->whitespace_width == 0) {
            symbol->whitespace_width = 10;
            main_width = 51 + comp_offset;
        }
    }

    latch = 0;
    r = 0;
    /* Isolate add-on text */
    if (is_extendable(symbol->symbology)) {
        for (i = 0; i < local_text_length; i++) {
            if (latch == 1) {
                addon[r] = local_text[i];
                r++;
            }
            if (symbol->text[i] == '+') {
                latch = 1;
            }
        }
    }
    addon[r] = '\0';

    if (local_text_length != 0) {
        textoffset = 9;
    } else {
        textoffset = 0;
    }
    xoffset = symbol->border_width + symbol->whitespace_width;
    yoffset = symbol->border_width;
    image_width = 2 * (symbol->width + xoffset + xoffset);
    image_height = 2 * (symbol->height + textoffset + yoffset + yoffset);

    if (((symbol->output_options & BARCODE_BOX) != 0)
            || ((symbol->output_options & BARCODE_BIND) != 0)) {
        default_text_posn = image_height - 8;
    } else {
        default_text_posn =
            image_height - 8 - symbol->border_width - symbol->border_width;
    }

    row_posn = textoffset + yoffset;
    next_yposn = textoffset + yoffset;
    row_height = 0;

    /* Plot the body of the symbol to the pixel buffer */
    for (r = 0; r < symbol->rows; r++) {
        this_row = symbol->rows - r - 1;    /* invert r otherwise plots upside down */
        row_posn += row_height;
        plot_yposn = next_yposn;
        if (symbol->row_height[this_row] == 0) {
            row_height = large_bar_height;
        } else {
            row_height = symbol->row_height[this_row];
        }
        next_yposn = (int)(row_posn + row_height);
        plot_height = next_yposn - plot_yposn;

        i = 0;
        if (module_is_set(symbol, this_row, 0)) {
            latch = 1;
        } else {
            latch = 0;
        }

        do {
            block_width = 0;
            do {
                block_width++;
            } while (module_is_set(symbol, this_row, i + block_width) ==
                     module_is_set(symbol, this_row, i));
            if ((addon_latch == 0) && (r == 0) && (i > main_width)) {
                plot_height = (int)(row_height - 5.0);
                plot_yposn = (int)(row_posn - 5.0);
                addon_text_posn = row_posn + row_height - 8.0;
                addon_latch = 1;
            }
            if (latch == 1) {
                /* a bar */
                cairo_draw_bar(cr, (i + xoffset) * 2, block_width * 2,
                               plot_yposn * 2, plot_height * 2, image_width,
                               image_height);
                latch = 0;
            } else {
                /* a space */
                latch = 1;
            }
            i += block_width;

        } while (i < symbol->width);
    }

    xoffset += comp_offset;

    if ((((symbol->symbology == BARCODE_EANX) && (symbol->rows == 1))
            || (symbol->symbology == BARCODE_EANX_CC))
            || (symbol->symbology == BARCODE_ISBNX)) {
        /* guard bar extensions and text formatting for EAN8 and EAN13 */
        switch (local_text_length) {
        case 8:     /* EAN-8 */
        case 11:
        case 14:
            cairo_draw_bar(cr, (0 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            cairo_draw_bar(cr, (2 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            cairo_draw_bar(cr, (32 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            cairo_draw_bar(cr, (34 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            cairo_draw_bar(cr, (64 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            cairo_draw_bar(cr, (66 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            for (i = 0; i < 4; i++) {
                textpart[i] = symbol->text[i];
            }
            textpart[4] = '\0';
            textpos = 2 * (17 + xoffset);

            cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                              image_width, image_height);
            for (i = 0; i < 4; i++) {
                textpart[i] = symbol->text[i + 4];
            }
            textpart[4] = '\0';
            textpos = 2 * (50 + xoffset);
            cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                              image_width, image_height);
            textdone = 1;
            switch (strlen(addon)) {
            case 2:
                textpos = 2 * (xoffset + 86);
                cairo_draw_string(cr, addon, textpos,
                                  image_height - (addon_text_posn * 2) - 13,
                                  smalltext, image_width, image_height);
                break;
            case 5:
                textpos = 2 * (xoffset + 100);
                cairo_draw_string(cr, addon, textpos,
                                  image_height - (addon_text_posn * 2) - 13,
                                  smalltext, image_width, image_height);
                break;
            }

            break;
        case 13:        /* EAN 13 */
        case 16:
        case 19:
            cairo_draw_bar(cr, (0 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            cairo_draw_bar(cr, (2 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            cairo_draw_bar(cr, (46 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            cairo_draw_bar(cr, (48 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            cairo_draw_bar(cr, (92 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);
            cairo_draw_bar(cr, (94 + xoffset) * 2, 1 * 2,
                           (4 + (int) yoffset) * 2, 5 * 2, image_width,
                           image_height);

            textpart[0] = symbol->text[0];
            textpart[1] = '\0';
            textpos = 2 * (-7 + xoffset);
            cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                              image_width, image_height);
            for (i = 0; i < 6; i++) {
                textpart[i] = symbol->text[i + 1];
            }
            textpart[6] = '\0';
            textpos = 2 * (24 + xoffset);
            cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                              image_width, image_height);
            for (i = 0; i < 6; i++) {
                textpart[i] = symbol->text[i + 7];
            }
            textpart[6] = '\0';
            textpos = 2 * (71 + xoffset);
            cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                              image_width, image_height);
            textdone = 1;
            switch (strlen(addon)) {
            case 2:
                textpos = 2 * (xoffset + 114);
                cairo_draw_string(cr, addon, textpos,
                                  image_height - (addon_text_posn * 2) - 13,
                                  smalltext, image_width, image_height);
                break;
            case 5:
                textpos = 2 * (xoffset + 128);
                cairo_draw_string(cr, addon, textpos,
                                  image_height - (addon_text_posn * 2) - 13,
                                  smalltext, image_width, image_height);
                break;
            }
            break;

        }
    }

    if (((symbol->symbology == BARCODE_UPCA) && (symbol->rows == 1))
            || (symbol->symbology == BARCODE_UPCA_CC)) {
        /* guard bar extensions and text formatting for UPCA */
        latch = 1;

        i = 0 + comp_offset;
        do {
            block_width = 0;
            do {
                block_width++;
            } while (module_is_set(symbol, symbol->rows - 1, i + block_width) ==
                     module_is_set(symbol, symbol->rows - 1, i));
            if (latch == 1) {
                /* a bar */
                cairo_draw_bar(cr, (i + xoffset - comp_offset) * 2,
                               block_width * 2, (4 + (int) yoffset) * 2, 5 * 2,
                               image_width, image_height);
                latch = 0;
            } else {
                /* a space */
                latch = 1;
            }
            i += block_width;
        } while (i < 11 + comp_offset);
        cairo_draw_bar(cr, (46 + xoffset) * 2, 1 * 2, (4 + (int) yoffset) * 2,
                       5 * 2, image_width, image_height);
        cairo_draw_bar(cr, (48 + xoffset) * 2, 1 * 2, (4 + (int) yoffset) * 2,
                       5 * 2, image_width, image_height);
        latch = 1;
        i = 85 + comp_offset;
        do {
            block_width = 0;
            do {
                block_width++;
            } while (module_is_set(symbol, symbol->rows - 1, i + block_width) ==
                     module_is_set(symbol, symbol->rows - 1, i));
            if (latch == 1) {
                /* a bar */
                cairo_draw_bar(cr, (i + xoffset - comp_offset) * 2,
                               block_width * 2, (4 + (int) yoffset) * 2, 5 * 2,
                               image_width, image_height);
                latch = 0;
            } else {
                /* a space */
                latch = 1;
            }
            i += block_width;
        } while (i < 96 + comp_offset);
        textpart[0] = symbol->text[0];
        textpart[1] = '\0';
        textpos = 2 * (-5 + xoffset);
        cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                          image_width, image_height);
        for (i = 0; i < 5; i++) {
            textpart[i] = symbol->text[i + 1];
        }
        textpart[5] = '\0';
        textpos = 2 * (27 + xoffset);
        cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                          image_width, image_height);
        for (i = 0; i < 5; i++) {
            textpart[i] = symbol->text[i + 6];
        }
        textpart[6] = '\0';
        textpos = 2 * (68 + xoffset);
        cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                          image_width, image_height);
        textpart[0] = symbol->text[11];
        textpart[1] = '\0';
        textpos = 2 * (100 + xoffset);
        cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                          image_width, image_height);
        textdone = 1;
        switch (strlen(addon)) {
        case 2:
            textpos = 2 * (xoffset + 116);
            cairo_draw_string(cr, addon, textpos,
                              image_height - (addon_text_posn * 2) - 13, smalltext,
                              image_width, image_height);
            break;
        case 5:
            textpos = 2 * (xoffset + 130);
            cairo_draw_string(cr, addon, textpos,
                              image_height - (addon_text_posn * 2) - 13, smalltext,
                              image_width, image_height);
            break;
        }

    }

    if (((symbol->symbology == BARCODE_UPCE) && (symbol->rows == 1))
            || (symbol->symbology == BARCODE_UPCE_CC)) {
        /* guard bar extensions and text formatting for UPCE */
        cairo_draw_bar(cr, (0 + xoffset) * 2, 1 * 2, (4 + (int) yoffset) * 2,
                       5 * 2, image_width, image_height);
        cairo_draw_bar(cr, (2 + xoffset) * 2, 1 * 2, (4 + (int) yoffset) * 2,
                       5 * 2, image_width, image_height);
        cairo_draw_bar(cr, (46 + xoffset) * 2, 1 * 2, (4 + (int) yoffset) * 2,
                       5 * 2, image_width, image_height);
        cairo_draw_bar(cr, (48 + xoffset) * 2, 1 * 2, (4 + (int) yoffset) * 2,
                       5 * 2, image_width, image_height);
        cairo_draw_bar(cr, (50 + xoffset) * 2, 1 * 2, (4 + (int) yoffset) * 2,
                       5 * 2, image_width, image_height);

        textpart[0] = symbol->text[0];
        textpart[1] = '\0';
        textpos = 2 * (-5 + xoffset);
        cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                          image_width, image_height);
        for (i = 0; i < 6; i++) {
            textpart[i] = symbol->text[i + 1];
        }
        textpart[6] = '\0';
        textpos = 2 * (24 + xoffset);
        cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                          image_width, image_height);
        textpart[0] = symbol->text[7];
        textpart[1] = '\0';
        textpos = 2 * (55 + xoffset);
        cairo_draw_string(cr, textpart, textpos, default_text_posn, smalltext,
                          image_width, image_height);
        textdone = 1;
        switch (strlen(addon)) {
        case 2:
            textpos = 2 * (xoffset + 70);
            cairo_draw_string(cr, addon, textpos,
                              image_height - (addon_text_posn * 2) - 13, smalltext,
                              image_width, image_height);
            break;
        case 5:
            textpos = 2 * (xoffset + 84);
            cairo_draw_string(cr, addon, textpos,
                              image_height - (addon_text_posn * 2) - 13, smalltext,
                              image_width, image_height);
            break;
        }

    }

    xoffset -= comp_offset;

    /* Put boundary bars or box around symbol */
    if (((symbol->output_options & BARCODE_BOX) != 0)
            || ((symbol->output_options & BARCODE_BIND) != 0)) {
        /* boundary bars */
        cairo_draw_bar(cr, 0, (symbol->width + xoffset + xoffset) * 2,
                       textoffset * 2, symbol->border_width * 2, image_width,
                       image_height);
        cairo_draw_bar(cr, 0, (symbol->width + xoffset + xoffset) * 2,
                       (textoffset + symbol->height +
                        symbol->border_width) * 2, symbol->border_width * 2,
                       image_width, image_height);
        if ((symbol->output_options & BARCODE_BIND) != 0) {
            if ((symbol->rows > 1) && (is_stackable(symbol->symbology) == 1)) {
                /* row binding */
                for (r = 1; r < symbol->rows; r++) {
                    cairo_draw_bar(cr, xoffset * 2, symbol->width * 2,
                                   ((r * row_height) + textoffset + yoffset -
                                    1) * 2, 2 * 2, image_width, image_height);
                }
            }
        }
    }

    if ((symbol->output_options & BARCODE_BOX) != 0) {
        /* side bars */
        cairo_draw_bar(cr, 0, symbol->border_width * 2, textoffset * 2,
                       (symbol->height + (2 * symbol->border_width)) * 2,
                       image_width, image_height);
        cairo_draw_bar(cr,
                       (symbol->width + xoffset + xoffset -
                        symbol->border_width) * 2, symbol->border_width * 2,
                       textoffset * 2,
                       (symbol->height + (2 * symbol->border_width)) * 2,
                       image_width, image_height);
    }

    /* Put the human readable text at the bottom */
    if ((textdone == 0) && (local_text_length != 0)) {
        cairo_text_extents(cr, (char *)local_text, &extents);
        textpos = (image_width / 2) - (extents.width / 2);
        cairo_draw_string(cr, (char *) local_text, textpos, default_text_posn,
                          smalltext, image_width, image_height);
    }

    free(local_text);

    return error_number;
}

ZINT_EXTERN int
ZBarcode_Print_Cairo(struct zint_symbol *symbol, cairo_t *cr)
{
    static char const *SSET = "0123456789ABCDEF";
    int error_number = 0;
    const char *locale = NULL;

    locale = setlocale(LC_ALL, "C");

    do {

        /* sort out colour options */
        to_upper((unsigned char *)symbol->fgcolour);
        to_upper((unsigned char *)symbol->bgcolour);

        if (strlen(symbol->fgcolour) != 6) {
            strcpy(symbol->errtxt, "Malformed foreground colour target");
            error_number =  ZINT_ERROR_INVALID_OPTION;
            break;
        }

        if (strlen(symbol->bgcolour) != 6) {
            strcpy(symbol->errtxt, "Malformed background colour target");
            error_number =  ZINT_ERROR_INVALID_OPTION;
            break;
        }

        error_number = is_sane(SSET, (unsigned char *)symbol->fgcolour, strlen(symbol->fgcolour));
        if (error_number) {
            strcpy(symbol->errtxt, "Malformed foreground colour target");
            error_number =  ZINT_ERROR_INVALID_OPTION;
            break;
        }

        error_number = is_sane(SSET, (unsigned char *)symbol->bgcolour, strlen(symbol->bgcolour));
        if (error_number) {
            strcpy(symbol->errtxt, "Malformed background colour target");
            error_number =  ZINT_ERROR_INVALID_OPTION;
            break;
        }

        double red = 0.0;
        double green = 0.0;
        double blue = 0.0;

        cairo_save(cr);
        cairo_hex_to_colour(&red, &green, &blue, (*symbol).fgcolour);
        cairo_set_source_rgb(cr, red, green, blue);
        cairo_set_line_width(cr, 1.0);
        cairo_scale(cr, (*symbol).scale, (*symbol).scale);

        if (BARCODE_MAXICODE == symbol->symbology) {
            error_number = cairo_plot_maxi(symbol, cr);
        } else {
            error_number = cairo_plot_default(symbol, cr);
        }

        cairo_restore(cr);


    } while (0);

    if (locale) {
        setlocale(LC_ALL, locale);
    }

    return error_number;
}
