/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "run_dialog.h"

struct run_dialog                       g_run_dialog;

static int
run_dialog_bind(
  struct run_dialog*const               io_dlg,
  GtkBuilder*const                      i_builder)
{
  int                                   l_exit;
  struct run_dialog_status*             l_status;

  l_exit= 0;

  do
  {

    (*io_dlg).m_status_grid= GTK_GRID(
      gtk_builder_get_object(i_builder, "status_grid"));

    (*io_dlg).m_open_checkbutton= GTK_CHECK_BUTTON(
      gtk_builder_get_object(i_builder, "open_checkbutton"));

    (*io_dlg).m_cancel_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "cancel_button"));

    l_status= &(*io_dlg).m_status[0];

    (*l_status).m_progressbar= GTK_PROGRESS_BAR(
      gtk_builder_get_object(i_builder, "progressbar1"));

    (*l_status).m_tuple_label= GTK_LABEL(
      gtk_builder_get_object(i_builder, "tuple_label"));

    (*l_status).m_report_label= GTK_LABEL(
      gtk_builder_get_object(i_builder, "report_label"));

  }while(0);

  return l_exit;
}

extern void
run_dialog_deinitialize_instance()
{
  unsigned                              l_slot;

  g_mutex_lock(&g_run_dialog_mutex);

  for (l_slot= 0; g_run_dialog.m_status_slots > l_slot; l_slot++)
  {
    g_free(g_run_dialog.m_status[l_slot].m_basename);
    g_free(g_run_dialog.m_status[l_slot].m_reportfile);
  }

  if (g_run_dialog.m_dialog)
  {
    gtk_widget_destroy(GTK_WIDGET(g_run_dialog.m_dialog));
  }

  g_free(g_run_dialog.m_status);

  memset(&g_run_dialog, 0, sizeof(g_run_dialog));

  g_mutex_unlock(&g_run_dialog_mutex);

  return;
}

extern int
run_dialog_initialize_instance(
  char const*                           i_reportfile)
{
  GtkBuilder*                           l_builder;
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_rc;

  memset(&g_run_dialog, 0, sizeof(g_run_dialog));

  l_builder= 0;
  l_dialog= 0;
  l_error= 0;
  l_exit= 0;

  do
  {

    l_builder= gtk_builder_new();
    l_rc= wrap_gtk_builder_add_from_file(l_builder, "runreport.glade", &l_error);

    if (0 == l_rc)
    {
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_dialog= GTK_DIALOG(gtk_builder_get_object(l_builder, "run_dialog"));

    if (0 == l_dialog)
    {
      l_error= g_error_new(
        GENERAL,
        GENERIC,
        "Unable to find dialog object: 'run_dialog'");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    g_object_ref(l_dialog);
    gtk_window_set_modal(GTK_WINDOW(l_dialog), 1);

    g_run_dialog.m_dialog= l_dialog;
    g_run_dialog.m_status= g_new0(struct run_dialog_status, 1);
    g_run_dialog.m_status_slots= 1;
    g_run_dialog.m_status[0].m_basename= g_path_get_basename(i_reportfile);
    g_run_dialog.m_status[0].m_reportfile= g_strdup(i_reportfile);

    run_dialog_bind(&g_run_dialog, l_builder);

    gtk_window_set_transient_for(GTK_WINDOW(l_dialog), get_main_window());
    gtk_widget_show_all(GTK_WIDGET(l_dialog));
    gtk_widget_grab_focus(gtk_dialog_get_widget_for_response(
      GTK_DIALOG(l_dialog), GTK_RESPONSE_CANCEL));

    gtk_label_set_text(g_run_dialog.m_status[0].m_tuple_label, "0");
    gtk_label_set_text(g_run_dialog.m_status[0].m_report_label, g_run_dialog.m_status[0].m_basename);

  }while(0);

  if (l_builder)
  {
    g_object_unref(l_builder);
  }

  if (l_error)
  {
    _error_display_prompt(l_error);
  }

  g_clear_error(&l_error);

  return l_exit;
}

extern gboolean
run_dialog_end(
  gpointer                              i_data)
{

  g_mutex_lock(&g_run_dialog_mutex);

  if (g_run_dialog.m_dialog)
  {
    gtk_button_set_label(g_run_dialog.m_cancel_button, "Ok");
  }

  g_mutex_unlock(&g_run_dialog_mutex);

  return G_SOURCE_REMOVE;
}

extern int
run_dialog_lookup(
  char const*                           i_reportfile)
{
  int                                   l_exit;
  int                                   l_rc;
  unsigned                              l_slot;

  l_exit= -1;

  for (l_slot= 0; g_run_dialog.m_status_slots > l_slot; l_slot++)
  {
    l_rc= strcmp(i_reportfile, g_run_dialog.m_status[l_slot].m_reportfile);
    if (0 == l_rc)
    {
      l_exit= (int)l_slot;
      break;
    }
  }

  return l_exit;
}

static void
run_dialog_new_level(
  char const*                           i_reportfile)
{
  char*                                 l_basename;
  struct run_dialog_status*             l_status;

  g_run_dialog.m_status= (struct run_dialog_status*)
    realloc(
      g_run_dialog.m_status,
      (1+g_run_dialog.m_status_slots)*sizeof(struct run_dialog_status));

  g_run_dialog.m_status_active= g_run_dialog.m_status_slots;
  l_status= &g_run_dialog.m_status[g_run_dialog.m_status_slots];

  l_basename= g_path_get_basename(i_reportfile);
  (*l_status).m_progressbar= GTK_PROGRESS_BAR(gtk_progress_bar_new());
  (*l_status).m_tuple_label= GTK_LABEL(gtk_label_new("0"));
  (*l_status).m_report_label= GTK_LABEL(gtk_label_new(l_basename));
  (*l_status).m_tuple= 0;
  (*l_status).m_basename= l_basename;
  (*l_status).m_reportfile= g_strdup(i_reportfile);

  gtk_grid_insert_row(g_run_dialog.m_status_grid, g_run_dialog.m_status_slots);

  gtk_grid_attach(
    g_run_dialog.m_status_grid,
    GTK_WIDGET((*l_status).m_progressbar),
    0,
    g_run_dialog.m_status_slots,
    1,
    1);

  gtk_grid_attach(
    g_run_dialog.m_status_grid,
    GTK_WIDGET((*l_status).m_tuple_label),
    1,
    g_run_dialog.m_status_slots,
    1,
    1);

  gtk_grid_attach(
    g_run_dialog.m_status_grid,
    GTK_WIDGET((*l_status).m_report_label),
    2,
    g_run_dialog.m_status_slots,
    1,
    1);
    
  g_run_dialog.m_status_slots++;

  gtk_widget_show_all(GTK_WIDGET(g_run_dialog.m_status_grid));

  return;
}

extern void
run_dialog_switch(
  char const*                           i_reportfile)
{
  int                                   l_slot;

  g_mutex_lock(&g_run_dialog_mutex);

  do
  {

    l_slot= run_dialog_lookup(i_reportfile);

    if (0 <= l_slot)
    {
      g_run_dialog.m_status_active= (unsigned)l_slot;
      break;
    }

    run_dialog_new_level(i_reportfile);

  }while(0);

  g_mutex_unlock(&g_run_dialog_mutex);

  return;
}

extern gboolean
run_dialog_update(
  gpointer                              i_data)
{
  struct run_dialog_status*             l_status;
  char                                  l_text[1+31];
  unsigned                              l_slot;

  g_mutex_lock(&g_run_dialog_mutex);

  if (g_run_dialog.m_dialog)
  {
    l_slot= (unsigned)(ptrdiff_t)i_data;
    l_status= &g_run_dialog.m_status[l_slot];
    gtk_progress_bar_pulse((*l_status).m_progressbar);
    snprintf(l_text, sizeof(l_text), "%ld", (*l_status).m_tuple);
    gtk_label_set_text((*l_status).m_tuple_label, l_text);
  }

  g_mutex_unlock(&g_run_dialog_mutex);

  return G_SOURCE_REMOVE;
}
