/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "option.h"
#include "version.h"
#include "wallet.h"

extern void
option_assign(
  struct option*const                   o_option)
{

  memset(o_option, 0, sizeof(*o_option));

  (*o_option).m_prompt= g_hash_table_new_full(
    g_str_hash,
    g_str_equal,
    g_free,
    g_free);

  return;
}

extern void
option_discharge(
  struct option*const                   io_option)
{

  if ((*io_option).m_outputfile)
  {
    g_free((*io_option).m_outputfile);
  }

  if ((*io_option).m_reportfile)
  {
    g_free((*io_option).m_reportfile);
  }

  if ((*io_option).m_prompt)
  {
    g_hash_table_destroy((*io_option).m_prompt);
  }

  memset(io_option, 0, sizeof(*io_option));

  return;
}

static char*
option_password_parse(
  char const**                          o_next,
  char const*                           i_param)
{
  char*                                 l_dst;
  size_t                                l_size;
  char*                                 l_slice;
  char const*                           l_src;

  l_src= i_param;
  l_size= strlen(i_param);
  l_slice= (char*)g_malloc0(1+l_size);
  l_dst= l_slice;

  do
  {

    if (0 == l_src[0])
    {
      break;
    }

    if ('\\' == l_src[0])
    {
      l_src++;
    }
    else if (':' == l_src[0])
    {
      l_src++;
      break;
    }
  
    l_dst[0]= l_src[0];
    l_dst++;
    l_src++;

  }while(1);

  (*o_next)= l_src;

  return l_slice;
}

static int
option_prompt_add_hash(
  GHashTable*                           io_prompt,
  char *const*                          i_param)
{
  int                                   l_exit;
  char*                                 l_key;
  char const*                           l_next;
  size_t                                l_slot;
  char*                                 l_value;

  l_exit= 0;

  for(l_slot= 0; i_param[l_slot]; l_slot++)
  {

    if (0 == i_param[l_slot][0])
    {
      continue;
    }

    l_key= option_password_parse(&l_next, i_param[l_slot]);
    l_value= option_password_parse(&l_next, l_next);
    g_strstrip(l_key);
    g_strstrip(l_value);
    g_hash_table_insert(io_prompt, l_key, l_value);
  }

  return l_exit;
}

static int
option_password_add_hash(
  char *const*                          i_param)
{
  int                                   l_exit;
  struct password*                      l_pwd;
  size_t                                l_slot;
  char const*                           l_next;

  l_exit= 0;
  l_slot= 0;

  for (l_slot= 0; i_param[l_slot]; l_slot++)
  {

    if (0 == i_param[l_slot][0])
    {
      continue;
    }

    l_pwd= (struct password*)g_malloc0(sizeof(*l_pwd));
    (*l_pwd).m_password= option_password_parse(&l_next, i_param[l_slot]);
    (*l_pwd).m_username= option_password_parse(&l_next, l_next);
    (*l_pwd).m_aspect= option_password_parse(&l_next, l_next);
    wallet_insert(l_pwd);
  }

  return l_exit;
}

extern int
option_parse(
  struct option*const                   io_option,
  int                                   i_argc,
  char **                               i_argv)
{
  static char*                          l_format;
  static char*                          l_outputfile;
  static char**                         l_prompt;
  static char*                          l_reportfile;
  static char**                         l_password;
  static GOptionEntry                   l_entries[]=
  {
    {"output", 0, 0, G_OPTION_ARG_FILENAME, &l_outputfile, "output file name", "outputfile"},
    {"password", 0, 0, G_OPTION_ARG_STRING_ARRAY, &l_password, "password", "password[:username[:aspect]]"},
    {"prompt", 0, 0, G_OPTION_ARG_STRING_ARRAY, &l_prompt, "query prompt", "field:value"},
    {"report", 0, 0, G_OPTION_ARG_FILENAME, &l_reportfile, "report to run", "reportname"},
/*
    {"format", 0, 0, G_OPTION_ARG_STRING, &l_format, "output format (pdf)", "pdf"},
*/
    {0}
  };
  gboolean                              l_bool;
  GOptionContext*                       l_ctx;
  GError*                               l_error;
  int                                   l_exit;
  unsigned                              l_slot;
  char const*                           l_stage;
  char                                  l_summary[64];

  l_ctx= 0;
  l_error= 0;
  l_exit= 0;
  l_format= 0;
  l_outputfile= 0;
  l_password= 0;
  l_prompt= 0;
  l_reportfile= 0;
  l_stage= version_get_stage_text(g_version.m_stage);

  snprintf(
    l_summary,
    sizeof(l_summary),
    "iwrite - %d.%d.%d.%d (%s.%s.%s-%s)",
    g_version.m_major,
    g_version.m_minor,
    g_version.m_stage,
    g_version.m_micro,
    g_version.m_stamp.m_year,
    g_version.m_stamp.m_month,
    g_version.m_stamp.m_day,
    l_stage);

  do
  {

    l_ctx= g_option_context_new(0);
    g_option_context_add_main_entries(l_ctx, l_entries, 0);
    g_option_context_add_group(l_ctx, gtk_get_option_group(FALSE));
    g_option_context_set_summary(l_ctx, l_summary);
    l_bool= g_option_context_parse(l_ctx, &i_argc, &i_argv, &l_error);

    if (0 == l_bool)
    {
      _error_log(l_error);
      g_warning("Unable to parse command line: %s\n", l_error->message);
      l_exit= -1;
      break;
    }

    if (l_prompt)
    {
      l_exit|= option_prompt_add_hash((*io_option).m_prompt, l_prompt);
    }

    if (l_password)
    {
      l_exit|= option_password_add_hash(l_password);
    }

    if (0 == l_reportfile)
    {
      if (l_outputfile || l_format)
      {
        g_warning("Report file missing from command line\n");
        l_exit|= -1;
      }
    }

    (*io_option).m_reportfile= l_reportfile;
    l_reportfile= 0;

    if (l_outputfile)
    {
      (*io_option).m_outputfile= l_outputfile;
      l_outputfile= 0;
    }

  }while(0);

  if (l_ctx)
  {
    g_option_context_free(l_ctx);
  }

  g_clear_error(&l_error);

  if (l_prompt)
  {
    for(l_slot= 0; l_prompt[l_slot]; l_slot++)
    {
      g_free(l_prompt[l_slot]);
    }
  }

  if (l_password)
  {
    for(l_slot= 0; l_password[l_slot]; l_slot++)
    {
      g_free(l_password[l_slot]);
    }
  }

  g_free(l_format);
  g_free(l_outputfile);
  g_free(l_reportfile);

  return l_exit;
}
