/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "layout_pdf.h"
#include "layout_subreport.h"
#include "query_exec2.h"

static int
layout_subreport_check_recursive(
  GError**                              o_error,
  struct rw const*const                 i_rw,
  char const*                           i_reportfile)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_rc;
  struct rw const*                      l_rw;

  l_error= 0;
  l_exit= 0;
  l_rw= i_rw;

  do
  {

    if (0 == l_rw)
    {
      break;
    }

    l_rc= strcmp(i_reportfile, (*i_rw).m_reportfile);

    if (0 == l_rc)
    {
      l_error= g_error_new(
        LAYOUT,
        LAYOUT_SUBREPORT_RECURSIVE,
        "Recursive subreport now allowed: ('%s')",
        i_reportfile);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_rw= (*l_rw).m_prev;

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

static int
layout_subreport_load_spec(
  GError**                              o_error,
  struct layout_spec*const              o_spec,
  char const*                           i_reportfile)
{
  GError*                               l_error;
  int                                   l_exit;  
  enum report_style                     l_style;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;
  l_xml= 0;

  do
  {

    l_exit= report_util_open_report(&l_error, &l_style, &l_xml, i_reportfile);

    if (l_exit)
    {
      break;
    }

    if (style_layout != l_style)
    {
      l_error= g_error_new(
        LAYOUT,
        LAYOUT_SUBREPORT_NOT_LAYOUT,
        "Subreport is not 'layout' style: ('%s')",
        i_reportfile);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_exit= layout_spec_read(&l_error, o_spec, l_xml);

    if (l_exit)
    {
      break;
    }

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  xml_free(l_xml);

  return l_exit;
}

extern int
layout_subreport(
  struct rw*const                       io_rw,
  struct item_report const*const        i_object)
{
  int                                   l_eof;
  GError*                               l_error;
  int                                   l_exit;
  struct rw                             l_rw;
  struct layout_spec                    l_spec;

  l_exit= 0;
  l_error= 0;
  layout_spec_assign(&l_spec);
  memset(&l_rw, 0, sizeof(l_rw));
  query_exec_assign(&l_rw.m_exec);
  l_rw.m_surface= (*io_rw).m_surface;
  l_rw.m_cr= (*io_rw).m_cr;
  l_rw.m_pos_x= (*i_object).m_box.m_pos_x;
  l_rw.m_pos_y= (*io_rw).m_pos_y + (*i_object).m_box.m_pos_y;
  l_rw.m_page_number= 1;
  l_rw.m_time= (*io_rw).m_time;
  l_rw.m_interactive= (*io_rw).m_interactive;
  l_rw.m_reportfile= (*i_object).m_filename;
  l_rw.m_prev= io_rw;

  do
  {

    if (run_dialog_get_cancelled())
    {
      l_exit= -1;
      break;
    }

    if (0 == (*i_object).m_filename || 0 == (*i_object).m_filename[0])
    {
      break;
    }

    if (l_rw.m_interactive)
    {
      run_dialog_switch((*i_object).m_filename);
      g_run_dialog.m_status[g_run_dialog.m_status_active].m_tuple= 0;
    }

    l_exit= layout_subreport_check_recursive(&l_error, io_rw, (*i_object).m_filename);

    if (l_exit)
    {
      break;
    }

    l_exit= layout_subreport_load_spec(&l_error, &l_spec, (*i_object).m_filename);

    if (l_exit)
    {
      break;
    }
    
    layout_spec_paper_convert_to_points(&l_spec);

    l_exit= report_util_resolve_password(&l_spec.m_aspect, l_rw.m_interactive);

    if (l_exit)
    {
      break;
    }

    l_exit= query_exec2_prepare(&l_error, &l_rw.m_exec, &l_spec.m_query, &(*io_rw).m_exec);

    if (l_exit)
    {
      break;
    }

    l_exit= query_exec_connect(&l_error, &l_rw.m_exec, &l_spec.m_aspect);

    if (l_exit)
    {
      break;
    }

    l_exit= query_exec_run(&l_error, &l_rw.m_exec);

    if (l_exit)
    {
      break;
    }

    l_exit= query_exec_next(&l_error, &l_eof, &l_rw.m_exec);

    if (l_exit)
    {
      break;
    }
    
    l_exit= layout_pdf_main(&l_error, &l_rw, &l_spec, l_eof);

    if (l_exit)
    {
      break;
    }

    (*io_rw).m_pos_y= l_rw.m_pos_y;

  }while(0);

  if (l_rw.m_interactive)
  {
    run_dialog_switch((*io_rw).m_reportfile);
    if (l_error)
    {
      _error_display_prompt(l_error);
    }
  }

  g_clear_error(&l_error);

  layout_spec_discharge(&l_spec);
  query_exec_discharge(&l_rw.m_exec);

  if (l_exit)
  {
    g_run_dialog.m_cancel= 1;
  }

  return l_exit;
}
