/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "layout_pdf.h"
#include "report_util.h"

extern int
layout_pdf_main(
  GError**                              o_error,
  struct rw*const                       io_rw,
  struct layout_spec *const             io_layout,
  int const                             i_eof)
{
  GError*                               l_error;
  int                                   l_exit;
  unsigned                              l_slot;

  l_error= 0;
  l_exit= 0;

  do
  {

    if ((*io_layout).m_cover.m_header.m_item)
    {
      rw_item_set_value((*io_layout).m_cover.m_header.m_item, io_rw);
      cairo_save((*io_rw).m_cr);
      item_draw((*io_layout).m_cover.m_header.m_item, (*io_rw).m_cr, 1.0, 0);
      cairo_restore((*io_rw).m_cr);
      cairo_show_page((*io_rw).m_cr);
    }

    if (0 == i_eof)
    {
      rw_layout_clear_accumulator(io_layout);
      rw_layout_set_fence(io_rw, io_layout);
      rw_level_break_set_value(io_layout, io_rw);
      rw_item_set_value((*io_layout).m_report.m_header.m_item, io_rw);

      for (l_slot= 0; (*io_layout).m_group_slots > l_slot; l_slot++)
      {
        rw_item_set_value((*io_layout).m_group[l_slot].m_header.m_item, io_rw);
      }

      layout_util_draw_headers(io_layout, io_rw);
      l_exit= layout_loop(&l_error, io_rw, io_layout);
    }

    if ((0 == l_exit) && (*io_layout).m_cover.m_footer.m_item)
    {
      rw_item_set_value((*io_layout).m_cover.m_footer.m_item, io_rw);
      cairo_save((*io_rw).m_cr);
      item_draw((*io_layout).m_cover.m_footer.m_item, (*io_rw).m_cr, 1.0, 0);
      cairo_restore((*io_rw).m_cr);
      cairo_show_page((*io_rw).m_cr);
    }

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

static int
pdf_run_exec(
  struct layout_spec *const             io_layout,
  struct goop const*const               i_goop)
{ 
  int                                   l_eof;
  GError*                               l_error;
  int                                   l_exit;
  struct rw                             l_rw;

  l_error= 0;
  l_exit= 0;
  rw_assign(&l_rw);
  l_rw.m_interactive= (*i_goop).m_interactive;
  l_rw.m_reportfile= (*i_goop).m_reportfile;

  do
  {

    l_exit= rw_cairo_pdf_surface_create(
      &l_error,
      &l_rw,
      (*io_layout).m_paper.m_width,
      (*io_layout).m_paper.m_height,
      (*i_goop).m_outputfile);

    if (l_exit)
    {
      break;
    }

    l_exit= report_util_resolve_password(&(*io_layout).m_aspect, (*i_goop).m_interactive);

    if (l_exit)
    {
      break;
    }

    l_exit= query_exec_connect_prepare_first(
      &l_error,
      &l_eof,
      &l_rw.m_exec,
      &(*io_layout).m_query,
      &(*io_layout).m_aspect,
      (*i_goop).m_prompt,
      (*i_goop).m_interactive);

    if (l_exit)
    {
      break;
    }

    l_exit= layout_pdf_main(&l_error, &l_rw, io_layout, l_eof);

  }while(0);

  if (l_error)
  {
    if ((*i_goop).m_interactive)
    {
      _error_display_prompt(l_error);
    }
  }

  g_clear_error(&l_error);

  rw_discharge(&l_rw);

  return l_exit;
}

struct pdf_thread_data
{
  struct layout_spec*                   m_layout;
  struct goop const*                    m_goop;
};

static gpointer
pdf_run_thread(
  gpointer                              i_data)
{
  int                                   l_exit;
  struct pdf_thread_data*               l_info;

  l_info= (struct pdf_thread_data*)i_data;

  l_exit= pdf_run_exec((*l_info).m_layout, (*l_info).m_goop);

  if (0 == run_dialog_get_cancelled())
  {
    gdk_threads_add_idle(run_dialog_end, 0);
  }

  g_thread_exit((gpointer)(ptrdiff_t)l_exit);

  return (gpointer)(ptrdiff_t)l_exit;
}

extern int
layout_pdf_run(
  struct layout_spec *const             io_layout,
  struct goop const*const               i_goop)
{
  int                                   l_exit;
  struct pdf_thread_data                l_info;
  int                                   l_rc;
  GThread*                              l_thread;

  memset(&g_run_dialog, 0, sizeof(g_run_dialog));
  l_exit= 0;
  l_thread= 0;
  memset(&l_info, 0, sizeof(l_info));

  do
  {

    if (0 == (*i_goop).m_interactive)
    {
      l_exit= pdf_run_exec(io_layout, i_goop);
      break;
    }

    l_info.m_layout= io_layout;
    l_info.m_goop= i_goop;

    l_exit= run_dialog_initialize_instance((*i_goop).m_reportfile);

    if (l_exit)
    {
      break;
    }

    l_thread= g_thread_new("pdf", pdf_run_thread, &l_info);

    gtk_dialog_run(g_run_dialog.m_dialog);
    g_run_dialog.m_cancel= 1;

    l_exit= (int)(ptrdiff_t)g_thread_join(l_thread);

    if (l_exit)
    {
      break;
    }

    l_rc= gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON(g_run_dialog.m_open_checkbutton));

    if (0 == l_rc)
    {
      l_exit= 1;
      break;
    }

  }while(0);

  run_dialog_deinitialize_instance();

  if (l_thread)
  {
    g_thread_unref(l_thread);
  }

  return l_exit;
}
