/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "layout.h"

struct break_dialog
{
  GtkEntry*                             m_page_entry;
  GtkEntry*                             m_group_entry;
  GtkGrid*                              m_group_grid;
  GtkTreeView*                          m_tree_view;
  GtkTreeModel*                         m_tree_model;
};

struct break_group
{
  int                                   m_has_ref; 
  struct layout_division_group          m_division;
};

static void
break_property_group_add(
  struct break_dialog*const             io_bd,
  char const*                           i_text)
{
  GtkListStore*                         l_store;
  GtkTreeIter                           l_iter;
  int                                   l_rc;
  gchar*                                l_value;
  int                                   l_found;

  l_found= 0;

  do
  {

    l_rc= gtk_tree_model_get_iter_first((*io_bd).m_tree_model, &l_iter);

    do
    {
  
      if (0 == l_rc)
      {
        break;
      }

      gtk_tree_model_get((*io_bd).m_tree_model, &l_iter, 0, &l_value, -1);
      l_rc= strcmp(l_value, i_text);
      g_free(l_value);

      if (0 == l_rc)
      {
        l_found= 1;
        break;
      }

      l_rc= gtk_tree_model_iter_next((*io_bd).m_tree_model, &l_iter);

    }while(1);

  }while(0);

  if (0 == l_found)
  {
    l_store= GTK_LIST_STORE((*io_bd).m_tree_model);
    gtk_list_store_append (l_store, &l_iter);
    gtk_list_store_set( l_store, &l_iter, 0, i_text, -1);
  }

  return;
}

static void
break_property_addbutton_cb(
  GtkWidget *                           i_widget,
  gpointer                              i_data)
{
  struct break_dialog*                  l_dlg;
  gchar const*                          l_text;

  l_dlg= (struct break_dialog*)i_data;
  l_text= gtk_entry_get_text((*l_dlg).m_group_entry);

  if ((l_text) && l_text[0])
  {
    break_property_group_add(l_dlg, l_text);
  }

  return;
}

static void
break_property_removebutton_cb(
  GtkWidget *                           i_widget,
  gpointer                              i_user_data)
{
  struct break_dialog*                  l_dlg;
  GtkTreeIter                           l_iter;
  int                                   l_rc;
  GtkTreeSelection*                     l_selection;
  GtkListStore*                         l_store;

  l_dlg= (struct break_dialog*)i_user_data;

  do
  {

    l_selection= gtk_tree_view_get_selection((*l_dlg).m_tree_view);

    l_rc= gtk_tree_selection_get_selected(l_selection, 0, &l_iter);

    if (0 == l_rc)
    {
      break;
    }

    l_store= GTK_LIST_STORE((*l_dlg).m_tree_model);
    gtk_list_store_remove(l_store, &l_iter);

  }while(0);

  return;
}

static int
break_property_bind(
  struct break_dialog*const             io_bd,
  GtkBuilder*const                      i_builder)
{
  GtkButton*                            l_button;
  int                                   l_exit;

  l_exit= 0;

  do
  {

    (*io_bd).m_page_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "page_entry"));

    (*io_bd).m_group_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "group_entry"));

    l_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "group_add_button"));

    g_signal_connect(
      GTK_WIDGET(l_button),
      "clicked",
      G_CALLBACK(break_property_addbutton_cb),
      io_bd);

    l_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "group_remove_button"));

    g_signal_connect(
      GTK_WIDGET(l_button),
      "clicked",
      G_CALLBACK(break_property_removebutton_cb),
      io_bd);

    (*io_bd).m_group_grid= GTK_GRID(
      gtk_builder_get_object(i_builder, "group_grid"));

  }while(0);

  return l_exit;
}

static void
break_property_set_values(
  struct break_dialog*const             io_bd,
  struct layout const*const             i_layout)
{
  unsigned                              l_slot;

  if ((*i_layout).m_page_break.m_column)
  {
    gtk_entry_set_text((*io_bd).m_page_entry, (*i_layout).m_page_break.m_column);
  }

  for(l_slot= 0; (*i_layout).m_group_slots > l_slot ; l_slot++)
  {
    break_property_group_add(io_bd, (*i_layout).m_group[l_slot].m_break.m_column);
  }

  return;
}

static gboolean
break_group_lookup(
  unsigned*const                        o_index,
  struct layout*const                   io_layout,
  gchar const*                          i_tag)
{
  gboolean                              l_found;
  gint                                  l_slot;

  (*o_index)= 0;
  l_found= 0;

  for (l_slot=0; (*io_layout).m_group_slots > l_slot; l_slot++, (*o_index)++)
  {

    l_found= (0 == strcmp((*io_layout).m_group[l_slot].m_break.m_column, i_tag));

    if (l_found)
    {
      break;
    }

  }

  return l_found;
}

static gboolean
break_list_lookup(
  gchar *const*                         i_list,
  unsigned const                        i_list_slots,
  gchar const*                          i_tag)
{
  gboolean                              l_found;
  gint                                  l_slot;

  l_found= 0;

  for (l_slot=0; i_list_slots > l_slot; l_slot++)
  {

    l_found= (0 == strcmp(i_list[l_slot], i_tag));

    if (l_found)
    {
      break;
    }

  }

  return l_found;
}

static void
break_groups_remove(
  struct layout*const                   io_layout,
  gchar *const*                         i_list,
  unsigned const                        i_list_slots)
{
  gint                                  l_bottom_row;
  gboolean                              l_found;
  gint                                  l_group_slot;
  struct layout_division_group*         l_head;
  gint                                  l_offset;
  gint                                  l_row;
  gint                                  l_slot;
  struct layout_division_group*         l_tail;
  gint                                  l_top_row;

  l_row= 0;

  if ((*io_layout).m_cover.m_header.m_page)
  {
    l_row++;
    l_row++;
  }

  if ((*io_layout).m_report.m_header.m_page)
  {
    l_row++;
    l_row++;
  }

  l_group_slot= 0;

  do
  {

    if ((*io_layout).m_group_slots <= l_group_slot)
    {
      break;
    }

    l_found= break_list_lookup(
      i_list,
      i_list_slots,
      (*io_layout).m_group[l_group_slot].m_break.m_column);

    if (l_found)
    {
      l_group_slot++;
      continue;
    }

    if ((*io_layout).m_report.m_detail.m_page)
    {
      l_offset= 2;
    }
    else
    {
      l_offset= 0;
    }

    l_bottom_row= ((*io_layout).m_group_slots * 2);
    for (l_slot= l_group_slot+1; (*io_layout).m_group_slots > l_slot; l_slot++)
    {
      l_bottom_row+= 2;
    }

    l_bottom_row+= l_row + l_offset;

    gtk_grid_remove_row((*io_layout).m_grid, l_bottom_row);
    gtk_grid_remove_row((*io_layout).m_grid, l_bottom_row);


    l_top_row= (2*l_group_slot);
    l_top_row+= l_row;

    gtk_grid_remove_row((*io_layout).m_grid, l_top_row);
    gtk_grid_remove_row((*io_layout).m_grid, l_top_row);

    if (1 < (*io_layout).m_group_slots)
    {
      l_head= &(*io_layout).m_group[l_group_slot];
      l_tail= &(*io_layout).m_group[1+l_group_slot];

      for (l_slot= l_group_slot; (*io_layout).m_group_slots > l_slot; l_slot++)
      {
        (*l_head)= (*l_tail);
        l_head++;
        l_tail++;
      }

      (*io_layout).m_group= realloc(
        (*io_layout).m_group,
        (*io_layout).m_group_slots * sizeof(struct layout_division_group));
    }
    else
    {
      g_free((*io_layout).m_group);
      (*io_layout).m_group= 0;
    }

    (*io_layout).m_group_slots--;

  }while(1);

  return;
}

struct break_group*
break_group_new(
  struct layout*const                   io_layout,
  gchar *const*                         i_list,
  unsigned const                        i_list_slots)
{
  struct break_group*                   l_break;
  gboolean                              l_found;
  struct layout_division_group*         l_group;
  unsigned                              l_index;
  struct paper                          l_paper;
  unsigned                              l_slot;

  l_break= (struct break_group*)g_malloc0(i_list_slots * sizeof(*l_break));

  l_paper= (*io_layout).m_paper;
  l_paper.m_height= POINTS_PER_INCH * 1.0;

  for (l_slot= 0; i_list_slots > l_slot; l_slot++)
  {

    l_found= break_group_lookup(&l_index, io_layout, i_list[l_slot]);
    l_group= &l_break[l_slot].m_division;
    
    if (l_found)
    {
      l_break[l_slot].m_has_ref= 1;
      (*l_group)= (*io_layout).m_group[l_index];
      g_object_ref((*l_group).m_header.m_page);
      g_object_ref((*l_group).m_footer.m_page);
      continue;
    }

    g_free((*l_group).m_break.m_column);
    (*l_group).m_break.m_column= g_strdup(i_list[l_slot]);

    g_free((*l_group).m_header.m_label);
    (*l_group).m_header.m_label= g_strdup_printf("%s.Header", i_list[l_slot]);

    g_free((*l_group).m_footer.m_label);
    (*l_group).m_footer.m_label= g_strdup_printf("%s.Footer", i_list[l_slot]);

    (*l_group).m_header.m_page= IWR_PAGE(iwr_page_new());
    iwr_page_set_section_type((*l_group).m_header.m_page, section_group_header);
    iwr_page_set_paper((*l_group).m_header.m_page, &l_paper);

    (*l_group).m_footer.m_page= IWR_PAGE(iwr_page_new());
    iwr_page_set_section_type((*l_group).m_footer.m_page, section_group_footer);
    iwr_page_set_paper((*l_group).m_footer.m_page, &l_paper);
  }

  return l_break;
}

static void
break_remove_all_groups(
  struct layout*const                   io_layout)
{
  unsigned                              l_slot;
  unsigned                              l_row;

  l_row= 0;

  if ((*io_layout).m_cover.m_header.m_page)
  {
    l_row++;
    l_row++;
  }

  if ((*io_layout).m_report.m_header.m_page)
  {
    l_row++;
    l_row++;
  }

  if ((*io_layout).m_report.m_detail.m_page)
  {
    l_row++;
    l_row++;
  }

  l_row+= 2 * (*io_layout).m_group_slots;

  for (l_slot= 0; (*io_layout).m_group_slots > l_slot; l_slot++)
  {
    gtk_grid_remove_row((*io_layout).m_grid, l_row);
    gtk_grid_remove_row((*io_layout).m_grid, l_row);
  }

  l_row= 0;

  if ((*io_layout).m_cover.m_header.m_page)
  {
    l_row++;
    l_row++;
  }

  if ((*io_layout).m_report.m_header.m_page)
  {
    l_row++;
    l_row++;
  }

  for (l_slot= 0; (*io_layout).m_group_slots > l_slot; l_slot++)
  {
    gtk_grid_remove_row((*io_layout).m_grid, l_row);
    gtk_grid_remove_row((*io_layout).m_grid, l_row);
  }

  g_free((*io_layout).m_group);
  (*io_layout).m_group_slots= 0;

  return;
}

static void
break_groups_add(
  struct layout*const                   io_layout,
  gchar *const*                         i_list,
  unsigned const                        i_list_slots)
{
  struct break_group*                   l_break;
  unsigned                              l_slot;
  unsigned                              l_row;
  GtkWidget*                            l_widget;

  l_break= break_group_new(io_layout, i_list, i_list_slots);
  break_remove_all_groups(io_layout);

  l_row= 0;

  if ((*io_layout).m_cover.m_header.m_page)
  {
    l_row++;
    l_row++;
  }

  if ((*io_layout).m_report.m_header.m_page)
  {
    l_row++;
    l_row++;
  }

  if ((*io_layout).m_report.m_detail.m_page)
  {
    l_row++;
    l_row++;
  }

  l_slot= i_list_slots;

  do
  {

    if (0 == l_slot)
    {
      break;
    }

    l_slot--;

    l_widget= gtk_button_new_with_label(l_break[l_slot].m_division.m_footer.m_label);
    gtk_button_set_relief(GTK_BUTTON(l_widget), GTK_RELIEF_NONE);
    gtk_widget_set_sensitive(l_widget, 0);
    gtk_grid_insert_row((*io_layout).m_grid, l_row);
    gtk_grid_attach((*io_layout).m_grid, l_widget, 0, l_row++, 1, 1);
    gtk_grid_insert_row((*io_layout).m_grid, l_row);
    gtk_grid_attach((*io_layout).m_grid, GTK_WIDGET(l_break[l_slot].m_division.m_footer.m_page), 0, l_row++, 1, 1);

  }while(1);

  l_row= 0;

  if ((*io_layout).m_cover.m_header.m_page)
  {
    l_row++;
    l_row++;
  }

  if ((*io_layout).m_report.m_header.m_page)
  {
    l_row++;
    l_row++;
  }

  for (l_slot= 0; i_list_slots > l_slot; l_slot++)
  {
    l_widget= gtk_button_new_with_label(l_break[l_slot].m_division.m_header.m_label);
    gtk_button_set_relief(GTK_BUTTON(l_widget), GTK_RELIEF_NONE);
    gtk_widget_set_sensitive(l_widget, 0);
    gtk_grid_insert_row((*io_layout).m_grid, l_row);
    gtk_grid_attach((*io_layout).m_grid, l_widget, 0, l_row++, 1, 1);
    gtk_grid_insert_row((*io_layout).m_grid, l_row);
    gtk_grid_attach((*io_layout).m_grid, GTK_WIDGET(l_break[l_slot].m_division.m_header.m_page), 0, l_row++, 1, 1);
  }

  (*io_layout).m_group= (struct layout_division_group*)g_malloc0(i_list_slots * sizeof(struct layout_division_group));
  (*io_layout).m_group_slots= i_list_slots;

  for (l_slot= 0; i_list_slots > l_slot; l_slot++)
  {
    if (l_break[l_slot].m_has_ref)
    {
      g_object_unref(l_break[l_slot].m_division.m_header.m_page);
      g_object_unref(l_break[l_slot].m_division.m_footer.m_page);
    }
    (*io_layout).m_group[l_slot]= l_break[l_slot].m_division;
  }

  g_free(l_break);

  layout_set_layout_size(io_layout);

  return;
}

static void
break_property_copy_values(
  struct layout*const                   io_layout,
  struct break_dialog const*const       i_bd)
{
  GtkTreeIter                           l_iter;
  gchar**                               l_list;
  unsigned                              l_list_slot;
  unsigned                              l_list_slots;
  int                                   l_rc;
  gchar*                                l_text;
  gchar const*                          l_textc;

  l_list= 0;
  l_list_slots= 0;
  
  l_textc= gtk_entry_get_text((*i_bd).m_page_entry);

  g_free((*io_layout).m_page_break.m_column);
  (*io_layout).m_page_break.m_column= 0;

  if (l_textc && l_textc[0])
  {
    (*io_layout).m_page_break.m_column= g_strdup(l_textc);
  }

  do
  {

    l_rc= gtk_tree_model_get_iter_first((*i_bd).m_tree_model, &l_iter);

    do
    {
  
      if (0 == l_rc)
      {
        break;
      }

      gtk_tree_model_get((*i_bd).m_tree_model, &l_iter, 0, &l_text, -1);

      l_list= realloc(l_list, sizeof(*l_list)*(1+l_list_slots));
      l_list[l_list_slots]= l_text;
      l_list_slots++;

      l_rc= gtk_tree_model_iter_next((*i_bd).m_tree_model, &l_iter);

    }while(1);

    break_groups_remove(io_layout, l_list, l_list_slots);

    if (l_list_slots)
    {
      break_groups_add(io_layout, l_list, l_list_slots);
    }

  }while(0);

  for (l_list_slot=0; l_list_slots > l_list_slot; l_list_slot++)
  {
    g_free(l_list[l_list_slot]);
  }

  if (l_list)
  {
    g_free(l_list);
  }

  gtk_widget_show_all(GTK_WIDGET((*io_layout).m_grid));
  layout_set_layout_size(io_layout);

  return;
}

static void
break_property_create_model_view(
  struct break_dialog *                 io_bd)
{
  GtkCellRenderer*                      l_renderer;
  GtkListStore*                         l_store;

  (*io_bd).m_tree_view= GTK_TREE_VIEW(gtk_tree_view_new());
  gtk_widget_set_hexpand(GTK_WIDGET((*io_bd).m_tree_view), 1);
  gtk_widget_set_vexpand(GTK_WIDGET((*io_bd).m_tree_view), 1);
  gtk_tree_view_set_headers_visible((*io_bd).m_tree_view, 0);
  gtk_tree_view_set_reorderable((*io_bd).m_tree_view, 1);

  l_renderer= gtk_cell_renderer_text_new();

  gtk_tree_view_insert_column_with_attributes(
    (*io_bd).m_tree_view,
    -1,
    "Group",
    l_renderer,
    "text", 
    0,
    NULL);

  l_store= gtk_list_store_new(1, G_TYPE_STRING);

  (*io_bd).m_tree_model= GTK_TREE_MODEL(l_store);

  gtk_tree_view_set_model((*io_bd).m_tree_view, (*io_bd).m_tree_model);
  g_object_unref(l_store);

  gtk_grid_attach(
    (*io_bd).m_group_grid, 
    GTK_WIDGET((*io_bd).m_tree_view),
    0,
    1,
    1,
    1);

  return;
}

extern int
layout_break_property(
  struct layout*const                   io_layout)
{
  struct break_dialog                   l_dlg;
  GtkBuilder*                           l_builder;
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_rc;

  l_builder= 0;
  l_dialog= 0;
  l_error= 0;
  l_exit= 0;
  memset(&l_dlg, 0, sizeof(l_dlg));

  do
  {

    l_builder= gtk_builder_new();

    l_rc= wrap_gtk_builder_add_from_file(l_builder, "break.glade", &l_error);

    if (0 == l_rc)
    {
      l_exit= -1;
      break;
    }

    l_dialog= (GtkDialog*)gtk_builder_get_object(l_builder, "break_dialog");

    if (0 == l_dialog)
    {
      l_error= g_error_new(
        GENERAL,
        GENERIC,
        "Unable to find dialog object: 'break_dialog'");
      l_exit= -1;
      break;
    }

    break_property_bind(&l_dlg, l_builder);
    break_property_create_model_view(&l_dlg);
    break_property_set_values(&l_dlg, io_layout);

    gtk_window_set_transient_for(GTK_WINDOW(l_dialog), get_main_window());
    gtk_widget_show_all(GTK_WIDGET(l_dialog));
    gtk_window_set_modal(GTK_WINDOW(l_dialog), 1);

    gtk_widget_grab_focus(gtk_dialog_get_widget_for_response(
      GTK_DIALOG(l_dialog), GTK_RESPONSE_OK));

    l_exit= gtk_dialog_run(l_dialog);

    if (GTK_RESPONSE_OK != l_exit) 
    {
      break;
    }

    break_property_copy_values(io_layout, &l_dlg);

  }while(0);

  if (l_builder)
  {
    g_object_unref(l_builder);
  }

  if (l_error)
  {      
    _error_log(l_error);
    _error_display_prompt(l_error);
  }

  g_clear_error(&l_error);

  if (l_dialog)
  {
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  return l_exit;
}
