/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "iwsql_dll.h"

extern void
iwsql_dll_assign(
  struct iwsql_dll*const                o_driver)
{

  memset(o_driver, 0, sizeof(*o_driver));

  return;
}

extern int
iwsql_dll_connect(
  GError**                              o_error,
  struct iwsql_dll const*const          i_driver,
  struct iwsql_aspect const*const       i_aspect)
{
  GError*                               l_error;
  char*                                 l_msg;
  int                                   l_exit;

  l_error= 0;
  l_msg= 0;

  l_exit= (*(*i_driver).m_connect)(&l_msg, i_aspect, (*i_driver).m_sql_handle);

  if (l_exit)
  {
    l_error= g_error_new(
      IWSQL_DLL,
      IWSQL_DLL_CONNECT,
      "Unable to connect to database: error='%s'",
      l_msg);
    _error_log(l_error);
    g_propagate_error(o_error, l_error);
  }

  g_free(l_msg);

  return l_exit;
}

extern void
iwsql_dll_discharge(
  struct iwsql_dll*const                io_driver)
{

    if ((*io_driver).m_delete && (*io_driver).m_sql_handle)
    {
      (*(*io_driver).m_delete)(&(*io_driver).m_sql_handle);
    }

    if ((*io_driver).m_dll_handle)
    {
      g_module_close((*io_driver).m_dll_handle);
    }

    memset(io_driver, 0, sizeof(*io_driver));

  return;
}

extern int
iwsql_dll_exec(
  GError**                              o_error,
  struct iwsql_dll const*const          i_driver,
  char const*                           i_sql)
{
  GError*                               l_error;
  int                                   l_exit;
  char*                                 l_msg;

  l_error= 0;
  l_msg= 0;

  if (i_sql && i_sql[0])
  {
    l_exit= (*(*i_driver).m_exec)(&l_msg, i_sql, (*i_driver).m_sql_handle);

    if (l_exit)
    {
      l_error= g_error_new(
        IWSQL_DLL,
        IWSQL_DLL_EXEC,
        "Unable to prepare query: error='%s'",
        l_msg);
      _error_log(l_error);
    }

    g_free(l_msg);
  }
  else
  {
    l_error= g_error_new(IWSQL_DLL, IWSQL_DLL_EMPTY_QUERY, "Empty query");
    _error_log(l_error);
    l_exit= -1;
  }

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
iwsql_dll_load(
  GError**                              o_error,
  struct iwsql_dll*const                o_driver,
  struct driver_module const*const      i_module)
{
  GError*                               l_error;
  int                                   l_exit;
  GModule*                              l_handle;
  gboolean                              l_rc;

  l_error= 0;
  l_exit= 0;
  (*o_error)= 0;

  do
  {

    l_handle= g_module_open((*i_module).m_dll, G_MODULE_BIND_LAZY);

    if (0 == l_handle)
    {
      l_error= g_error_new(
        IWSQL_DLL,
        IWSQL_DLL_OPEN,
        "Unable to open module ('%s'): error='%s'",
        (*i_module).m_dll,
        g_module_error());
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    (*o_driver).m_dll_handle= l_handle;

    l_rc= g_module_symbol(
      l_handle,
      IWSQL_METHOD_CONNECT,
      (gpointer)&(*o_driver).m_connect);

    if (0 == l_rc)
    {
      l_error= g_error_new(
        IWSQL_DLL,
        IWSQL_DLL_LOAD,
        "Missing module function ('%s'['%s']): %s",
        (*i_module).m_dll,
        IWSQL_METHOD_CONNECT,
        g_module_error()); 
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_rc= g_module_symbol(
      l_handle,
      IWSQL_METHOD_DELETE,
      (gpointer)&(*o_driver).m_delete);

    if (0 == l_rc)
    {
      l_error= g_error_new(
        IWSQL_DLL,
        IWSQL_DLL_LOAD,
        "Missing module function ('%s'['%s']): %s",
        (*i_module).m_dll,
        IWSQL_METHOD_DELETE,
        g_module_error()); 
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_rc= g_module_symbol(
      l_handle,
      IWSQL_METHOD_EXEC,
      (gpointer)&(*o_driver).m_exec);

    if (0 == l_rc)
    {
      l_error= g_error_new(
        IWSQL_DLL,
        IWSQL_DLL_LOAD,
        "Missing module function ('%s'['%s']): %s",
        (*i_module).m_dll,
        IWSQL_METHOD_DELETE,
        g_module_error()); 
      _error_log(l_error);
      l_exit= -1;
      break;
    }
    
    l_rc= g_module_symbol(
      l_handle,
      IWSQL_METHOD_NEW,
      (gpointer)&(*o_driver).m_new);

    if (0 == l_rc)
    {
      l_error= g_error_new(
        IWSQL_DLL,
        IWSQL_DLL_LOAD,
        "Missing module function ('%s'['%s']): %s",
        (*i_module).m_dll,
        IWSQL_METHOD_NEW,
        g_module_error()); 
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_rc= g_module_symbol(
      l_handle,
      IWSQL_METHOD_NEXT,
      (gpointer)&(*o_driver).m_next);

    if (0 == l_rc)
    {
      l_error= g_error_new(
        IWSQL_DLL,
        IWSQL_DLL_LOAD,
        "Missing module function ('%s'['%s']): %s",
        (*i_module).m_dll,
        IWSQL_METHOD_NEXT,
        g_module_error()); 
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_rc= g_module_symbol(
      l_handle,
      IWSQL_METHOD_FETCH_RESULT,
      (gpointer)&(*o_driver).m_result);

    if (0 == l_rc)
    {
      l_error= g_error_new(
        IWSQL_DLL,
        IWSQL_DLL_LOAD,
        "Missing module function ('%s'['%s']): %s",
        (*i_module).m_dll,
        IWSQL_METHOD_FETCH_RESULT,
        g_module_error()); 
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_rc= g_module_symbol(
      l_handle,
      IWSQL_METHOD_SHOW_DATABASES,
      (gpointer)&(*o_driver).m_show_databases);

    if (0 == l_rc)
    {
      l_error= g_error_new(
        IWSQL_DLL,
        IWSQL_DLL_LOAD,
        "Missing module function ('%s'['%s']): %s",
        (*i_module).m_dll,
        IWSQL_METHOD_SHOW_DATABASES,
        g_module_error()); 
      _error_log(l_error);
      l_exit= -1;
      break;
    }

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
iwsql_dll_new(
  struct iwsql_dll *const               io_driver)
{

  (*(*io_driver).m_new)(&(*io_driver).m_sql_handle);

  return;
}

extern int
iwsql_dll_next(
  GError**                              o_error,
  int*const                             o_eof,
  struct iwsql_dll *const               io_driver)
{
  GError*                               l_error;
  int                                   l_exit;
  char*                                 l_msg;

  (*o_eof)= 0;
  l_error= 0;
  l_exit= 0;
  l_msg= 0;

  l_exit= (*(*io_driver).m_next)(&l_msg, o_eof, (*io_driver).m_sql_handle);

  if (l_exit)
  {
    l_error= g_error_new(
      IWSQL_DLL,
      IWSQL_DLL_FETCH,
      "Unable to fetch row: error='%s'",
      l_msg);
    _error_log(l_error);
    g_propagate_error(o_error, l_error);
  }

  g_free(l_msg);

  return l_exit;
}

extern void
iwsql_dll_result(
  struct iwsql_attr const**             o_attr,
  unsigned*const                        o_slots,
  struct iwsql_dll const*const          i_driver)
{

  (*o_attr)= 0;
  (*o_slots)= 0;
  (*(*i_driver).m_result)(o_attr, o_slots, (*i_driver).m_sql_handle);

  return;
}

extern int
iwsql_dll_show_databases(
  GError**                              o_error,
  char***                               o_list,
  unsigned*const                        o_slots,
  struct iwsql_dll *const               io_driver)
{
  GError*                               l_error;
  int                                   l_exit;
  char*                                 l_msg;

  l_error= 0;
  l_exit= 0;
  l_msg= 0;

  l_exit= (*(*io_driver).m_show_databases)(
    &l_msg, o_list, o_slots, (*io_driver).m_sql_handle);

  if (l_exit)
  {
    l_error= g_error_new(
      IWSQL_DLL,
      IWSQL_DLL_FETCH,
      "Unable to fetch databases: error='%s'",
      l_msg);
    _error_log(l_error);
    g_propagate_error(o_error, l_error);
  }

  g_free(l_msg);

  return l_exit;
}
