/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_page_number.h"

extern void
item_page_number_assign(
  struct item_page_number*const         o_page_number,
  struct item_method*const              o_method)
{

  memset(o_page_number, 0, sizeof(*o_page_number));

  (*o_page_number).m_box.m_len_x= 30.0;
  (*o_page_number).m_box.m_len_y= 18.0;
  (*o_page_number).m_font= pango_font_description_new();
  (*o_page_number).m_color.alpha= 1.0;
  (*o_page_number).m_border.m_style= line_style_solid;
  (*o_page_number).m_border.m_thickness= 1.0;
  (*o_page_number).m_border.m_color.alpha= 1.0;
  (*o_page_number).m_fill.m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, page_number);

  return;
}

extern void
item_page_number_copy(
  struct item_page_number *const        o_page_number,
  struct item_page_number const*const   i_page_number)
{

  memcpy(o_page_number, i_page_number, sizeof((*o_page_number)));
  (*o_page_number).m_font= pango_font_description_copy((*i_page_number).m_font);
  (*o_page_number).m_page_number= 0;

  return;
}

extern void
item_page_number_discharge(
  struct item_page_number*const         io_page_number)
{

  g_free((*io_page_number).m_format);

  if ((*io_page_number).m_font)
  {
    pango_font_description_free((*io_page_number).m_font);
  }

  memset(io_page_number, 0, sizeof(*io_page_number));

  return;
}

static void
item_page_number_get_text(
  char**                                o_text,
  struct item_page_number const*const   i_page_number,
  struct item_draw_context const*const  i_ctx)
{
  char const*                           l_format;
  unsigned long                         l_page_number; 

  if (item_draw_design == (*i_ctx).m_option.m_type)
  {
    l_page_number= 1;
  }
  else
  {
    l_page_number= (*i_page_number).m_page_number;
  }

  if ((*i_page_number).m_format && (*i_page_number).m_format[0])
  {
    l_format= (*i_page_number).m_format;
  }
  else
  {
    l_format= "%ld";
  }

  (*o_text)= g_strdup_printf(l_format, l_page_number);

  return;
}

extern void
item_page_number_draw(
  struct item_page_number const*const   i_page_number,
  struct item_draw_context const*const  i_ctx)
{
  double                                l_center_x;
  double                                l_center_y;
  PangoFontDescription*                 l_font;
  PangoLayout*                          l_layout;
  cairo_matrix_t                        l_matrix;
  size_t                                l_modulus;
  double                                l_radians;
  double                                l_shear_x;
  double                                l_shear_y;
  char*                                 l_text;

  l_font= 0;
  l_layout= 0;
  l_text= 0;

  do
  {

    item_page_number_get_text(&l_text, i_page_number, i_ctx);

    if (item_draw_print == (*i_ctx).m_option.m_type)
    {
      l_modulus= ((*i_page_number).m_page_number % 2);

      if (page_number_all != (*i_page_number).m_type)
      {
        if (page_number_even == (*i_page_number).m_type)
        {
          if (l_modulus)
          {
            break;
          }
        }
        else if (page_number_odd == (*i_page_number).m_type)
        {
          if (0 == l_modulus)
          {
            break;
          }
        }
      }
    }

    cairo_new_path((*i_ctx).m_cr);
    cairo_save((*i_ctx).m_cr);
    cairo_scale((*i_ctx).m_cr, (*i_ctx).m_scale, (*i_ctx).m_scale);

    l_layout= pango_cairo_create_layout((*i_ctx).m_cr);
    pango_layout_set_width(l_layout, -1);
    pango_layout_set_text(l_layout, l_text, -1);
    pango_layout_set_font_description(l_layout, (*i_page_number).m_font); 

    l_center_x= (*i_page_number).m_box.m_pos_x + ((*i_page_number).m_box.m_len_x / 2.0);
    l_center_y= (*i_page_number).m_box.m_pos_y + ((*i_page_number).m_box.m_len_y / 2.0);

    cairo_translate((*i_ctx).m_cr, l_center_x, l_center_y);
    l_radians= geom_deg2rad((*i_page_number).m_transform.m_rotation);
    cairo_rotate((*i_ctx).m_cr, l_radians);
    cairo_translate((*i_ctx).m_cr, -l_center_x, -l_center_y);

    l_shear_x= tan(geom_deg2rad((*i_page_number).m_transform.m_shear_x));
    l_shear_y= tan(geom_deg2rad((*i_page_number).m_transform.m_shear_y));
    
    cairo_matrix_init(
      &l_matrix,
      1.0, l_shear_y,
      l_shear_x, 1.0,
      (*i_page_number).m_box.m_pos_x, (*i_page_number).m_box.m_pos_y);
    cairo_transform((*i_ctx).m_cr, &l_matrix);

    if ((*i_page_number).m_fill.m_active)
    {
      cairo_set_source_rgba(
        (*i_ctx).m_cr,
        (*i_page_number).m_fill.m_color.red,
        (*i_page_number).m_fill.m_color.green,
        (*i_page_number).m_fill.m_color.blue,
        (*i_page_number).m_fill.m_color.alpha);

      cairo_rectangle(
        (*i_ctx).m_cr,
        0.0,
        0.0,
        (*i_page_number).m_box.m_len_x,
        (*i_page_number).m_box.m_len_y);

      cairo_fill_preserve((*i_ctx).m_cr);
    }

    if ((*i_page_number).m_border.m_active)
    {

      item_common_cairo_dash_set((*i_ctx).m_cr, (*i_page_number).m_border.m_style);
      cairo_set_line_width((*i_ctx).m_cr, (*i_page_number).m_border.m_thickness);
      cairo_set_source_rgba(
        (*i_ctx).m_cr,
        (*i_page_number).m_border.m_color.red,
        (*i_page_number).m_border.m_color.green,
        (*i_page_number).m_border.m_color.blue,
        (*i_page_number).m_border.m_color.alpha);

      cairo_rectangle(
        (*i_ctx).m_cr,
        0.0,
        0.0,
        (*i_page_number).m_box.m_len_x,
        (*i_page_number).m_box.m_len_y);

      cairo_stroke_preserve((*i_ctx).m_cr);
    }

    cairo_set_source_rgba(
      (*i_ctx).m_cr,
      (*i_page_number).m_color.red,
      (*i_page_number).m_color.green,
      (*i_page_number).m_color.blue,
      (*i_page_number).m_color.alpha);

    pango_cairo_update_layout((*i_ctx).m_cr, l_layout);
    pango_cairo_show_layout((*i_ctx).m_cr, l_layout);

    if ((*i_ctx).m_option.m_highlight && 0 == (*i_ctx).m_selected)
    {
      item_common_cairo_dash_set((*i_ctx).m_cr, line_style_solid);
      cairo_set_line_width((*i_ctx).m_cr, 1.0);
      cairo_set_source_rgb((*i_ctx).m_cr, 0.50, 0.50, 0.00);
      cairo_rectangle(
        (*i_ctx).m_cr,
        0,
        0,
        (*i_page_number).m_box.m_len_x,
        (*i_page_number).m_box.m_len_y);
      cairo_stroke((*i_ctx).m_cr);
    }

    cairo_restore((*i_ctx).m_cr);

  }while(0);

  g_free(l_text);

  if (l_font)
  {
    pango_font_description_free(l_font);
  }

  if (l_layout)
  {
    g_object_unref(l_layout);
  }

  return;
}

extern void
item_page_number_get_bbox(
  struct geom_rect*const                o_bbox,
  cairo_t*                              io_cr,
  double const                          i_scale,
  struct item_page_number const*const   i_page_number)
{
  item_common_box_get_bbox(
    o_bbox, io_cr, &(*i_page_number).m_box, &(*i_page_number).m_transform, i_scale);
  return;
}

extern void
item_page_number_get_position(
  double*const                          o_pos_x,
  double*const                          o_pos_y,
  struct item_page_number const*const   i_page_number)
{

  (*o_pos_x)= (*i_page_number).m_box.m_pos_x;
  (*o_pos_y)= (*i_page_number).m_box.m_pos_y;

  return;
}

extern void
item_page_number_get_rotation(
  double*const                          o_rotation,
  struct item_page_number const*const   i_page_number)
{

  (*o_rotation)= (*i_page_number).m_transform.m_rotation;

  return;
}

extern void
item_page_number_get_size(
  double*const                          o_len_x,
  double*const                          o_len_y,
  struct item_page_number const*const   i_page_number)
{

  (*o_len_x)= (*i_page_number).m_box.m_len_x;
  (*o_len_y)= (*i_page_number).m_box.m_len_y;

  return;
}

extern void
item_page_number_get_shear(
  double*const                          o_shear_x,
  double*const                          o_shear_y,
  struct item_page_number const*const   i_page_number)
{

  (*o_shear_x)= (*i_page_number).m_transform.m_shear_x;
  (*o_shear_y)= (*i_page_number).m_transform.m_shear_y;

  return;
}


extern int
item_page_number_in_event(
  struct item_page_number const*const   i_page_number,
  cairo_t*                              io_cr,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;

  l_hit= item_common_box_in_event(
    io_cr, &(*i_page_number).m_box, &(*i_page_number).m_transform, i_ctx);

  return l_hit;
}

extern void
item_page_number_move(
  struct item_page_number*const         io_page_number,
  struct geom_point const*const         i_event)
{
  item_common_box_move(&(*io_page_number).m_box, i_event);
  return;
}

extern void
item_page_number_normalize(
  struct item_page_number*const         io_page_number)
{
  item_common_box_normalize(&(*io_page_number).m_box);
  return;
}

extern void
item_page_number_set_position(
  struct item_page_number*const         io_page_number,
  double const                          i_pos_x,
  double const                          i_pos_y)
{

  (*io_page_number).m_box.m_pos_x= i_pos_x;
  (*io_page_number).m_box.m_pos_y= i_pos_y;

  return;
}

extern void
item_page_number_set_rotation(
  struct item_page_number*const         io_page_number,
  double                                i_rotation)
{

  (*io_page_number).m_transform.m_rotation= i_rotation;

  return;
}

extern void
item_page_number_set_size(
  struct item_page_number*const         io_page_number,
  double                                i_len_x,
  double                                i_len_y)
{

  (*io_page_number).m_box.m_len_x= i_len_x;
  (*io_page_number).m_box.m_len_y= i_len_y;

  return;
}

extern void
item_page_number_set_shear(
  struct item_page_number*const         io_page_number,
  double                                i_shear_x,
  double                                i_shear_y)
{

  (*io_page_number).m_transform.m_shear_x= i_shear_x;
  (*io_page_number).m_transform.m_shear_y= i_shear_y;

  return;
}

extern int
item_page_number_read(
  GError**                              o_error,
  struct item_page_number*const         io_page_number,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  int                                   l_rc;
  GdkRGBA                               l_rgba;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_rect:
          item_common_box_read(&l_error, &(*io_page_number).m_box, l_xml);
          break;
        case element_tag_transform:
          item_common_read_transform(&l_error, &(*io_page_number).m_transform, l_xml);
          break;
        case element_tag_font:
          (*io_page_number).m_font= pango_font_description_from_string(l_value);
          break;
        case element_tag_color:
          l_rc= gdk_rgba_parse(&l_rgba, l_value);
          if (l_rc)
          {
            (*io_page_number).m_color= l_rgba;
          }
          break;
        case element_tag_border:
          l_exit= item_common_read_border(&l_error, &(*io_page_number).m_border, l_xml);
          break;
        case element_tag_fill:
          l_exit= item_common_read_fill(&l_error, &(*io_page_number).m_fill, l_xml);
          break;
        case element_tag_type:
          (*io_page_number).m_type= page_number_all;

          do
          {
            l_rc= strcmp("even", l_value);
            if (0 == l_rc)
            {
              (*io_page_number).m_type= page_number_even;
              break;
            }

            l_rc= strcmp("odd", l_value);
            if (0 == l_rc)
            {
              (*io_page_number).m_type= page_number_odd;
              break;
            }
          }while(0);

          break;
        case element_tag_format:
          if (l_value[0])
          {
            g_free((*io_page_number).m_format);
            (*io_page_number).m_format= g_strdup(l_value);
          }
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_page_number_resize(
  struct item_page_number*const         io_page_number,
  struct item_resize_event const*const  i_ctx)
{

  item_common_box_resize(&(*io_page_number).m_box, i_ctx, &(*io_page_number).m_transform);

  return;
}

extern void
item_page_number_rotate(
  struct item_page_number*const         io_page_number,
  double const                          i_rotation)
{
  (*io_page_number).m_transform.m_rotation= i_rotation;
}
  
extern void
item_page_number_shear(
  struct item_page_number*const         io_page_number,
  double const                          i_delta_x,
  double const                          i_delta_y)
{

  (*io_page_number).m_transform.m_shear_x+= i_delta_x;
  (*io_page_number).m_transform.m_shear_y+= i_delta_y;

  return;
}

extern int
item_page_number_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_page_number const*const   i_page_number)
{
  int                                   l_exit;
  char*                                 l_text;

  l_exit= 0;
  l_text= 0;

  fprintf(io_fp, "\t\t\t<item name=\"page_number\">\n");
  item_common_box_write(o_error, io_fp, &(*i_page_number).m_box);
  item_common_write_transform(o_error, io_fp, &(*i_page_number).m_transform);
  l_text= pango_font_description_to_string((*i_page_number).m_font);
  fprintf(io_fp, "\t\t\t\t<font>%s</font>\n", l_text);
  g_free(l_text);
  l_text= gdk_rgba_to_string(&(*i_page_number).m_color);
  fprintf(io_fp, "\t\t\t\t<color>%s</color>\n", l_text);
  g_free(l_text);
  item_common_write_border(o_error, io_fp, &(*i_page_number).m_border);
  item_common_write_fill(o_error, io_fp, &(*i_page_number).m_fill);

  switch((*i_page_number).m_type)
  {
    case page_number_all:
      l_text= "all";
      break;
    case page_number_even:
      l_text= "even";
      break;
    case page_number_odd:
      l_text= "odd";
      break;
    default:
      l_text=""; /* programming error */
      break;
  }

  fprintf(io_fp, "\t\t\t\t<type>%s</type>\n", l_text);
  fprintf(io_fp, "\t\t\t\t<format>%s</format>\n", ((*i_page_number).m_format) ? (*i_page_number).m_format : "");
  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
