/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_image.h"

extern void
item_image_assign(
  struct item_image*const               o_image,
  struct item_method*const              o_method)
{

  memset(o_image, 0, sizeof(*o_image));

  (*o_image).m_box.m_len_x= 30.0;
  (*o_image).m_box.m_len_y= 30.0;
  (*o_image).m_border.m_thickness= 1.0;
  (*o_image).m_border.m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, image);

  return;
}

extern void
item_image_copy(
  struct item_image *const              o_image,
  struct item_image const*const         i_image)
{

  memcpy(o_image, i_image, sizeof(*o_image));

  if ((*i_image).m_image)
  {
    (*o_image).m_image= gdk_pixbuf_copy((*i_image).m_image);
  }

  if ((*i_image).m_image_sized)
  {
    (*o_image).m_image_sized= gdk_pixbuf_copy((*i_image).m_image_sized);
  }

  return;
}

extern void
item_image_destroy(
  struct item_image*const                io_image)
{

  if ((*io_image).m_image) 
  {
    g_object_unref((*io_image).m_image);
    (*io_image).m_image= 0;
  }

  if ((*io_image).m_image_sized) 
  {
    g_object_unref((*io_image).m_image_sized);
    (*io_image).m_image_sized= 0;
  }

  return;
}

extern void
item_image_discharge(
  struct item_image*const                io_image)
{

  item_image_destroy(io_image);
  memset(io_image, 0, sizeof(*io_image));

  return;
}

extern void
item_image_draw(
  struct item_image const*const         i_image,
  struct item_draw_context const*const  i_ctx)
{

  do
  {

    if ((*i_image).m_border.m_active)
    {
      item_common_cairo_dash_set((*i_ctx).m_cr, (*i_image).m_border.m_style);
      cairo_set_line_width((*i_ctx).m_cr, (*i_image).m_border.m_thickness);
      cairo_set_source_rgba(
        (*i_ctx).m_cr,
        (*i_image).m_border.m_color.red,
        (*i_image).m_border.m_color.green,
        (*i_image).m_border.m_color.blue,
        (*i_image).m_border.m_color.alpha);
      cairo_rectangle(
        (*i_ctx).m_cr,
        (*i_image).m_box.m_pos_x,
        (*i_image).m_box.m_pos_y,
        (*i_image).m_box.m_len_x,
        (*i_image).m_box.m_len_y);
      cairo_stroke((*i_ctx).m_cr);
    }

    if ((*i_image).m_image_sized)
    {
      gdk_cairo_set_source_pixbuf(
        (*i_ctx).m_cr, 
        (*i_image).m_image_sized,
        (*i_image).m_box.m_pos_x,
        (*i_image).m_box.m_pos_y);

      cairo_paint((*i_ctx).m_cr);
    }
    else if ((item_draw_design == (*i_ctx).m_option.m_type) && 
      (0 == (*i_image).m_border.m_active))
    {
      item_common_cairo_dash_set((*i_ctx).m_cr, line_style_dash);
      cairo_set_line_width((*i_ctx).m_cr, 1.0);
      cairo_set_source_rgb((*i_ctx).m_cr, 0.0, 0.0, 0.0);
      cairo_rectangle(
        (*i_ctx).m_cr,
        (*i_image).m_box.m_pos_x,
        (*i_image).m_box.m_pos_y,
        (*i_image).m_box.m_len_x,
        (*i_image).m_box.m_len_y);
      cairo_stroke((*i_ctx).m_cr);
    }

    if (0 == (*i_ctx).m_selected)
    {
      if ((*i_ctx).m_option.m_highlight &&
        (item_draw_design == (*i_ctx).m_option.m_type))
      {
        item_common_box_draw_hightlight((*i_ctx).m_cr, &(*i_image).m_box, 1.0);
      }
      break;
    }

  }while(0);

  return;
}

extern void
item_image_get_bbox(
  struct geom_rect*const                o_bbox,
  cairo_t*                              io_cr,
  double const                          i_scale,
  struct item_image const*const         i_image)
{

  (*o_bbox)= (*i_image).m_box;

  return;
}

extern void
item_image_get_position(
  double*const                          o_pos_x,
  double*const                          o_pos_y,
  struct item_image const*const         i_image)
{

  (*o_pos_x)= (*i_image).m_box.m_pos_x;
  (*o_pos_y)= (*i_image).m_box.m_pos_y;

  return;
}

extern void
item_image_get_rotation(
  double*const                          o_rotation,
  struct item_image const*const         i_image)
{

  (*o_rotation)= 0.0;

  return;
}

extern void
item_image_get_size(
  double*const                          o_len_x,
  double*const                          o_len_y,
  struct item_image const*const         i_image)
{

  (*o_len_x)= (*i_image).m_box.m_len_x;
  (*o_len_y)= (*i_image).m_box.m_len_y;

  return;
}

extern void
item_image_get_shear(
  double*const                          o_shear_x,
  double*const                          o_shear_y,
  struct item_image const*const         i_image)
{
  
  (*o_shear_x)= 0.0;
  (*o_shear_y)= 0.0;

  return;
}

extern int
item_image_in_event(
  struct item_image const*const         i_image,
  cairo_t*                              io_cr,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;
  struct item_transform                 l_transform;

  memset(&l_transform, 0, sizeof(l_transform));

  l_hit= item_common_box_in_event(io_cr, &(*i_image).m_box, &l_transform, i_ctx);

  return l_hit;
}

extern int
item_image_load(
  GError**                              o_error,
  struct item_image*const               io_image,
  char const*                           i_filename)
{
  GError*                               l_error;
  int                                   l_exit;

  l_error= 0;
  l_exit= 0;

  do
  {

    if ((0 == i_filename) || (0 == i_filename[0]))
    {
      break;
    }

    item_image_destroy(io_image);

    (*io_image).m_image= gdk_pixbuf_new_from_file(i_filename, &l_error);

    if (l_error)
    {
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    (*io_image).m_image_sized= gdk_pixbuf_scale_simple(
      (*io_image).m_image,
      (*io_image).m_box.m_len_x,
      (*io_image).m_box.m_len_y,
      GDK_INTERP_BILINEAR);

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_image_move(
  struct item_image*const               io_image,
  struct geom_point const*const         i_event)
{
  item_common_box_move(&(*io_image).m_box, i_event);
  return;
}

extern void
item_image_normalize(
  struct item_image*const               io_image)
{

  item_common_box_normalize(&(*io_image).m_box);

  do
  {

    if (0 == (*io_image).m_image)
    {
      break;
    }

    if ((*io_image).m_image_sized)
    {
      g_object_unref((*io_image).m_image_sized);
      (*io_image).m_image_sized= 0;
    }

    (*io_image).m_image_sized= gdk_pixbuf_scale_simple(
      (*io_image).m_image,
      (*io_image).m_box.m_len_x,
      (*io_image).m_box.m_len_y,
      GDK_INTERP_BILINEAR);

  }while(0);

  return;
}

extern void
item_image_set_position(
  struct item_image*const               io_image,
  double const                          i_pos_x,
  double const                          i_pos_y)
{

  (*io_image).m_box.m_pos_x= i_pos_x;
  (*io_image).m_box.m_pos_y= i_pos_y;

  return;
}

extern void
item_image_set_rotation(
  struct item_image*const               io_image,
  double                                i_rotation)
{
  return;
}

extern void
item_image_set_size(
  struct item_image*const               io_image,
  double                                i_len_x,
  double                                i_len_y)
{

  (*io_image).m_box.m_len_x= i_len_x;
  (*io_image).m_box.m_len_y= i_len_y;

  return;
}

extern void
item_image_set_shear(
  struct item_image*const               io_image,
  double const                          i_shear_x,
  double const                          i_shear_y)
{
  return;
}

extern int
item_image_read(
  GError**                              o_error,
  struct item_image*const               io_image,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  enum element_tag_type                 l_type;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_rect:
          item_common_box_read(&l_error, &(*io_image).m_box, l_xml);
          break;
        case element_tag_border:
          l_exit= item_common_read_border(&l_error, &(*io_image).m_border, l_xml);
          break;
        case element_tag_data:
          l_exit= pixbuf_read(&l_error, &(*io_image).m_image, l_xml);
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (0 == l_exit && (*io_image).m_image)
  {
    (*io_image).m_image_sized= gdk_pixbuf_scale_simple(
      (*io_image).m_image,
      (*io_image).m_box.m_len_x,
      (*io_image).m_box.m_len_y,
      GDK_INTERP_BILINEAR);
  }

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_image_resize(
  struct item_image*const               io_image,
  struct item_resize_event const*const  i_ctx)
{
  struct item_transform                 l_transform;

  memset(&l_transform, 0, sizeof(l_transform));
  item_common_box_resize(&(*io_image).m_box, i_ctx, &l_transform);

  return;
}

extern void
item_image_rotate(
  struct item_image*const               io_image,
  double const                          i_rotation)
{
}
  
extern void
item_image_shear(
  struct item_image*const               io_image,
  double const                          i_delta_x,
  double const                          i_delta_y)
{
}

extern int
item_image_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_image const*const         i_image)
{
  int                                   l_exit;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t<item name=\"image\">\n");
  item_common_box_write(o_error, io_fp, &(*i_image).m_box);
  item_common_write_border(o_error, io_fp, &(*i_image).m_border);

  if ((*i_image).m_image)
  {
    pixbuf_write(io_fp, (*i_image).m_image);
  }

  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
