/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_frame.h"

extern void
item_frame_assign(
  struct item_frame*const               o_object,
  struct item_method*const              o_method)
{

  memset(o_object, 0, sizeof(*o_object));

  (*o_object).m_border.m_style= line_style_solid;
  (*o_object).m_box.m_len_x= 30.0;
  (*o_object).m_box.m_len_y= 18.0;
  (*o_object).m_border.m_thickness= 1.0;
  (*o_object).m_border.m_color.alpha= 1.0;
  (*o_object).m_fill.m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, frame);

  return;
}

extern void
item_frame_copy(
  struct item_frame *const              o_object,
  struct item_frame const*const         i_object)
{

  memcpy(o_object, i_object, sizeof(*o_object));

  return;
}

extern void
item_frame_discharge(
  struct item_frame*const                io_object)
{
  memset(io_object, 0, sizeof(*io_object));
}

extern void
item_frame_draw(
  struct item_frame const*const         i_object,
  struct item_draw_context const*const  i_ctx)
{
  double                                l_center_x;
  double                                l_center_y;
  cairo_matrix_t                        l_matrix;
  double                                l_shear_x;
  double                                l_shear_y;
  double                                l_x1;
  double                                l_x2;
  double                                l_y1;
  double                                l_y2;

  l_center_x= (*i_object).m_box.m_pos_x + ((*i_object).m_box.m_len_x / 2.0);
  l_center_y= (*i_object).m_box.m_pos_y + ((*i_object).m_box.m_len_y / 2.0);
  l_shear_x = tan(geom_deg2rad((*i_object).m_transform.m_shear_x));
  l_shear_y = tan(geom_deg2rad((*i_object).m_transform.m_shear_y));

  cairo_new_path((*i_ctx).m_cr);
  cairo_save((*i_ctx).m_cr);
  cairo_scale((*i_ctx).m_cr, (*i_ctx).m_scale, (*i_ctx).m_scale);
  cairo_translate((*i_ctx).m_cr, l_center_x, l_center_y);
  cairo_rotate((*i_ctx).m_cr, geom_deg2rad((*i_object).m_transform.m_rotation));
  cairo_translate((*i_ctx).m_cr, -l_center_x, -l_center_y);
  cairo_matrix_init(
      &l_matrix,
      1.0, l_shear_y,
      l_shear_x, 1.0,
      (*i_object).m_box.m_pos_x, (*i_object).m_box.m_pos_y);
  cairo_transform((*i_ctx).m_cr, &l_matrix);
  cairo_rectangle((*i_ctx).m_cr, 0.0, 0.0, (*i_object).m_box.m_len_x, (*i_object).m_box.m_len_y);

  if ((*i_object).m_fill.m_active)
  {
    cairo_set_source_rgba(
      (*i_ctx).m_cr,
      (*i_object).m_fill.m_color.red,
      (*i_object).m_fill.m_color.green,
      (*i_object).m_fill.m_color.blue,
      (*i_object).m_fill.m_color.alpha);
    cairo_fill_preserve((*i_ctx).m_cr);
  }

  cairo_restore((*i_ctx).m_cr);

  if (0 == (*i_ctx).m_option.m_highlight || (*i_ctx).m_selected)
  {
    item_common_cairo_dash_set((*i_ctx).m_cr, (*i_object).m_border.m_style);
    cairo_set_line_width((*i_ctx).m_cr, (*i_object).m_border.m_thickness);
    cairo_set_source_rgba(
      (*i_ctx).m_cr,
      (*i_object).m_border.m_color.red,
      (*i_object).m_border.m_color.green,
      (*i_object).m_border.m_color.blue,
      (*i_object).m_border.m_color.alpha);
    cairo_stroke((*i_ctx).m_cr);
  }
  else
  {
    cairo_stroke_extents((*i_ctx).m_cr, &l_x1, &l_y1, &l_x2, &l_y2);
    item_common_cairo_dash_set((*i_ctx).m_cr, line_style_solid);
    cairo_set_line_width((*i_ctx).m_cr, 1.0);
    cairo_set_source_rgb((*i_ctx).m_cr, 0.50, 0.50, 0.00);
    cairo_rectangle((*i_ctx).m_cr, l_x1, l_y1, l_x2-l_x1, l_y2-l_y1);
    cairo_stroke((*i_ctx).m_cr);
  }

  return;
}

extern void
item_frame_get_bbox(
  struct geom_rect*const                o_bbox,
  cairo_t*                              io_cr,
  double const                          i_scale,
  struct item_frame const*const         i_object)
{

  item_common_box_get_bbox(o_bbox, io_cr, &(*i_object).m_box, &(*i_object).m_transform, i_scale);

  return;
}

extern void
item_frame_get_position(
  double*const                          o_pos_x,
  double*const                          o_pos_y,
  struct item_frame const*const         i_object)
{

  (*o_pos_x)= (*i_object).m_box.m_pos_x;
  (*o_pos_y)= (*i_object).m_box.m_pos_y;

  return;
}

extern void
item_frame_get_rotation(
  double*const                          o_rotation,
  struct item_frame const*const         i_object)
{

  (*o_rotation)= (*i_object).m_transform.m_rotation;

  return;
}

extern void
item_frame_get_size(
  double*const                          o_len_x,
  double*const                          o_len_y,
  struct item_frame const*const         i_object)
{

  (*o_len_x)= (*i_object).m_box.m_len_x;
  (*o_len_y)= (*i_object).m_box.m_len_y;

  return;
}

extern void
item_frame_get_shear(
  double*const                          o_shear_x,
  double*const                          o_shear_y,
  struct item_frame const*const         i_object)
{

  (*o_shear_x)= (*i_object).m_transform.m_shear_x;
  (*o_shear_y)= (*i_object).m_transform.m_shear_y;

  return;
}

extern int
item_frame_in_event(
  struct item_frame const*const         i_object,
  cairo_t*                              io_cr,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;

  l_hit= item_common_box_in_event(io_cr, &(*i_object).m_box, &(*i_object).m_transform, i_ctx);

  return l_hit;
}

extern void
item_frame_move(
  struct item_frame*const               io_object,
  struct geom_point const*const         i_event)
{
  item_common_box_move(&(*io_object).m_box, i_event);
  return;
}

extern void
item_frame_normalize(
  struct item_frame*const               io_object)
{
  item_common_box_normalize(&(*io_object).m_box);
  return;
}

extern void
item_frame_set_position(
  struct item_frame*const               io_object,
  double const                          i_pos_x,
  double const                          i_pos_y)
{

  (*io_object).m_box.m_pos_x= i_pos_x;
  (*io_object).m_box.m_pos_y= i_pos_y;

  return;
}

extern void
item_frame_set_rotation(
  struct item_frame*const               io_object,
  double                                i_rotation)
{

  (*io_object).m_transform.m_rotation= i_rotation;

  return;
}

extern void
item_frame_set_size(
  struct item_frame*const               io_object,
  double                                i_len_x,
  double                                i_len_y)
{

  (*io_object).m_box.m_len_x= i_len_x;
  (*io_object).m_box.m_len_y= i_len_y;

  return;
}

extern void
item_frame_set_shear(
  struct item_frame*const               io_object,
  double                                i_shear_x,
  double                                i_shear_y)
{

  (*io_object).m_transform.m_shear_x= i_shear_x;
  (*io_object).m_transform.m_shear_y= i_shear_y;

  return;
}


extern int
item_frame_read(
  GError**                              o_error,
  struct item_frame*const               io_object,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  enum element_tag_type                 l_type;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_rect:
          item_common_box_read(&l_error, &(*io_object).m_box, l_xml);
          break;
        case element_tag_transform:
          item_common_read_transform(&l_error, &(*io_object).m_transform, l_xml);
          break;
        case element_tag_border:
          l_exit= item_common_read_border(&l_error, &(*io_object).m_border, l_xml);
          (*io_object).m_border.m_active= 1;
          break;
        case element_tag_fill:
          l_exit= item_common_read_fill(&l_error, &(*io_object).m_fill, l_xml);
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_frame_resize(
  struct item_frame*const               io_object,
  struct item_resize_event const*const  i_ctx)
{

  item_common_box_resize(&(*io_object).m_box, i_ctx, &(*io_object).m_transform);

  return;
}

extern void
item_frame_rotate(
  struct item_frame*const               io_object,
  double const                          i_rotation)
{
  (*io_object).m_transform.m_rotation= i_rotation;
}
  
extern void
item_frame_shear(
  struct item_frame*const               io_object,
  double const                          i_delta_x,
  double const                          i_delta_y)
{

  (*io_object).m_transform.m_shear_x+= i_delta_x;
  (*io_object).m_transform.m_shear_y+= i_delta_y;

  return;
}

extern int
item_frame_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_frame const*const         i_object)
{
  int                                   l_exit;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t<item name=\"frame\">\n");
  item_common_box_write(o_error, io_fp, &(*i_object).m_box);
  item_common_write_transform(o_error, io_fp, &(*i_object).m_transform);
  item_common_write_border(o_error, io_fp, &(*i_object).m_border);
  item_common_write_fill(o_error, io_fp, &(*i_object).m_fill);
  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
