/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item.h"

extern void
item_assign(
  struct item*const                     o_item)
{
  memset(o_item, 0, sizeof(*o_item));
  return;
}

extern void
item_clear(
  struct item*const                     io_item)
{
  struct item_node*                     l_node;
  struct item_node*                     l_next;

  l_node= (*io_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_next= (*l_node).m_next;

    (*(*l_node).m_method.m_discharge)(&(*l_node).m_object);

    g_free(l_node);
    l_node= l_next;

  }while(1);

  memset(io_item, 0, sizeof(*io_item));

  return;
}

extern void
item_deselect_all(
  struct item*const                     io_item)
{
  struct item_node*                     l_node;

  l_node= (*io_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    (*l_node).m_selected= 0;

    l_node= (*l_node).m_next;
  
  }while(1);

  return;
}

extern void
item_discharge(
  struct item*const                     io_item)
{
  item_clear(io_item);
  return;
}

extern void
item_draw(
  struct item*const                     io_item,
  cairo_t*const                         i_cr,
  double const                          i_scale,
  struct item_draw_options const*const  i_option)
{
  struct item_draw_context              l_ctx;
  struct item_node*                     l_node;

  memset(&l_ctx, 0, sizeof(l_ctx));
  l_node= (*io_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    if (i_option)
    {
      l_ctx.m_option= (*i_option);
    }

    l_ctx.m_cr= i_cr;
    l_ctx.m_scale= i_scale;

    if (item_draw_design == l_ctx.m_option.m_type)
    {
      l_ctx.m_selected= (*l_node).m_selected;
    }

    (*(*l_node).m_method.m_draw)(&(*l_node).m_object, &l_ctx);

    l_node= (*l_node).m_next;

  }while(1);

  return;
}

extern void
item_move_down(
  struct item*const                     io_item,
  struct item_node*const                io_node)
{
  struct item_node*                     l_node;

  do
  {
    l_node= (*io_node).m_prev;

    if (0 == l_node)
    {
      break;
    }
  
    item_node_unlink(io_item, io_node);
    item_node_insert_before(io_item, l_node, io_node);

  }while(0);

  return;
}

extern void
item_move_to_back(
  struct item*const                     io_item,
  struct item_node*const                io_node)
{
  struct item_node*                     l_node;

  do
  {
    l_node= (*io_node).m_prev;

    if (0 == l_node)
    {
      break;
    }
  
    item_node_unlink(io_item, io_node);
    item_node_insert_before(io_item, (*io_item).m_head, io_node);

  }while(0);

  return;
}

extern void
item_move_to_front(
  struct item*const                     io_item,
  struct item_node*const                io_node)
{
  struct item_node*                     l_node;

  do
  {
    l_node= (*io_node).m_next;

    if (0 == l_node)
    {
      break;
    }
  
    item_node_unlink(io_item, io_node);
    item_node_append(io_item, io_node);

  }while(0);

  return;
}

extern void
item_move_up(
  struct item*const                     io_item,
  struct item_node*const                io_node)
{
  struct item_node*                     l_node;

  do
  {
    l_node= (*io_node).m_next;

    if (0 == l_node)
    {
      break;
    }
  
    item_node_unlink(io_item, io_node);
    item_node_insert_after(io_item, l_node, io_node);

  }while(0);

  return;
}

extern void
item_node_get_nth(
  struct item_node**                    o_node,
  struct item*const                     io_item,
  unsigned const                        i_slot)
{
  struct item_node*                     l_node;
  unsigned                              l_slot;

  (*o_node)= 0;
  l_node= (*io_item).m_head;
  l_slot= 0;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    if (i_slot == l_slot)
    {
      (*o_node)= l_node;
      break;
    }

    l_node= (*l_node).m_next;
    l_slot++;

  }while(1);

  return;
}

extern int
item_node_in_event(
  struct item_node**                    o_node,
  cairo_t*                              io_cr,
  struct item const*const               i_item,
  struct geom_point const*const         i_event,
  double const                          i_scale)
{
  struct item_in_event                  l_ctx;
  int                                   l_hit;
  struct item_node*                     l_node;

  (*o_node)= 0;
  memset(&l_ctx, 0, sizeof(l_ctx));
  l_ctx.m_event= *i_event;
  l_ctx.m_scale= i_scale; 
  l_hit= 0;
  l_node= (*i_item).m_tail;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_hit= (*(*l_node).m_method.m_in_event)(&(*l_node).m_object, io_cr, &l_ctx);

    if (l_hit)
    {
      (*o_node)= l_node;
      break;
    }

    l_node= (*l_node).m_prev;

  }while(1);

  return l_hit;
}

extern void
item_selected_move(
  struct item*const                     io_item,
  struct geom_point const*const         i_event)
{
  struct item_node*                     l_node;

  l_node= (*io_item).m_tail;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    if ((*l_node).m_selected)
    {
      (*(*l_node).m_method.m_move)(&(*l_node).m_object, i_event);
    }

    l_node= (*l_node).m_prev;
  
  }while(1);

  return;
}

extern void
item_normalize(
  struct item*const                     io_item)
{
  struct item_node*                     l_node;

  l_node= (*io_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    (*(*l_node).m_method.m_normalize)(&(*l_node).m_object);

    l_node= (*l_node).m_next;
  
  }while(1);

  return;
}

extern void
item_selected_normalize(
  struct item*const                     io_item)
{
  struct item_node*                     l_node;

  l_node= (*io_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    if ((*l_node).m_selected)
    {
      (*(*l_node).m_method.m_normalize)(&(*l_node).m_object);
    }

    l_node= (*l_node).m_next;
  
  }while(1);

  return;
}

extern void
item_selected_remove(
  struct item*const                     io_item)
{
  struct item_node*                     l_node;
  struct item_node*                     l_next;

  l_node= (*io_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_next= (*l_node).m_next;

    if ((*l_node).m_selected)
    {
      item_node_remove(io_item, l_node);
    }

    l_node= l_next;

  
  }while(1);

  return;
}

extern void
item_selected_resize(
  struct item*const                     io_item,
  struct item_resize_event const*const  i_ctx)
{
  struct item_node*                     l_node;

  l_node= (*io_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    if ((*l_node).m_selected)
    {
      (*(*l_node).m_method.m_resize)(&(*l_node).m_object, i_ctx);
    }

    l_node= (*l_node).m_next;
  
  }while(1);

  return;
}

extern void
item_node_append(
  struct item*const                     io_item,
  struct item_node*const                i_node)
{

  if (0 == (*io_item).m_head)
  {
    (*io_item).m_head= i_node;
  }
  else
  {
    (*i_node).m_prev= (*io_item).m_tail;
    (*io_item).m_tail->m_next= i_node;
  }

  (*io_item).m_tail= i_node;

  return;
}

extern void
item_node_insert_after(
  struct item*const                     io_item,
  struct item_node*const                i_node,
  struct item_node*const                i_node_new)
{

  if (0 == (*i_node).m_next)
  {
    (*io_item).m_tail= i_node_new;
  }
  else
  {
    (*i_node).m_next->m_prev= i_node_new;
  }

  (*i_node_new).m_next= (*i_node).m_next;
  (*i_node_new).m_prev= i_node;
  (*i_node).m_next= i_node_new;

  return;
}

extern void
item_node_insert_before(
  struct item*const                     io_item,
  struct item_node*const                i_node,
  struct item_node*const                i_node_new)
{

  if (0 == (*i_node).m_prev)
  {
    (*io_item).m_head= i_node_new;
  }
  else
  {
    (*i_node).m_prev->m_next= i_node_new;
  }

  (*i_node_new).m_next= i_node;
  (*i_node_new).m_prev= (*i_node).m_prev;
  (*i_node).m_prev= i_node_new;

  return;
}

extern void
item_node_new(
  struct item_node**                    o_node,
  struct item*const                     io_item,
  enum item_type const                  i_type)
{
  struct item_node*                     l_node;

  l_node= (struct item_node*)g_malloc0(sizeof(*l_node));

  switch(i_type)
  {
    case item_type_circle:
      item_circle_assign(&(*l_node).m_object.m_circle, &(*l_node).m_method);
      break;
    case item_type_date:
      item_date_assign(&(*l_node).m_object.m_date, &(*l_node).m_method);
      break;
    case item_type_line:
      item_line_assign(&(*l_node).m_object.m_line, &(*l_node).m_method);
      break;
    case item_type_fbarcode:
      item_fbarcode_assign(&(*l_node).m_object.m_fbarcode, &(*l_node).m_method);
      break;
    case item_type_fimage:
      item_fimage_assign(&(*l_node).m_object.m_fimage, &(*l_node).m_method);
      break;
    case item_type_fnumber:
      item_fnumber_assign(&(*l_node).m_object.m_fnumber, &(*l_node).m_method);
      break;
    case item_type_ftext:
      item_ftext_assign(&(*l_node).m_object.m_ftext, &(*l_node).m_method);
      break;
    case item_type_frame:
      item_frame_assign(&(*l_node).m_object.m_frame, &(*l_node).m_method);
      break;
    case item_type_page_number:
      item_page_number_assign(&(*l_node).m_object.m_page_number, &(*l_node).m_method);
      break;
    case item_type_fsum:
      item_fsum_assign(&(*l_node).m_object.m_fsum, &(*l_node).m_method);
      break;
    case item_type_report:
      item_report_assign(&(*l_node).m_object.m_report, &(*l_node).m_method);
      break;
    case item_type_text:
      item_text_assign(&(*l_node).m_object.m_text, &(*l_node).m_method);
      break;
    case item_type_image:
      item_image_assign(&(*l_node).m_object.m_image, &(*l_node).m_method);
      break;
    default:
      g_free(l_node);
      l_node= 0;
      break;
  }

  if (l_node)
  {
    item_deselect_all(io_item);
    item_node_append(io_item, l_node);
    (*l_node).m_type= i_type;
    (*l_node).m_selected= 1;
  }

  (*o_node)= l_node;

  return;
}

extern void
item_node_property(
  struct item_node*const                io_node)
{

  (*(*io_node).m_method.m_property)(&(*io_node).m_object);

  return;
}

extern void
item_node_remove(
  struct item*const                     io_item,
  struct item_node*                     i_node)
{

  do
  {

    item_node_unlink(io_item, i_node);
    (*(*i_node).m_method.m_discharge)(&(*i_node).m_object);

    g_free(i_node);

  }while(0);

  return;
}

extern void
item_node_unlink(
  struct item*const                     io_item,
  struct item_node*                     i_node)
{

  if (0 == (*i_node).m_prev)
  {
    (*io_item).m_head= (*i_node).m_next;
  }
  else
  {
    (*i_node).m_prev->m_next= (*i_node).m_next;
  }

  if (0 == (*i_node).m_next)
  {
    (*io_item).m_tail= (*i_node).m_prev;
  }
  else
  {
    (*i_node).m_next->m_prev= (*i_node).m_prev;
  }
  
  (*i_node).m_next= 0;
  (*i_node).m_prev= 0;

  return;
}

extern int
item_node_write(
  GError**                              o_error,
  FILE*const                            i_fp,
  struct item const*const               i_item)
{
  struct item_node const*               l_node;

  l_node= (*i_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    (*(*l_node).m_method.m_write)(o_error, i_fp, &(*l_node).m_object);

    l_node= (*l_node).m_next;
  
  }while(1);

  return 0;
}

extern void
item_selected_rotate(
  struct item*const                     io_item,
  double const                          i_rotation)
{
  struct item_node*                     l_node;

  l_node= (*io_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    if ((*l_node).m_selected)
    {
      (*(*l_node).m_method.m_rotate)(&(*l_node).m_object, i_rotation);
    }

    l_node= (*l_node).m_next;
  
  }while(1);

  return;
}

extern void
item_selected_shear(
  struct item*const                     io_item,
  double const                          i_delta_x,
  double const                          i_delta_y)
{
  struct item_node*                     l_node;

  l_node= (*io_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    if ((*l_node).m_selected)
    {
      (*(*l_node).m_method.m_shear)(&(*l_node).m_object, i_delta_x, i_delta_y);
    }

    l_node= (*l_node).m_next;
  
  }while(1);

  return;
}

extern int
item_read(
  GError**                              o_error,
  struct item *const                    io_item,
  struct xml *const                     i_xml)
{
  char const*                           l_attr_name;
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  struct item_node*                     l_node;
  enum element_tag_type                 l_type;

  l_error= 0;
  l_exit= 0;

  do
  {

    l_attr_name= xml_lookup_attribute_value(i_xml, "name");

    if (0 == l_attr_name)
    {
      l_error= g_error_new(
        ITEM,
        ITEM_XML_MISSING_ATTRIBUTE,
        "Missing 'name' attribute\n");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    item_common_lookup(&l_found, &l_type, l_attr_name);

    if (0 == l_found)
    {
      l_error= g_error_new(
        ITEM,
        ITEM_XML_UNKNOWN_ITEM,
        "Unkown item ('%s')\n",
        l_attr_name);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    item_node_new(&l_node, io_item, (enum item_type)l_type);

    if (0 == l_node)
    {
      break;
    }

    l_exit= (*(*l_node).m_method.m_read)(&l_error, &(*l_node).m_object, i_xml);

    (*l_node).m_selected= 0;

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}
