/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "export_csv.h"
#include "run_dialog.h"

static int
print_detail_cell(
  FILE*const                            io_fp,
  struct iwsql_attr const*const         i_attr)
{
  int                                   l_exit;
  char*                                 l_ptr;
  
  l_exit= 0;
  
  do
  {

    if (0 == (*i_attr).m_value)
    {
      break;
    }

    fprintf(io_fp, "\"");

    for (l_ptr= (*i_attr).m_value; *l_ptr; l_ptr++)
    {
      if ('"' == *l_ptr)
      {
        fprintf(io_fp, "\"\"");
      }
      else
      {
        fprintf(io_fp, "%c", *l_ptr);
      }
    }
    
    fprintf(io_fp, "\"");

  }while(0);

  return l_exit;
}

static int
csv_print_detail(
  FILE*const                            io_fp,
  struct query_exec const*const         i_exec)
{
  int                                   l_exit;
  unsigned                              l_slot_sql;
  unsigned                              l_slot_attr;
  struct query_exec_node const*         l_node;
  
  l_exit= 0;
  for (l_node= &(*i_exec).m_sql[0], l_slot_sql= 0;
    (*i_exec).m_count > l_slot_sql;
    l_slot_sql++,
    l_node++)
  {

    l_slot_attr= 0; 

    do
    {

      if ((*l_node).m_attr_count <= l_slot_attr)
      {
        break;
      }

      print_detail_cell(io_fp, &(*l_node).m_attr[l_slot_attr]);

      l_slot_attr++;

      if ((*l_node).m_attr_count > l_slot_attr)
      {
        fprintf(io_fp, ",");
      }

    }while(1);  

    fprintf(io_fp, "\r\n");

  }

  return l_exit;
}

static int
csv_print_header(
  FILE*const                            io_fp,
  struct query_exec const*const         i_exec)
{
  int                                   l_exit;
  unsigned                              l_slot_sql;
  unsigned                              l_slot_attr;
  struct query_exec_node const*         l_node;
  
  l_exit= 0;
  for (l_node= &(*i_exec).m_sql[0], l_slot_sql= 0;
    (*i_exec).m_count > l_slot_sql;
    l_slot_sql++,
    l_node++)
  {

    l_slot_attr= 0; 

    do
    {

      if ((*l_node).m_attr_count <= l_slot_attr)
      {
        break;
      }

      fprintf(io_fp, "\"%s\"", (*l_node).m_attr[l_slot_attr].m_tag);

      l_slot_attr++;

      if ((*l_node).m_attr_count > l_slot_attr)
      {
        fprintf(io_fp, ",");
      }

    }while(1);  

  }

  if (l_slot_attr)
  {
    fprintf(io_fp, "\r\n");
  }

  return l_exit;
}

static int
csv_run_loop(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct query_exec *const              io_exec)
{
  int                                   l_exit;
  int                                   l_eof;
  gint64                                l_time1;
  gint64                                l_time2;

  l_exit= 0;
  l_time1= g_get_monotonic_time();

  do
  {

    csv_print_detail(io_fp, io_exec);

    if (run_dialog_get_cancelled())
    {
      l_exit= -1;
      break;
    }

    l_exit= query_exec_next(o_error, &l_eof, io_exec);

    if (l_exit)
    {
      break;
    }

    if (l_eof)
    {
      break;
    }

    run_dialog_increment_tuple();

    if (run_dialog_get_instantiated())
    {
      l_time2= g_get_monotonic_time();
      if (G_USEC_PER_SEC < (l_time2 - l_time1))
      {
        l_time1= l_time2;
        gdk_threads_add_idle(run_dialog_update, 0);
      }
    }

  }while(1);

  if (run_dialog_get_instantiated())
  {
    gdk_threads_add_idle(run_dialog_update, 0);
  }

  return l_exit;
}

static int
csv_run_exec(
  struct exports_spec *const            io_export,
  struct goop const*const               i_goop)
{ 
  int                                   l_eof;
  GError*                               l_error;
  struct query_exec                     l_exec;
  int                                   l_exit;
  FILE*                                 l_fp;

  l_exit= 0;
  l_error= 0;
  query_exec_assign(&l_exec);
  l_fp= 0;

  do
  {
    
    l_fp= g_fopen((*i_goop).m_outputfile, "w+");

    if (0 == l_fp)
    {
      l_error= g_error_new(
        CSV,
        CSV_CREATE_FILE,
        "Unable to create file ('%s'); errno(%d)='%s'",
        (*i_goop).m_outputfile,
        errno,
        strerror(errno));
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_exit= report_util_resolve_password(&(*io_export).m_aspect, (*i_goop).m_interactive);

    if (l_exit)
    {
      break;
    }

    l_exit= query_exec_connect_prepare_first(
      &l_error,
      &l_eof,
      &l_exec,
      &(*io_export).m_query,
      &(*io_export).m_aspect,
      (*i_goop).m_prompt,
      (*i_goop).m_interactive);

    if (l_exit)
    {
      break;
    }

    if (l_eof)
    {
      break;
    }

    if ((*io_export).m_config.m_print_header)
    {
      csv_print_header(l_fp, &l_exec);
    }

    csv_run_loop(&l_error, l_fp, &l_exec);

  }while(0);

  if (l_error)
  {
    if ((*i_goop).m_interactive)
    {
      _error_display_prompt(l_error);
    }
  }

  g_clear_error(&l_error);

  if (l_fp)
  {
    fclose(l_fp);
  }

  query_exec_discharge(&l_exec);

  return l_exit;
}

struct csv_thread_data
{
  struct exports_spec*                  m_export;
  struct goop const*                    m_goop;
};

static gpointer
csv_run_thread(
  gpointer                              i_data)
{
  int                                   l_exit;
  struct csv_thread_data*               l_info;

  l_info= (struct csv_thread_data*)i_data;

  l_exit= csv_run_exec((*l_info).m_export, (*l_info).m_goop);

  if (0 == run_dialog_get_cancelled())
  {
    gdk_threads_add_idle(run_dialog_end, 0);
  }

  g_thread_exit((gpointer)(ptrdiff_t)l_exit);

  return (gpointer)(ptrdiff_t)l_exit;
}

extern int
export_csv_run(
  struct exports_spec *const            io_export,
  struct goop const*const               i_goop)
{
  int                                   l_exit;
  struct csv_thread_data                l_info;
  int                                   l_rc;
  GThread*                              l_thread;

  memset(&g_run_dialog, 0, sizeof(g_run_dialog));
  l_exit= 0;
  l_thread= 0;
  memset(&l_info, 0, sizeof(l_info));

  do
  {

    if (0 == (*i_goop).m_interactive)
    {
      l_exit= csv_run_exec(io_export, i_goop);
      break;
    }

    l_info.m_export= io_export;
    l_info.m_goop= i_goop;

    l_exit= run_dialog_initialize_instance((*i_goop).m_reportfile);

    if (l_exit)
    {
      break;
    }

    l_thread= g_thread_new("csv", csv_run_thread, &l_info);

    gtk_dialog_run(run_dialog_get_dialog());
    g_run_dialog.m_cancel= 1;

    l_exit= (int)(ptrdiff_t)g_thread_join(l_thread);

    if (l_exit)
    {
      break;
    }

    l_rc= gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON(g_run_dialog.m_open_checkbutton));

    if (0 == l_rc)
    {
      l_exit= 1;
      break;
    }

  }while(0);

  run_dialog_deinitialize_instance();

  if (l_thread)
  {
    g_thread_unref(l_thread);
  }

  return l_exit;
}
