/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "database_config.h"
#include "wallet.h"

extern void
database_config_assign(
  struct database_config*const          o_config)
{

  memset(o_config, 0, sizeof(*o_config));

  return;
}

extern void
database_config_discharge(
  struct database_config*const          io_config)
{
  unsigned                              l_slot;

  if ((*io_config).m_aspect)
  {
    for (l_slot= 0; (*io_config).m_count > l_slot; l_slot++)
    {
      database_aspect_discharge(&(*io_config).m_aspect[l_slot]);
    }
    g_free((*io_config).m_aspect);
  }

  memset(io_config, 0, sizeof(*io_config));

  return;
}

extern void
database_config_load(
  struct database_config*const          o_config)
{
  gboolean                              l_bool;
  char const*                           l_configfile;
  gsize                                 l_group_count;
  gchar**                               l_groups;
  struct database_aspect*               l_aspect;
  GKeyFile*                             l_key;
  struct password*                      l_pwd;
  gsize                                 l_slot;
  char*                                 l_value;

  l_configfile= get_database_config_filename();
  l_key= 0;
  l_groups= 0;

  do
  {

    if ((*o_config).m_aspect)
    {
      g_free((*o_config).m_aspect);
      (*o_config).m_aspect= 0;
    }

    (*o_config).m_count= 0;

    l_key= g_key_file_new();
    l_groups= 0;

    l_bool= g_key_file_load_from_file(l_key, l_configfile, G_KEY_FILE_NONE,0);

    if (0 == l_bool)
    {
      break;
    }

    l_groups= g_key_file_get_groups(l_key, &l_group_count);      

    if (0 == l_group_count)
    {
      break;
    }

    for(l_slot= 0; l_slot < l_group_count; l_slot++)
    {

      (*o_config).m_aspect= (struct database_aspect*)
        realloc(
          (*o_config).m_aspect, 
          (1+(*o_config).m_count)*sizeof(struct database_aspect));

      l_aspect= &(*o_config).m_aspect[(*o_config).m_count];
      database_aspect_assign(l_aspect);
      (*o_config).m_count++;

      strncpy((*l_aspect).m_tag, l_groups[l_slot], sizeof((*l_aspect).m_tag));

      l_value= g_key_file_get_value(l_key, l_groups[l_slot], "database", 0);
      if (l_value)
      {
        strncpy((*l_aspect).m_dbname, l_value, sizeof((*l_aspect).m_dbname));
        g_free(l_value);
      }

      l_value= g_key_file_get_value(l_key, l_groups[l_slot], "driver", 0);
      if (l_value)
      {
        strncpy((*l_aspect).m_driver, l_value, sizeof((*l_aspect).m_driver));
        g_free(l_value);
      }

      l_value= g_key_file_get_value(l_key, l_groups[l_slot], "host", 0);
      if (l_value)
      {
        strncpy((*l_aspect).m_host, l_value, sizeof((*l_aspect).m_host));
        g_free(l_value);
      }

      l_value= g_key_file_get_value(l_key, l_groups[l_slot], "hostaddr", 0);
      if (l_value)
      {
        strncpy((*l_aspect).m_hostaddr, l_value, sizeof((*l_aspect).m_hostaddr));
        g_free(l_value);
      }

      l_value= g_key_file_get_value(l_key, l_groups[l_slot], "port", 0);
      if (l_value)
      {
        strncpy((*l_aspect).m_port, l_value, sizeof((*l_aspect).m_port));
        g_free(l_value);
      }

      l_value= g_key_file_get_value(l_key, l_groups[l_slot], "username", 0);
      if (l_value)
      {
        strncpy((*l_aspect).m_username, l_value, sizeof((*l_aspect).m_username));
        g_free(l_value);
      }

      (*l_aspect).m_prompt_password= g_key_file_get_boolean(l_key, l_groups[l_slot], "prompt_password", 0);

      l_pwd= wallet_lookup((*l_aspect).m_tag);

      if (0 == l_pwd)
      {
        l_pwd= wallet_lookup("");
      }

      if (l_pwd)
      {
        (*l_aspect).m_password= g_strdup((*l_pwd).m_password);
      }

    }

  }while(0);

  if (l_groups)
  {
    g_strfreev(l_groups);
  }

  if (l_key)
  {
    g_key_file_free(l_key);
  }

  return;
}

extern void
database_config_delete_group(
  struct database_config*const          io_config,
  unsigned const                        i_slot)
{
  struct database_aspect const*         l_aspect;
  gboolean                              l_bool;
  char const*                           l_configfile;
  GKeyFile*                             l_key;

  l_configfile= get_database_config_filename();
  l_key= 0;

  do
  {

    if (0 == (*io_config).m_count)
    {
      break;
    }

    l_aspect= &(*io_config).m_aspect[i_slot];

    l_key= g_key_file_new();

    l_bool= g_key_file_load_from_file(l_key, l_configfile, G_KEY_FILE_NONE,0);

    if (0 == l_bool)
    {
      break;
    }

    l_bool= g_key_file_remove_group(l_key, (*l_aspect).m_tag, 0);

    l_bool= g_key_file_save_to_file(l_key, l_configfile, 0);

  }while(0);

  if (l_key)
  {
    g_key_file_free(l_key);
  }

  database_config_discharge(io_config);
  database_config_assign(io_config);
  database_config_load(io_config);

  return;
}

extern void
database_config_populate_tag(
  char                                  o_tag[31+1],
  struct database_config const*const    i_config)
{
  int                                   l_rc;
  unsigned                              l_slot;
  char                                  l_tag[31+1];
  unsigned                              l_value;

  memset(o_tag, 0, 32);
  g_strlcpy(l_tag, DATABASE_CONFIG_DEFAULT_TAG, sizeof(l_tag));
  l_value= 0;

  do
  {

    l_rc= -1;

    for (l_slot= 0; (*i_config).m_count > l_slot; l_slot++)
    {
      l_rc= strcmp(l_tag, (*i_config).m_aspect[l_slot].m_tag);

      if (0 == l_rc)
      {
        break;
      }
    }

    if (l_rc)
    {
      memcpy(o_tag, l_tag, sizeof(l_tag));
      break;
    }

    l_value++;
    snprintf(l_tag, sizeof(l_tag), "%s%u", DATABASE_CONFIG_DEFAULT_TAG, l_value);

  }while(1);

  return;
}

extern void
database_config_save_aspect(
  struct database_aspect const*const    i_aspect)
{
  gboolean                              l_bool;
  char const*                           l_configfile;
  char const*                           l_tag;
  GKeyFile*                             l_key;

  l_configfile= get_database_config_filename();
  l_key= 0;

  do
  {

    if ((*i_aspect).m_tag[0])
    {
      l_tag= (*i_aspect).m_tag;
    }
    else
    {
      l_tag= DATABASE_CONFIG_DEFAULT_TAG;
    }

    l_key= g_key_file_new();

    l_bool= g_key_file_load_from_file(l_key, l_configfile, G_KEY_FILE_NONE,0);

    if (0 == l_bool)
    {
      break;
    }

    g_key_file_set_value(
      l_key,
      l_tag,
      "driver",
      (*i_aspect).m_driver);

    g_key_file_set_value(
      l_key,
      l_tag,
      "database",
      (*i_aspect).m_dbname);

    g_key_file_set_value(
      l_key,
      l_tag,
      "host",
      (*i_aspect).m_host);

    g_key_file_set_value(
      l_key,
      l_tag,
      "hostaddr",
      (*i_aspect).m_hostaddr);

    g_key_file_set_value(
      l_key,
      l_tag,
      "port",
      (*i_aspect).m_port);

    g_key_file_set_value(
      l_key,
      l_tag,
      "username",
      (*i_aspect).m_username);

    g_key_file_set_boolean(
      l_key,
      l_tag,
      "prompt_password",
      (*i_aspect).m_prompt_password);

    l_bool= g_key_file_save_to_file(l_key, l_configfile, 0);

  }while(0);

  if (l_key)
  {
    g_key_file_free(l_key);
  }

  return;
}
