/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "database_session.h"
#include "wallet.h"

enum
{
  RESPONSE_TEST=1,
  RESPONSE_NEW=2,
  RESPONSE_DELETE=3,
  RESPONSE_SAVE=4
};

struct database_dialog
{
  GtkEntry*                             m_tag_entry;
  GtkComboBoxText*                      m_tag_comboboxtext;
  GtkComboBoxText*                      m_driver_comboboxtext;
  GtkEntry*                             m_database_entry;
  GtkButton*                            m_database_button;
  GtkEntry*                             m_hostname_entry;
  GtkEntry*                             m_ipaddress_entry;
  GtkEntry*                             m_port_entry;
  GtkEntry*                             m_username_entry;
  GtkEntry*                             m_password_entry;
  GtkCheckButton*                       m_showtext_checkbutton;
  GtkCheckButton*                       m_prompt_checkbutton;
  GtkButton*                            m_test_button;
  GtkButton*                            m_new_button;
  GtkButton*                            m_save_button;
  GtkButton*                            m_delete_button;
  struct driver_config                  m_driver;
  struct database_config                m_config;
  int                                   m_active;
  guint                                 m_tag_signal_handle;
};

static void
database_property_copy_values(
  struct database_aspect *const         io_object,
  struct database_dialog const*const    i_dlg);

static void
tag_comboboxtext_cb(
  GtkWidget *                           i_widget,
  gpointer                              i_data);

struct driver_module const*
database_get_driver_module(
  struct database_dialog const*const    i_dlg)
{
  gint                                  l_active;
  struct driver_module const*           l_module;
  gchar const*                          l_text;

  l_module= 0;
  
  do
  {

    l_active= gtk_combo_box_get_active(GTK_COMBO_BOX((*i_dlg).m_driver_comboboxtext));

    if (-1 == l_active)
    {
      break;
    }

    l_text= gtk_combo_box_text_get_active_text((*i_dlg).m_driver_comboboxtext);

    if (0 == l_text)
    {
      break;
    }

    l_module= driver_config_lookup(&(*i_dlg).m_driver, l_text);

  }while(0);

  return l_module;
}

static void
database_clicked_button_cb(
  GtkButton*                            i_button,
  gpointer                              i_data)
{
  struct database_aspect                l_aspect;
  struct database_dialog*               l_dlg;
  char*                                 l_database;
  int                                   l_exit;

  database_aspect_assign(&l_aspect);
  l_database= 0;
  l_dlg= (struct database_dialog*)i_data;

  do
  {

    database_property_copy_values(&l_aspect, l_dlg);
    l_exit= database_aspect_database_property(&l_database, &l_aspect);

    if (l_aspect.m_password)
    {
      gtk_entry_set_text((*l_dlg).m_password_entry, l_aspect.m_password);
    }

    if (l_exit)
    {
      break;
    }

    gtk_entry_set_text((*l_dlg).m_database_entry, l_database);

  }while(0);

  database_aspect_discharge(&l_aspect);
  g_free(l_database);

  return;
}

static void
driver_comboboxtext_cb(
  GtkWidget *                           i_widget,
  gpointer                              i_data)
{
  struct database_dialog *              l_dlg;
  struct driver_module const*           l_module;
  gint                                  l_sensitive;

  l_dlg= (struct database_dialog*)i_data;
  l_sensitive= 0;
  
  do
  {

    l_module= database_get_driver_module(l_dlg);

    if (0 == l_module)
    {
      break;
    }

    l_sensitive= (*l_module).m_flags & 
      (DRIVER_DATABASE_IS_FILE|DRIVER_HAS_SHOW_DATABASE);

  }while(0);

  gtk_widget_set_sensitive(GTK_WIDGET((*l_dlg).m_database_button), l_sensitive);

  return;
}

static void
showtext_toggled_checkbutton_cb(
  GtkToggleButton *                     i_toggle,
  gpointer                              i_data)
{
  gboolean                              l_active;
  struct database_dialog*               l_dlg;

  l_dlg= (struct database_dialog*)i_data;
  l_active= gtk_toggle_button_get_active(i_toggle);
  gtk_entry_set_visibility((*l_dlg).m_password_entry, l_active);

  return;
}

static void
database_property_bind(
  struct database_dialog*const          o_dlg,
  GtkBuilder*const                      i_builder)
{

  do
  {

    (*o_dlg).m_tag_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "tag_entry"));

    (*o_dlg).m_tag_comboboxtext= GTK_COMBO_BOX_TEXT(
      gtk_builder_get_object(i_builder, "tag_comboboxtext"));

    (*o_dlg).m_tag_signal_handle= g_signal_connect(
      (*o_dlg).m_tag_comboboxtext,
      "changed",
      G_CALLBACK(tag_comboboxtext_cb),
      o_dlg);

    (*o_dlg).m_driver_comboboxtext= GTK_COMBO_BOX_TEXT(
      gtk_builder_get_object(i_builder, "driver_comboboxtext"));

    g_signal_connect(
      (*o_dlg).m_driver_comboboxtext,
      "changed",
      G_CALLBACK(driver_comboboxtext_cb),
      o_dlg);

    (*o_dlg).m_database_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "database_entry"));

    (*o_dlg).m_database_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "database_button"));

    gtk_widget_set_sensitive(GTK_WIDGET((*o_dlg).m_database_button), 0);

    g_signal_connect(
      (*o_dlg).m_database_button,
      "clicked",
      G_CALLBACK(database_clicked_button_cb),
      o_dlg);

    (*o_dlg).m_hostname_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "hostname_entry"));

    (*o_dlg).m_ipaddress_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "ip_entry"));

    (*o_dlg).m_port_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "port_entry"));

    (*o_dlg).m_username_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "username_entry"));

    (*o_dlg).m_password_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "password_entry"));

    gtk_entry_set_visibility((*o_dlg).m_password_entry, 0);

    (*o_dlg).m_showtext_checkbutton= GTK_CHECK_BUTTON(
      gtk_builder_get_object(i_builder, "showtext_checkbutton"));

    g_signal_connect(
      (*o_dlg).m_showtext_checkbutton,
      "toggled",
      G_CALLBACK(showtext_toggled_checkbutton_cb),
      o_dlg);

    (*o_dlg).m_prompt_checkbutton= GTK_CHECK_BUTTON(
      gtk_builder_get_object(i_builder, "promptpassword_checkbutton"));

    (*o_dlg).m_test_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "test_button"));

    (*o_dlg).m_new_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "new_button"));

    (*o_dlg).m_save_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "save_button"));

    (*o_dlg).m_delete_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "delete_button"));

  }while(0);

  return;
}

static void
database_property_copy_values(
  struct database_aspect *const         io_object,
  struct database_dialog const*const    i_dlg)
{
  gchar const*                          l_text;

  memset(io_object, 0, sizeof(*io_object));

  l_text= gtk_entry_get_text((*i_dlg).m_tag_entry);
  if (l_text && l_text[0])
  {
    strncpy((*io_object).m_tag, l_text, (sizeof((*io_object).m_tag)-1));
  }

  l_text= gtk_combo_box_text_get_active_text((*i_dlg).m_driver_comboboxtext);
  if (l_text)
  {
    strncpy((*io_object).m_driver, l_text, (sizeof((*io_object).m_driver)-1));
  }

  l_text= gtk_entry_get_text((*i_dlg).m_database_entry);
  if (l_text)
  {
    strncpy((*io_object).m_dbname, l_text, (sizeof((*io_object).m_dbname)-1));
  }

  l_text= gtk_entry_get_text((*i_dlg).m_hostname_entry);
  if (l_text)
  {
    strncpy((*io_object).m_host, l_text, (sizeof((*io_object).m_host)-1));
  }

  l_text= gtk_entry_get_text((*i_dlg).m_ipaddress_entry);
  if (l_text)
  {
    strncpy((*io_object).m_hostaddr, l_text, (sizeof((*io_object).m_hostaddr)-1));
  }

  l_text= gtk_entry_get_text((*i_dlg).m_port_entry);
  if (l_text)
  {
    strncpy((*io_object).m_port, l_text, (sizeof((*io_object).m_port)-1));
  }

  l_text= gtk_entry_get_text((*i_dlg).m_username_entry);
  if (l_text)
  {
    strncpy((*io_object).m_username, l_text, (sizeof((*io_object).m_username)-1));
  }

  g_free((*io_object).m_password);
  (*io_object).m_password= 0;
  l_text= gtk_entry_get_text((*i_dlg).m_password_entry);
  if (l_text && l_text[0])
  {
    (*io_object).m_password= g_strdup(l_text);
  }

  (*io_object).m_prompt_password=
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
      (*i_dlg).m_prompt_checkbutton));

  return;
}

static void
database_property_clear_values(
  struct database_dialog*const          io_dlg)
{
  GtkEntryBuffer*                       l_ebuf;

  g_signal_handler_block(
    (*io_dlg).m_tag_comboboxtext,
    (*io_dlg).m_tag_signal_handle);

  l_ebuf= gtk_entry_get_buffer((*io_dlg).m_tag_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  gtk_combo_box_set_active(GTK_COMBO_BOX((*io_dlg).m_tag_comboboxtext), -1);
  gtk_combo_box_set_active(GTK_COMBO_BOX((*io_dlg).m_driver_comboboxtext), -1);
  l_ebuf= gtk_entry_get_buffer((*io_dlg).m_database_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  l_ebuf= gtk_entry_get_buffer((*io_dlg).m_hostname_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  l_ebuf= gtk_entry_get_buffer((*io_dlg).m_ipaddress_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  l_ebuf= gtk_entry_get_buffer((*io_dlg).m_port_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  l_ebuf= gtk_entry_get_buffer((*io_dlg).m_username_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  l_ebuf= gtk_entry_get_buffer((*io_dlg).m_password_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON((*io_dlg).m_prompt_checkbutton),
    1);

  g_signal_handler_unblock(
    (*io_dlg).m_tag_comboboxtext,
    (*io_dlg).m_tag_signal_handle);

  return;
}

static void
database_property_tag_load(
  struct database_dialog *const         io_dlg)
{
  int                                   l_slot;

  g_signal_handler_block(
    (*io_dlg).m_tag_comboboxtext,
    (*io_dlg).m_tag_signal_handle);

  gtk_combo_box_text_remove_all((*io_dlg).m_tag_comboboxtext);

  for(l_slot= 0; (*io_dlg).m_config.m_count > l_slot; l_slot++)
  {
    gtk_combo_box_text_append_text(
      (*io_dlg).m_tag_comboboxtext,
      (*io_dlg).m_config.m_aspect[l_slot].m_tag);
  }

  gtk_combo_box_set_active(
    GTK_COMBO_BOX((*io_dlg).m_tag_comboboxtext),
    (*io_dlg).m_active);

  g_signal_handler_unblock(
    (*io_dlg).m_tag_comboboxtext,
    (*io_dlg).m_tag_signal_handle);

  return;
}

static void
database_property_delete(
  struct database_dialog *const         io_dlg)
{

  do
  {

    if (-1 == (*io_dlg).m_active)
    {
      break;
    }

    database_config_delete_group(&(*io_dlg).m_config, (*io_dlg).m_active);
    (*io_dlg).m_active= -1;
    database_property_tag_load(io_dlg);

  }while(0);

  database_property_clear_values(io_dlg);

  return;
}

static void
database_property_set_values(
  struct database_dialog*const          io_dlg,
  struct database_aspect const*const    i_object)
{
  int                                   l_rc;
  unsigned                              l_slot;

  do
  {

    gtk_entry_set_text((*io_dlg).m_tag_entry, (*i_object).m_tag);

    gtk_combo_box_set_active(
      GTK_COMBO_BOX((*io_dlg).m_tag_comboboxtext),
      (*io_dlg).m_active);

    gtk_combo_box_set_active(
      GTK_COMBO_BOX((*io_dlg).m_driver_comboboxtext),
      -1);

    for(l_slot= 0; (*io_dlg).m_driver.m_count > l_slot; l_slot++)
    {
      l_rc= strcmp(
        (*io_dlg).m_driver.m_module[l_slot].m_tag,
        (*i_object).m_driver);

      if (0 == l_rc)
      {
        gtk_combo_box_set_active(
          GTK_COMBO_BOX((*io_dlg).m_driver_comboboxtext),
          l_slot);
      }
    }

    gtk_entry_set_text((*io_dlg).m_database_entry, (*i_object).m_dbname);
    gtk_entry_set_text((*io_dlg).m_hostname_entry, (*i_object).m_host);
    gtk_entry_set_text((*io_dlg).m_ipaddress_entry, (*i_object).m_hostaddr);
    gtk_entry_set_text((*io_dlg).m_port_entry, (*i_object).m_port);
    gtk_entry_set_text((*io_dlg).m_username_entry, (*i_object).m_username);

    if ((*i_object).m_password)
    {
      gtk_entry_set_text((*io_dlg).m_password_entry, (*i_object).m_password);
    }
    else
    {
      gtk_entry_set_text((*io_dlg).m_password_entry, "");
    }
    
    gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON((*io_dlg).m_prompt_checkbutton),
      (*i_object).m_prompt_password);

  }while(0);

  return;
}

static void
database_property_preflight(
  struct database_dialog*const          io_dlg)
{
  unsigned                              l_slot;

  database_property_tag_load(io_dlg);

  gtk_combo_box_text_remove_all((*io_dlg).m_driver_comboboxtext);

  for(l_slot= 0; (*io_dlg).m_driver.m_count > l_slot; l_slot++)
  {
    gtk_combo_box_text_append_text(
      (*io_dlg).m_driver_comboboxtext,
      (*io_dlg).m_driver.m_module[l_slot].m_tag);
  }

  return;
}

static void
database_property_save(
  struct database_dialog*const          io_dlg)
{
  struct database_aspect*               l_aspect;
  struct password*                      l_pwd;

  do
  {

    if (-1 != (*io_dlg).m_active)
    {
      l_aspect= &(*io_dlg).m_config.m_aspect[(*io_dlg).m_active];
      database_property_copy_values(l_aspect, io_dlg);
      database_config_save_aspect(l_aspect);
      break;
    }

    (*io_dlg).m_config.m_aspect= 
      realloc(
        (*io_dlg).m_config.m_aspect,
        ((1+(*io_dlg).m_config.m_count) * sizeof(struct database_aspect)));

    (*io_dlg).m_active= (*io_dlg).m_config.m_count;
    l_aspect= &(*io_dlg).m_config.m_aspect[(*io_dlg).m_config.m_count];
    (*io_dlg).m_config.m_count++;

    database_property_copy_values(l_aspect, io_dlg);

    if (0 == (*l_aspect).m_tag[0])
    {
      database_config_populate_tag((*l_aspect).m_tag, &(*io_dlg).m_config);
    }

    database_config_save_aspect(l_aspect);
    database_property_tag_load(io_dlg);
    gtk_entry_set_text((*io_dlg).m_tag_entry, (*l_aspect).m_tag);

  }while(0);

  l_pwd= g_new0(struct password, 1);
  (*l_pwd).m_aspect= g_strdup((*l_aspect).m_tag);
  (*l_pwd).m_username= g_strdup((*l_aspect).m_username);
  (*l_pwd).m_password= g_strdup((*l_aspect).m_password);
  wallet_insert(l_pwd);

  return;
}

static void
database_property_test(
  struct database_dialog *const         io_dlg)
{
  struct database_aspect                l_aspect;
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_rc;
  struct database_session               l_session;

  l_error= 0;
  database_aspect_assign(&l_aspect);
  database_session_assign(&l_session);

  do
  {

    database_property_copy_values(&l_aspect, io_dlg);
    database_aspect_password_fill(&l_rc, &l_aspect);

    if (l_rc)
    {
      break;
    }

    l_rc= database_session_connect(&l_error, &l_session, &l_aspect);

    if (l_rc)
    {
      break;
    }

    if (l_aspect.m_password)
    { 
      gtk_entry_set_text((*io_dlg).m_password_entry, l_aspect.m_password);
    }

    l_dialog= GTK_DIALOG(gtk_message_dialog_new(
      get_main_window(),
      GTK_DIALOG_MODAL,
      GTK_MESSAGE_INFO,
      GTK_BUTTONS_OK,
      "Connection successfull"));
    gtk_dialog_run(l_dialog);
    gtk_widget_destroy(GTK_WIDGET(l_dialog));

  }while(0);

  if (l_error)
  {
    l_dialog= GTK_DIALOG(gtk_message_dialog_new(
      get_main_window(),
      GTK_DIALOG_MODAL,
      GTK_MESSAGE_ERROR,
      GTK_BUTTONS_CLOSE,
      (*l_error).message, 
      0));
    gtk_dialog_run(l_dialog);
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  g_clear_error(&l_error);

  database_aspect_discharge(&l_aspect);
  database_session_discharge(&l_session);

  return;
}

static void
tag_comboboxtext_cb(
  GtkWidget *                           i_widget,
  gpointer                              i_data)
{
  struct database_dialog *              l_dlg;
  struct database_aspect const*         l_aspect;

  l_dlg= (struct database_dialog*)i_data;
  (*l_dlg).m_active= gtk_combo_box_get_active(GTK_COMBO_BOX(i_widget));
  l_aspect= &(*l_dlg).m_config.m_aspect[(*l_dlg).m_active];
  database_property_set_values(l_dlg, l_aspect);

  return;
}

extern int
database_aspect_property(
  struct database_aspect *const         io_object)
{
  GtkBuilder*                           l_builder;
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_exit;
  struct database_dialog                l_dlg;
  int                                   l_rc;
  unsigned                              l_slot;

  l_builder= 0;
  l_dialog= 0;
  l_error= 0;
  l_exit= 0;

  memset(&l_dlg, 0, sizeof(l_dlg));
  l_dlg.m_active= -1;
  database_config_assign(&l_dlg.m_config);
  driver_config_assign(&l_dlg.m_driver);

  do
  {

    l_builder= gtk_builder_new();

    l_rc= wrap_gtk_builder_add_from_file(l_builder, "database.glade", &l_error);

    if (0 == l_rc)
    {
      l_exit= -1;
      break;
    }
  
    l_dialog= (GtkDialog*)gtk_builder_get_object(l_builder, "database_dialog");

    if (0 == l_dialog)
    {
      l_error= g_error_new(
        GENERAL,
        GENERIC,
        "Unable to find dialog object: 'database_dialog'");
      l_exit= -1;
      break;
    }

    gtk_window_set_transient_for(GTK_WINDOW(l_dialog), get_main_window());

    if (0 == (*io_object).m_tag[0])
    {
      g_strlcpy((*io_object).m_tag, DATABASE_CONFIG_DEFAULT_TAG, sizeof((*io_object).m_tag));
    }

    database_config_load(&l_dlg.m_config);

    if (0 == (*io_object).m_tag[0])
    {
      database_config_populate_tag((*io_object).m_tag, &l_dlg.m_config);
    }

    for (l_slot= 0; l_dlg.m_config.m_count > l_slot; l_slot++)
    {
      l_rc= strcmp(l_dlg.m_config.m_aspect[l_slot].m_tag, (*io_object).m_tag);
      if (0 == l_rc)
      {
        database_aspect_copy(&l_dlg.m_config.m_aspect[l_slot], io_object);
        l_dlg.m_active= l_slot;
        break;
      }
    }

    if (-1 == l_dlg.m_active)
    {
      l_dlg.m_config.m_aspect= 
        realloc(
          l_dlg.m_config.m_aspect,
          ((1+l_dlg.m_config.m_count) * sizeof(struct database_aspect)));
      database_aspect_copy(
        &l_dlg.m_config.m_aspect[l_dlg.m_config.m_count],
        io_object);
      l_dlg.m_active= l_dlg.m_config.m_count;
      l_dlg.m_config.m_count++;
    }

    l_exit= driver_config_load(&l_error, &l_dlg.m_driver);

    if (l_exit)
    {
      _error_display_prompt(l_error);
      g_clear_error(&l_error);
      l_error= 0;
      l_exit= 0; 
    }

    database_property_bind(&l_dlg, l_builder);
    database_property_preflight(&l_dlg);

    if (-1 != l_dlg.m_active)
    {
      database_property_set_values(
        &l_dlg,
        &l_dlg.m_config.m_aspect[l_dlg.m_active]);
    }

    gtk_widget_show_all(GTK_WIDGET(l_dialog));
    gtk_window_set_modal(GTK_WINDOW(l_dialog), 1);

    do
    {

      l_exit= gtk_dialog_run(GTK_DIALOG(l_dialog));

      if (GTK_RESPONSE_DELETE_EVENT == l_exit)
      {
        break;
      }

      if (GTK_RESPONSE_CANCEL == l_exit)
      {
        break;
      }

      if (GTK_RESPONSE_OK == l_exit) 
      {
        database_property_save(&l_dlg);
        database_property_copy_values(io_object, &l_dlg);
        break;
      }

      switch(l_exit)
      {
        case RESPONSE_TEST:
          database_property_test(&l_dlg);
          break;
        case RESPONSE_NEW:
          l_dlg.m_active= -1;
          database_property_clear_values(&l_dlg);
          break;
        case RESPONSE_DELETE:
          database_property_delete(&l_dlg);
          break;
        case RESPONSE_SAVE:
          database_property_save(&l_dlg);
          break;
        default:
          break;
      }

    }while(1);

  }while(0);

  driver_config_discharge(&l_dlg.m_driver);
  database_config_discharge(&l_dlg.m_config);

  if (l_dialog)
  {
    gtk_widget_destroy((GtkWidget*)l_dialog);
  }

  if (l_builder)
  {
    g_object_unref(l_builder);
  }

  if (l_error)
  {
    _error_log(l_error);
    _error_display_prompt(l_error);
  }

  g_clear_error(&l_error);

  return l_exit;
}
