/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "database_session.h"
#include "wallet.h"

struct database_dialog
{
  GtkListBox*                           m_database_listbox;
};

static void
database_bind(
  struct database_dialog*const          o_dlg,
  GtkBuilder*const                      i_builder)
{

  do
  {

    (*o_dlg).m_database_listbox= GTK_LIST_BOX(
      gtk_builder_get_object(i_builder, "database_listbox"));

  }while(0);

  return;
}

static void
database_set_values(
  struct database_dialog*const          io_dlg,
  char *const*                          i_list,
  unsigned const                        i_list_slots)
{
  unsigned                              l_slot;
  GtkLabel*                             l_label;

  for (l_slot= 0; i_list_slots > l_slot; l_slot++)
  {
    l_label= GTK_LABEL(gtk_label_new(i_list[l_slot]));
    gtk_list_box_insert((*io_dlg).m_database_listbox, GTK_WIDGET(l_label), -1);
  }

  gtk_widget_show_all(GTK_WIDGET((*io_dlg).m_database_listbox));
 
  return;
}

static int
database_choose_list(
  char**                                o_database,
  char *const*                          i_list,
  unsigned const                        i_list_slots)
{
  GtkBuilder*                           l_builder;
  GtkDialog*                            l_dialog;
  struct database_dialog                l_dlg;
  GError*                               l_error;
  int                                   l_exit;
  gint                                  l_index;
  int                                   l_rc;
  GtkListBoxRow*                        l_row;

  (*o_database)= 0;
  l_builder= 0;
  l_dialog= 0;
  l_error= 0;
  l_exit= 0;

  memset(&l_dlg, 0, sizeof(l_dlg));

  do
  {

    l_builder= gtk_builder_new();

    l_rc= wrap_gtk_builder_add_from_file(l_builder, "database.glade", &l_error);

    if (0 == l_rc)
    {
      l_exit= -1;
      break;
    }
  
    l_dialog= (GtkDialog*)gtk_builder_get_object(l_builder, "list_dialog");

    if (0 == l_dialog)
    {
      l_error= g_error_new(
        GENERAL,
        GENERIC,
        "Unable to find dialog object: 'list_dialog'");
      l_exit= -1;
      break;
    }

    gtk_window_set_transient_for(GTK_WINDOW(l_dialog), get_main_window());
    gtk_widget_show_all(GTK_WIDGET(l_dialog));
    gtk_window_set_modal(GTK_WINDOW(l_dialog), 1);

    gtk_widget_grab_focus(gtk_dialog_get_widget_for_response(
      GTK_DIALOG(l_dialog), GTK_RESPONSE_OK));

    database_bind(&l_dlg, l_builder);
    database_set_values(&l_dlg, i_list, i_list_slots);

    l_exit= gtk_dialog_run(l_dialog);

    if (GTK_RESPONSE_OK != l_exit) 
    {
      break;
    }

    l_row= gtk_list_box_get_selected_row(l_dlg.m_database_listbox);

    if (0 == l_row)
    {
      l_exit= -1;
      break;
    }

    l_index= gtk_list_box_row_get_index(l_row);

    (*o_database)= g_strdup(i_list[l_index]);

    l_exit= 0;

  }while(0);

  if (l_dialog)
  {
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  if (l_builder)
  {
    g_object_unref(l_builder);
  }

  if (l_error)
  {
    _error_log(l_error);
    _error_display_prompt(l_error);
    g_clear_error(&l_error);
  }

  return l_exit;
}

static int
database_choose_query(
  char**                                o_database,
  struct database_aspect *const         io_aspect,
  struct driver_module const*const      i_module)
{
  GError*                               l_error;
  int                                   l_exit;
  char**                                l_list;
  unsigned                              l_list_slots;
  struct database_session               l_session;
  unsigned                              l_slot;

  (*o_database)= 0;
  l_error= 0;
  l_exit= 0;
  l_list= 0;
  l_list_slots= 0;
  database_session_assign(&l_session);

  do
  {

    database_aspect_password_fill(&l_exit, io_aspect);

    if (l_exit)
    {
      break;
    }

    l_exit= database_session_connect_module(
      &l_error,
      &l_session,
      i_module,
      io_aspect);

    if (l_exit)
    {
      break;
    }

    l_exit= database_session_query_show_databases(
      &l_error,
      &l_list,
      &l_list_slots,
      &l_session);

    if (l_exit)
    {
      break;
    }

    l_exit= database_choose_list(o_database, l_list, l_list_slots);

  }while(0);

  for (l_slot= 0; l_list_slots > l_slot; l_slot++)
  {
    g_free(l_list[l_slot]);
  }

  g_free(l_list);

  database_session_discharge(&l_session);

  if (l_error)
  {
    _error_display_prompt(l_error);
    g_clear_error(&l_error);
  }

  return l_exit;
}

static int
database_choose_file(
  char**                                o_database)
{
  GtkWidget*                            l_dialog;
  int                                   l_exit;
  char*                                 l_filename;
  int                                   l_rc;

  (*o_database)= 0;
  l_dialog= 0;
  l_exit= 0;

  do
  {

    l_dialog = gtk_file_chooser_dialog_new(
      "Open Database",
      get_main_window(),
      GTK_FILE_CHOOSER_ACTION_OPEN,
      "_Cancel",
      GTK_RESPONSE_CANCEL,
      "_Save",
      GTK_RESPONSE_ACCEPT,
      NULL);

    l_rc= gtk_dialog_run(GTK_DIALOG(l_dialog));

    if (GTK_RESPONSE_ACCEPT != l_rc)
    {
      l_exit= -1;
      break;
    }

    l_filename= gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(l_dialog));
    (*o_database)= g_strdup(l_filename);

  }while(0);

  if (l_dialog)
  {
    gtk_widget_destroy(l_dialog); 
  }

  return l_exit;
}

extern int
database_aspect_database_property(
  char**                                o_database,
  struct database_aspect *const         io_object)
{
  struct driver_config                  l_driver;
  GError*                               l_error;
  int                                   l_exit;
  struct driver_module const*           l_module;

  (*o_database)= 0;
  l_error= 0;
  l_exit= 0;
  driver_config_assign(&l_driver);

  do
  {

    if (0 == (*io_object).m_driver[0])
    {
      l_error= g_error_new(
        DATABASE_ASPECT,
        GLOBAL_DRIVER_CONFIG_NOTFOUND,
        "Missing driver name");
      l_exit= -1;
      break;
    }

    l_exit= driver_config_load(&l_error, &l_driver);

    if (l_exit)
    {
      break;
    }

    l_module= driver_config_lookup(&l_driver, (*io_object).m_driver);

    if (0 == l_module)
    {
      l_error= g_error_new(
        DATABASE_ASPECT,
        GLOBAL_DRIVER_CONFIG_NOTFOUND,
        "Driver not found: '%s'",
        (*io_object).m_driver);
      l_exit= -1;
      break;

    }
    
    if (DRIVER_DATABASE_IS_FILE & (*l_module).m_flags)
    {
      l_exit= database_choose_file(o_database);
      break;
    }

    l_exit= database_choose_query(o_database, io_object, l_module);

  }while(0);

  if (l_error)
  {
    _error_log(l_error);
    _error_display_prompt(l_error);
    g_clear_error(&l_error);
  }

  driver_config_discharge(&l_driver);

  return l_exit;
}
