/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "database_aspect.h"
#include "item_common.h"

extern void
database_aspect_assign(
  struct database_aspect *const         o_aspect)
{

  memset(o_aspect, 0, sizeof(*o_aspect));

  return;
}

extern void
database_aspect_copy(
  struct database_aspect *const         o_dst,
  struct database_aspect const*const    i_src)
{

  memcpy(o_dst, i_src, sizeof(*o_dst));
  (*o_dst).m_password= g_strdup((*i_src).m_password);

  return;
}

extern void
database_aspect_discharge(
  struct database_aspect *const         io_aspect)
{

  g_free((*io_aspect).m_password);
  memset(io_aspect, 0, sizeof(*io_aspect));

  return;
}

extern void
database_aspect_password_fill(
  int*const                             o_cancel,
  struct database_aspect *const         io_aspect)
{

  (*o_cancel)= 0;

  do
  {

    if (0 == (*io_aspect).m_prompt_password)
    {
      break;
    }

    if (((*io_aspect).m_password) && (*io_aspect).m_password[0])
    {
      break;
    }

    if ((*io_aspect).m_password)
    {
      g_free((*io_aspect).m_password);
      (*io_aspect).m_password= 0;
    }

    database_aspect_password_prompt(o_cancel, &(*io_aspect).m_password, (*io_aspect).m_tag);

  }while(0);

  return;
}

extern void
database_aspect_password_prompt(
  int*const                             o_cancel,
  char**                                o_password,
  char const*                           i_tag)
{
  GtkWidget*                            l_content;
  GtkDialog*                            l_dialog;
  GtkEntry*                             l_entry;
  int                                   l_rc;
  char const*                           l_text;

  (*o_cancel)= 0;
  (*o_password)= 0;

  l_dialog= GTK_DIALOG(gtk_message_dialog_new(
    get_main_window(),
    GTK_DIALOG_MODAL,
    GTK_MESSAGE_QUESTION,
    GTK_BUTTONS_OK_CANCEL,
    "Password for '%s'",
    i_tag));

  gtk_dialog_set_default_response(l_dialog, GTK_RESPONSE_OK);
  gtk_widget_grab_focus(gtk_dialog_get_widget_for_response(
      GTK_DIALOG(l_dialog), GTK_RESPONSE_OK));
  gtk_window_set_title(GTK_WINDOW(l_dialog), "Database password");  

  l_entry= GTK_ENTRY(gtk_entry_new());
  gtk_entry_set_visibility(l_entry, 0);
  gtk_entry_set_invisible_char(l_entry, '*');

  l_content= gtk_dialog_get_content_area(l_dialog);
  gtk_box_pack_end(GTK_BOX(l_content), GTK_WIDGET(l_entry), 1, 1, 0);
    
  gtk_widget_show_all(GTK_WIDGET(l_dialog));

  l_rc= gtk_dialog_run(l_dialog);

  if (l_rc == GTK_RESPONSE_OK)
  {
    l_text= gtk_entry_get_text(l_entry);
    if (l_text && l_text[0])
    {
      (*o_password)= g_strdup(l_text);
    }
  }
  else  
  {
    (*o_cancel)= 1;
  }

  gtk_widget_destroy(GTK_WIDGET(l_dialog));

  return;
}

extern int
database_aspect_read(
  GError**                              o_error,
  struct database_aspect *const         io_aspect,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  GList*                                l_node;
  char const*                           l_value;
  struct xml *                          l_xml;
  int                                   l_found;
  enum element_tag_type                 l_type;

  l_exit= 0;
  l_error= 0;

  do
  {

    l_value= xml_lookup_attribute_value(i_xml, "name");

    if (l_value)
    {
      g_strlcpy((*io_aspect).m_tag, l_value, sizeof((*io_aspect).m_tag));
    }

    if (0 == (*io_aspect).m_tag[0])
    {
      l_error= g_error_new(
        DATABASE_ASPECT,
        DATABASE_APSECT_XML_MISSING_TAG,
        "Missing aspect name");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_node= (*i_xml).m_children;

    do
    {

      if (0 == l_node)
      {
        break;
      }

      l_xml= (struct xml*)(*l_node).data;

      item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
      g_strstrip((*l_xml).m_string->str);
      l_value= (*l_xml).m_string->str;

      if (l_found)
      {
        switch(l_type)
        {
          case element_tag_driver:
            g_strlcpy((*io_aspect).m_driver, l_value, sizeof((*io_aspect).m_driver));
            break;
          case element_tag_dbname:
            g_strlcpy((*io_aspect).m_dbname, l_value, sizeof((*io_aspect).m_dbname));
            break;
          case element_tag_host:
            g_strlcpy((*io_aspect).m_host, l_value, sizeof((*io_aspect).m_host));
            break;
          case element_tag_hostaddr:
            g_strlcpy((*io_aspect).m_hostaddr, l_value, sizeof((*io_aspect).m_hostaddr));
            break;
          case element_tag_port:
            g_strlcpy((*io_aspect).m_port, l_value, sizeof((*io_aspect).m_port));
            break;
          case element_tag_username:
            g_strlcpy((*io_aspect).m_username, l_value, sizeof((*io_aspect).m_username));
            break;
          case element_tag_prompt_password:
            l_value= xml_lookup_attribute_value(l_xml, "on");
            (*io_aspect).m_prompt_password= xml_convert_boolean(l_value);
            break;
          default:
            g_warning("Unknown element ('%s')\n", (*l_xml).m_tag);
            break;
        }
      }
      else
      {
        g_warning("Unknown element ('%s')\n", (*l_xml).m_tag);
      }

      l_node= (*l_node).next;

    }while(1);

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
database_aspect_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct database_aspect const*const    i_aspect)
{
  int                                   l_exit;

  l_exit= 0;

  fprintf(io_fp, "\t<aspect name=\"%s\">\n", (*i_aspect).m_tag);
  fprintf(io_fp, "\t\t<driver>%s</driver>\n", (*i_aspect).m_driver);
  fprintf(io_fp, "\t\t<dbname>%s</dbname>\n", (*i_aspect).m_dbname);
  fprintf(io_fp, "\t\t<host>%s</host>\n", (*i_aspect).m_host);
  fprintf(io_fp, "\t\t<hostaddr>%s</hostaddr>\n", (*i_aspect).m_hostaddr);
  fprintf(io_fp, "\t\t<port>%s</port>\n", (*i_aspect).m_port);
  fprintf(io_fp, "\t\t<username>%s</username>\n", (*i_aspect).m_username);
  fprintf(io_fp, "\t\t<prompt_password on=\"%s\"/>\n", ((*i_aspect).m_prompt_password ? "true" : "false"));
  fprintf(io_fp, "\t</aspect>\n");

  return l_exit;
}
