/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "menu.h"
#include "layout.h"

#define WINDOW_WIDTH                    612
#define WINDOW_HEIGHT                   640

extern void
mainw_size_allocate(
  GtkWidget*                            i_widget,
  GdkRectangle*                         i_alloc,
  gpointer                              i_user_data)
{
  struct mainw*                         l_mainw;
  GdkRectangle                          l_adjust;

  l_adjust= (*i_alloc);
  l_adjust.height-= 18; /* menu */

  l_mainw= (struct mainw*)i_user_data;

  switch((*l_mainw).m_report.m_style)
  {
    case style_layout:
      layout_size_allocate(&(*l_mainw).m_report.m_object.m_layout, &l_adjust);
      break;
    case style_label:
      label_size_allocate(&(*l_mainw).m_report.m_object.m_label, &l_adjust);
      break;
    default:
      /* programming error */
      break;
  }

  return;
}

extern void
mainw_size_allocate_hack(
  struct mainw*const                    io_mainw)
{
  GtkAllocation                         l_alloc;
  GtkRequisition                        l_min;
  GtkRequisition                        l_nat;

  gtk_widget_get_preferred_size(GTK_WIDGET((*io_mainw).m_grid), &l_min, &l_nat);
  gtk_widget_get_allocation(GTK_WIDGET((*io_mainw).m_window), &l_alloc);
  g_signal_emit_by_name((*io_mainw).m_window, "size-allocate", &l_alloc, 0);

  return;
}

extern void
mainw_assign(
  struct mainw*const                    o_mainw)
{

  memset(o_mainw, 0, sizeof(*o_mainw));
  copybuf_assign(&(*o_mainw).m_copybuf);
  cmd_assign(&(*o_mainw).m_cmd);
  report_assign(&(*o_mainw).m_report);

  return;
}

extern void
mainw_cmd_push(
  struct mainw*const                    io_mainw,
  IwrPage*                              io_page,
  enum cmd_type const                   i_type)
{

  cmd_push(&(*io_mainw).m_cmd, io_page, i_type);
  mainw_edit_set_menu_state(io_mainw);

  return;
}

extern void
mainw_cmd_push_create(
  struct mainw*const                    io_mainw)
{
  IwrPage *                             l_page;

  l_page= 0;

  if (style_label == (*io_mainw).m_report.m_style)
  {
    l_page= (*io_mainw).m_report.m_object.m_label.m_page; 
  }
  else if (style_layout == (*io_mainw).m_report.m_style) 
  {
    l_page= layout_get_focused_page(&(*io_mainw).m_report.m_object.m_layout);
  }

  if (l_page)
  {
    mainw_cmd_push(io_mainw, l_page, cmd_create);
  }

  return;
}

extern int
mainw_create(
  struct mainw*const                    io_mainw)
{
  GtkBuilder*                           l_builder;
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_rc;
  GtkWidget*                            l_widget;

  l_builder= 0;
  l_error= 0;
  l_exit= 0;

  report_discharge(&(*io_mainw).m_report);
  report_assign_layout(&(*io_mainw).m_report);

  do
  {

    l_builder= gtk_builder_new();

    l_rc= wrap_gtk_builder_add_from_file(
      l_builder,
      "mainwindow.glade",
      &l_error);

    if (0 == l_rc)
    {
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    (*io_mainw).m_window= GTK_WINDOW(
      gtk_builder_get_object(l_builder, "mainwindow"));

    g_main_window= (*io_mainw).m_window;
  
    if (0 == (*io_mainw).m_window)
    {
      g_warning("Unable to find Window: 'mainwindow'\n");
      l_exit= -1;
      break;
    }

    menu_bind(l_builder, io_mainw);
    menu_layout_set_defaults();

    (*io_mainw).m_grid= GTK_GRID(gtk_builder_get_object(l_builder, "grid1"));

    gtk_window_set_title(GTK_WINDOW((*io_mainw).m_window), "iwrite");

    g_signal_connect(
      (*io_mainw).m_window,
      "destroy", 
      G_CALLBACK(quit_menuitem_activate_cb), 
      io_mainw);

    (*io_mainw).m_size_allocate_handle= g_signal_connect(
      (*io_mainw).m_window,
      "size-allocate",
      G_CALLBACK(mainw_size_allocate),
      io_mainw);

    (*io_mainw).m_paned= GTK_PANED(gtk_builder_get_object(l_builder, "paned1"));

    l_widget= report_gui_new(&(*io_mainw).m_report);
    layout_add_default_sections(&(*io_mainw).m_report.m_object.m_layout);

    gtk_paned_add1((*io_mainw).m_paned, l_widget);

    gtk_window_resize((*io_mainw).m_window, WINDOW_WIDTH, WINDOW_HEIGHT);

    gtk_widget_show_all(GTK_WIDGET((*io_mainw).m_window));

  }while(0);

  if (l_builder)
  {
    g_object_unref(l_builder);
  }

  g_clear_error(&l_error);

  return l_exit;
}

static int
mainw_create_default_layout(
  struct mainw*const                    io_mainw)
{
  GtkAllocation                         l_alloc;
  GtkWidget*                            l_widget;

  report_discharge(&(*io_mainw).m_report);
  report_assign_layout(&(*io_mainw).m_report);
  menu_layout_set_defaults();

  if (0 == (*io_mainw).m_size_allocate_handle)
  {
    (*io_mainw).m_size_allocate_handle= g_signal_connect(
      (*io_mainw).m_window,
      "size-allocate",
      G_CALLBACK(mainw_size_allocate),
      io_mainw);
  }

  l_widget= report_gui_new(&(*io_mainw).m_report);
  layout_add_default_sections(&(*io_mainw).m_report.m_object.m_layout);
  gtk_paned_add1((*io_mainw).m_paned, l_widget);

  gtk_widget_show_all(GTK_WIDGET((*io_mainw).m_window));

  gtk_widget_get_allocation(GTK_WIDGET((*io_mainw).m_window), &l_alloc);
  g_signal_emit_by_name((*io_mainw).m_window, "size-allocate", &l_alloc, 0);

  return 0;
}

void
mainw_discharge(
  struct mainw*const                    io_mainw)
{

  if ((*io_mainw).m_window)
  {
    gtk_widget_destroy(GTK_WIDGET((*io_mainw).m_window));
  }

  copybuf_discharge(&(*io_mainw).m_copybuf);
  cmd_discharge(&(*io_mainw).m_cmd);
  report_discharge(&(*io_mainw).m_report);
  memset(io_mainw, 0, sizeof(*io_mainw));

  return;
}

extern void
mainw_edit_clear(
  struct mainw*const                    io_mainw)
{

  /*copybuf_clear(&(*io_mainw).m_copybuf);*/
  cmd_clear(&(*io_mainw).m_cmd);
  mainw_edit_set_menu_state(io_mainw);

  return;
}

extern void
mainw_edit_copy(
  struct mainw*const                    io_mainw)
{
  struct item *                         l_item;
  IwrPage *                             l_page;

  l_page= 0;
  copybuf_clear(&(*io_mainw).m_copybuf);

  if (style_label == (*io_mainw).m_report.m_style)
  {
    l_page= (*io_mainw).m_report.m_object.m_label.m_page; 
  }
  else if (style_layout == (*io_mainw).m_report.m_style) 
  {
    l_page= layout_get_focused_page(&(*io_mainw).m_report.m_object.m_layout);
  }

  if (l_page)
  {
    iwr_page_get_item(&l_item, l_page);
    copybuf_copy_item_selected(&(*io_mainw).m_copybuf, l_item);
  }

  return;
}

extern void
mainw_edit_cut(
  struct mainw*const                    io_mainw)
{
  struct item *                         l_item;
  IwrPage *                             l_page;

  l_page= 0;
  copybuf_clear(&(*io_mainw).m_copybuf);

  if (style_label == (*io_mainw).m_report.m_style)
  {
    l_page= (*io_mainw).m_report.m_object.m_label.m_page; 
  }
  else if (style_layout == (*io_mainw).m_report.m_style) 
  {
    l_page= layout_get_focused_page(&(*io_mainw).m_report.m_object.m_layout);
  }

  if (l_page)
  {
    iwr_page_get_item(&l_item, l_page);
    copybuf_copy_item_selected(&(*io_mainw).m_copybuf, l_item);
    mainw_cmd_push(io_mainw, l_page, cmd_delete);
    gtk_widget_queue_draw(GTK_WIDGET(l_page));
  }

  return;
}

extern void
mainw_edit_paste(
  struct mainw*const                    io_mainw)
{
  struct item *                         l_item;
  IwrPage *                             l_page;

  l_page= 0;

  if (style_label == (*io_mainw).m_report.m_style)
  {
    l_page= (*io_mainw).m_report.m_object.m_label.m_page; 
  }
  else if (style_layout == (*io_mainw).m_report.m_style)
  {
    l_page= layout_get_focused_page(&(*io_mainw).m_report.m_object.m_layout);
  }

  if (l_page)
  {
    iwr_page_get_item(&l_item, l_page);
    copybuf_paste_item_selected(l_item, &(*io_mainw).m_copybuf);
    mainw_cmd_push(io_mainw, l_page, cmd_paste); 
    gtk_widget_queue_draw(GTK_WIDGET(l_page));
  }

  return;
}

extern void
mainw_edit_redo(
  struct mainw*const                    io_mainw)
{

  cmd_redo(&(*io_mainw).m_cmd);
  mainw_edit_set_menu_state(io_mainw);

  return;
}

extern void
mainw_edit_set_menu_state(
  struct mainw*const                    io_mainw)
{
  char const*                           l_label;

  l_label= cmd_get_undo_label(&(*io_mainw).m_cmd);
  menu_edit_undo_set_state(l_label);

  l_label= cmd_get_redo_label(&(*io_mainw).m_cmd);
  menu_edit_redo_set_state(l_label);

  return;
}

extern void
mainw_edit_undo(
  struct mainw*const                    io_mainw)
{

  cmd_undo(&(*io_mainw).m_cmd);
  mainw_edit_set_menu_state(io_mainw);

  return;
}

extern int
mainw_open(
  struct mainw*const                    io_mainw,
  char const*                           i_reportfile)
{
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_exit;
  GtkWidget*                            l_widget;

  l_dialog= 0;
  l_error= 0;

  do
  {

    database_field_destroy_cache();

    report_discharge(&(*io_mainw).m_report);
    report_assign(&(*io_mainw).m_report);
    report_assign_filename(&(*io_mainw).m_report, i_reportfile);

    l_exit= report_read(&l_error, &l_widget, &(*io_mainw).m_report);

    if (l_exit)
    {
      l_dialog= GTK_DIALOG(gtk_message_dialog_new(
        get_main_window(),
        GTK_DIALOG_MODAL,
        GTK_MESSAGE_ERROR,
        GTK_BUTTONS_CLOSE,
        "%s",
        (*l_error).message));
      gtk_dialog_run(l_dialog);
      gtk_widget_destroy(GTK_WIDGET(l_dialog));
      break;
    }

    gtk_paned_add1((*io_mainw).m_paned, l_widget);
    gtk_widget_show_all(GTK_WIDGET((*io_mainw).m_window));

    switch((*io_mainw).m_report.m_style)
    {
      case style_layout:
        menu_layout_set_defaults();
        menu_layout_set_sections(&(*io_mainw).m_report.m_object.m_layout);
        layout_set_scale(&(*io_mainw).m_report.m_object.m_layout, 1.0);
        if (0 == (*io_mainw).m_size_allocate_handle)
        {
          (*io_mainw).m_size_allocate_handle= g_signal_connect(
            (*io_mainw).m_window,
            "size-allocate",
            G_CALLBACK(mainw_size_allocate),
            io_mainw);
        }
        mainw_size_allocate_hack(io_mainw); 
        break;
      case style_export:
        menu_export_set_defaults();
        if ((*io_mainw).m_size_allocate_handle)
        {
          g_signal_handler_disconnect((*io_mainw).m_window, (*io_mainw).m_size_allocate_handle);
          (*io_mainw).m_size_allocate_handle= 0;
        }
        export_reload(
          &(*io_mainw).m_report.m_object.m_export,
          &(*io_mainw).m_report.m_aspect,
          &(*io_mainw).m_report.m_query);
        break;
      case style_label:
        menu_label_set_defaults();
        label_set_scale(&(*io_mainw).m_report.m_object.m_label, 1.0);
        if (0 == (*io_mainw).m_size_allocate_handle)
        {
          (*io_mainw).m_size_allocate_handle= g_signal_connect(
            (*io_mainw).m_window,
            "size-allocate",
            G_CALLBACK(mainw_size_allocate),
            io_mainw);
        }
        mainw_size_allocate_hack(io_mainw); 
        break;
      default:
        break;
    }

  }while(0);

  if (l_exit)
  {
    if (l_widget)
    {
      gtk_widget_destroy(l_widget);
    }
    mainw_create_default_layout(io_mainw);
  }

  g_clear_error(&l_error);

  return l_exit;
}

extern int
mainw_run(
  struct mainw*const                    io_mainw)
{
  GtkFileChooser*                       l_chooser;
  GtkWidget*                            l_dialog;
  GError*                               l_error;
  int                                   l_exit;
  char*                                 l_filename;
  GtkFileFilter*                        l_filter;
  struct goop                           l_goop;
  char*                                 l_ptr;
  int                                   l_rc;
  unsigned                              l_size;

  l_error= 0;
  l_exit= -1;
  l_dialog= 0;
  l_filename= 0;
  l_filter= 0;
  memset(&l_goop, 0, sizeof(l_goop));
  
  do
  {

    l_exit= mainw_save(io_mainw);

    if (l_exit)
    {
      break;
    }
    
    l_dialog = gtk_file_chooser_dialog_new(
      "Save File",
      (*io_mainw).m_window,
      GTK_FILE_CHOOSER_ACTION_SAVE,
      "_Cancel",
      GTK_RESPONSE_CANCEL,
      "_Save",
      GTK_RESPONSE_ACCEPT,
      NULL);

    l_chooser= GTK_FILE_CHOOSER(l_dialog);
    
    if ((*io_mainw).m_report.m_reportfile)
    {
      l_size= strlen((*io_mainw).m_report.m_reportfile);
      l_filename= (char*)g_malloc0(5+l_size);
      memcpy(l_filename, (*io_mainw).m_report.m_reportfile, l_size);
      l_ptr= strrchr(l_filename, '.');
      if (0 == l_ptr)
      {
        l_ptr= l_filename+l_size;
      }
      switch ((*io_mainw).m_report.m_style)
      {
        case style_layout:
          memcpy(l_ptr, ".pdf\0", 5);
          break;
        case style_export:
          memcpy(l_ptr, ".csv\0", 5);
          break;
        default:
          break;
      } 
      gtk_file_chooser_set_filename(l_chooser, l_filename);
      g_free(l_filename);
      l_filename= 0;
    }
    else
    {
      switch ((*io_mainw).m_report.m_style)
      {
        case style_label:
          /* fall through */
        case style_layout:
          gtk_file_chooser_set_current_name(l_chooser, "report.pdf");
          break;
        case style_export:
          gtk_file_chooser_set_current_name(l_chooser, "report.csv");
          break;
        default:
          break;
      } 
    }

    switch ((*io_mainw).m_report.m_style)
    {
      case style_layout:
      /* fall through */
      case style_label:
        l_filter= gtk_file_filter_new();
        gtk_file_filter_add_pattern(l_filter, "*.pdf");
        gtk_file_filter_set_name(l_filter, "PDF");
        gtk_file_chooser_add_filter(l_chooser, l_filter);
        break;
      case style_export:
        l_filter= gtk_file_filter_new();
        gtk_file_filter_add_pattern(l_filter, "*.csv");
        gtk_file_filter_set_name(l_filter, "CSV");
        gtk_file_chooser_add_filter(l_chooser, l_filter);
        break;
      default:
        break;
    }

    l_rc= gtk_dialog_run(GTK_DIALOG(l_dialog));

    if (GTK_RESPONSE_ACCEPT != l_rc)
    {
      break;
    }

    l_filename= gtk_file_chooser_get_filename(l_chooser);
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
    l_dialog= 0;

    l_goop.m_interactive= 1;
    l_goop.m_outputfile= l_filename;

    l_exit= report_util_run(
      &l_error,
      (*io_mainw).m_report.m_reportfile, 
      (*io_mainw).m_report.m_aspect.m_password,
      &l_goop);

    if (l_exit)
    {
      break;
    }

    mainw_xdg_open_file(l_filename);

  }while(0);

  if (l_filename)
  {
    g_free(l_filename);
  }

  if (l_dialog)
  {
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  if (l_error)
  {
    _error_display_prompt(l_error); 
    g_clear_error(&l_error);
  }

  return l_exit;
}

extern int
mainw_save(
  struct mainw*const                    io_mainw)
{
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_exit;

  l_error= 0;

  do
  {

    if (0 == (*io_mainw).m_report.m_reportfile)
    {
      l_exit= mainw_save_as(io_mainw); 
      break;
    }

    l_exit= report_write(&l_error, &(*io_mainw).m_report);

    if (l_exit)
    {
      l_dialog= GTK_DIALOG(gtk_message_dialog_new(
        get_main_window(),
        GTK_DIALOG_MODAL,
        GTK_MESSAGE_ERROR,
        GTK_BUTTONS_CLOSE,
        "%s",
        (*l_error).message));
      gtk_dialog_run(l_dialog);
      gtk_widget_destroy(GTK_WIDGET(l_dialog));
    }

  }while(0);

  g_clear_error(&l_error);

  return l_exit;
}

extern int
mainw_save_as(
  struct mainw*const                    io_mainw)
{
  GtkFileChooser*                       l_chooser;
  GtkWidget*                            l_dialog;
  int                                   l_exit;
  char*                                 l_filename;
  GtkFileFilter*                        l_filter;
  int                                   l_rc;

  l_exit= 0;
  l_dialog= 0;
  l_filter= 0;
  
  do
  {
    
    l_dialog = gtk_file_chooser_dialog_new(
      "Save File",
      (*io_mainw).m_window,
      GTK_FILE_CHOOSER_ACTION_SAVE,
      "_Cancel",
      GTK_RESPONSE_CANCEL,
      "_Save",
      GTK_RESPONSE_ACCEPT,
      NULL);

    l_chooser= GTK_FILE_CHOOSER(l_dialog);
    
    if ((*io_mainw).m_report.m_reportfile)
    {
      gtk_file_chooser_set_filename(l_chooser, (*io_mainw).m_report.m_reportfile);
    }
    else
    {
      gtk_file_chooser_set_current_name(l_chooser, "report");
    }

    l_filter= gtk_file_filter_new();
    gtk_file_filter_add_pattern(l_filter, "*.irpt");
    gtk_file_filter_set_name(l_filter, "iwrite report");
    gtk_file_chooser_add_filter(l_chooser, l_filter);

    l_rc= gtk_dialog_run(GTK_DIALOG(l_dialog));

    if (GTK_RESPONSE_ACCEPT != l_rc)
    {
      l_exit= -1;
      break;
    }

    l_filename= gtk_file_chooser_get_filename(l_chooser);
    report_assign_filename(&(*io_mainw).m_report, l_filename);
    g_free(l_filename);

    mainw_save(io_mainw);

  }while(0);

  if (l_dialog)
  {
    gtk_widget_destroy(l_dialog); 
  }

  return l_exit;
}

extern int
mainw_xdg_open_file(
  char const*                           i_filename)
{
  GtkDialog*                            l_dialog;
  gboolean                              l_exit;
  size_t                                l_size;
  char*                                 l_uri;
  GError*                               l_error;

  l_size= strlen(i_filename);
  l_size+= 8;
  l_uri= (char*)g_malloc0(l_size);
  snprintf(l_uri, l_size, "file://%s", i_filename);

  l_error= 0;
  l_exit= gtk_show_uri(0, l_uri, time(0), &l_error);

  if (l_error)
  {
    l_dialog= GTK_DIALOG(gtk_message_dialog_new(
      get_main_window(),
      GTK_DIALOG_MODAL,
      GTK_MESSAGE_ERROR,
      GTK_BUTTONS_CLOSE,
      "Unable to launch file '%s': %s",
      l_uri,
      l_error->message));
    gtk_dialog_run(l_dialog);
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  g_clear_error(&l_error);
  g_free(l_uri);

  return !l_exit;
}
