/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_text.h"

extern void
item_text_assign(
  struct item_text*const                o_text,
  struct item_method*const              o_method)
{

  memset(o_text, 0, sizeof(*o_text));

  (*o_text).m_box.m_len_x= 30.0;
  (*o_text).m_box.m_len_y= 18.0;
  (*o_text).m_font= pango_font_description_new();
  (*o_text).m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, text);

  return;
}

extern void
item_text_copy(
  struct item_text *const               o_text,
  struct item_text const*const          i_text)
{

  memcpy(o_text, i_text, sizeof(*o_text));
  (*o_text).m_font= pango_font_description_copy((*i_text).m_font);
  (*o_text).m_text= g_strdup((*i_text).m_text);
  (*o_text).m_value= 0;

  return;
}

extern void
item_text_discharge(
  struct item_text*const                io_text)
{
  if ((*io_text).m_font)
  {
    pango_font_description_free((*io_text).m_font);
  }

  if ((*io_text).m_text)
  {
    g_free((*io_text).m_text);
  }

  if ((*io_text).m_value)
  {
    g_free((*io_text).m_value);
  }

  memset(io_text, 0, sizeof(*io_text));

  return;
}

extern void
item_text_draw(
  struct item_text*                     i_text,
  struct item_draw_context const*const  i_ctx)
{
  PangoLayout*                          l_layout;
  double                                l_radians;
  char*                                 l_text;
  double                                l_center_x;
  double                                l_center_y;
  cairo_matrix_t                        l_matrix;
  double                                l_shear_x;
  double                                l_shear_y;
  
  l_layout= 0;

  do
  {

    cairo_new_path((*i_ctx).m_cr);
    cairo_save((*i_ctx).m_cr);
    cairo_scale((*i_ctx).m_cr, (*i_ctx).m_scale, (*i_ctx).m_scale);

    cairo_set_source_rgba(
      (*i_ctx).m_cr,
      (*i_text).m_color.red,
      (*i_text).m_color.green,
      (*i_text).m_color.blue,
      (*i_text).m_color.alpha);

    l_layout= pango_cairo_create_layout((*i_ctx).m_cr);

    if (item_draw_design == (*i_ctx).m_option.m_type)
    {
      if ((*i_text).m_text && (*i_text).m_text[0])
      {
        l_text= (*i_text).m_text;
      }
      else
      {
        l_text= "default";
      }
    }
    else
    {
      l_text= (*i_text).m_value;
    }

    if (l_text)
    {
      if (0 == (*i_text).m_autowrap)
      {
        pango_layout_set_width(l_layout, -1);
      }
      else
      {
        pango_layout_set_width(l_layout, (PANGO_SCALE*(*i_text).m_box.m_len_x));
        pango_layout_set_wrap(l_layout, PANGO_WRAP_WORD); 
      }
      pango_layout_set_markup(l_layout, l_text, -1);
    }

    pango_layout_set_font_description(l_layout, (*i_text).m_font);

    l_center_x= (*i_text).m_box.m_pos_x + ((*i_text).m_box.m_len_x / 2.0);
    l_center_y= (*i_text).m_box.m_pos_y + ((*i_text).m_box.m_len_y / 2.0);

    cairo_translate((*i_ctx).m_cr, l_center_x, l_center_y);
    l_radians= geom_deg2rad((*i_text).m_transform.m_rotation);
    cairo_rotate((*i_ctx).m_cr, l_radians);
    cairo_translate((*i_ctx).m_cr, -l_center_x, -l_center_y);

    l_shear_x= tan(geom_deg2rad((*i_text).m_transform.m_shear_x));
    l_shear_y= tan(geom_deg2rad((*i_text).m_transform.m_shear_y));
    
    cairo_matrix_init(
      &l_matrix,
      1.0, l_shear_y,
      l_shear_x, 1.0,
      (*i_text).m_box.m_pos_x, (*i_text).m_box.m_pos_y);
    cairo_transform((*i_ctx).m_cr, &l_matrix);

    pango_cairo_update_layout((*i_ctx).m_cr, l_layout);
    pango_cairo_show_layout((*i_ctx).m_cr, l_layout);

    if ((*i_ctx).m_option.m_highlight && 0 == (*i_ctx).m_selected)
    {
      item_common_cairo_dash_set((*i_ctx).m_cr, line_style_solid);
      cairo_set_line_width((*i_ctx).m_cr, 1.0);
      cairo_set_source_rgb((*i_ctx).m_cr, 0.50, 0.50, 0.00);
      cairo_rectangle(
        (*i_ctx).m_cr,
        0,
        0,
        (*i_text).m_box.m_len_x,
        (*i_text).m_box.m_len_y);
      cairo_stroke((*i_ctx).m_cr);
    }

    cairo_restore((*i_ctx).m_cr);

  }while(0);

  if (l_layout)
  {
    g_object_unref(l_layout);
  }

  return;
}

extern void
item_text_get_bbox(
  struct geom_rect*const                o_bbox,
  cairo_t*                              io_cr,
  double const                          i_scale,
  struct item_text const*const          i_text)
{

  item_common_box_get_bbox(o_bbox, io_cr, &(*i_text).m_box, &(*i_text).m_transform, i_scale);

  return;
}

extern void
item_text_get_position(
  double*const                          o_pos_x,
  double*const                          o_pos_y,
  struct item_text const*const          i_text)
{

  (*o_pos_x)= (*i_text).m_box.m_pos_x;
  (*o_pos_y)= (*i_text).m_box.m_pos_y;

  return;
}

extern void
item_text_get_rotation(
  double*const                          o_rotation,
  struct item_text const*const          i_text)
{

  (*o_rotation)= (*i_text).m_transform.m_rotation;

  return;
}

extern void
item_text_get_size(
  double*const                          o_len_x,
  double*const                          o_len_y,
  struct item_text const*const          i_text)
{

  (*o_len_x)= (*i_text).m_box.m_len_x;
  (*o_len_y)= (*i_text).m_box.m_len_y;

  return;
}

extern void
item_text_get_shear(
  double*const                          o_shear_x,
  double*const                          o_shear_y,
  struct item_text const*const          i_text)
{

  (*o_shear_x)= (*i_text).m_transform.m_shear_x;
  (*o_shear_y)= (*i_text).m_transform.m_shear_y;

  return;
}


extern int
item_text_in_event(
  struct item_text const*const          i_text,
  cairo_t*                              io_cr,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;

  l_hit= item_common_box_in_event(io_cr, &(*i_text).m_box, &(*i_text).m_transform, i_ctx);

  return l_hit;
}

extern void
item_text_move(
  struct item_text*const                io_text,
  struct geom_point const*const         i_event)
{

  item_common_box_move(&(*io_text).m_box, i_event);

  return;
}

extern void
item_text_normalize(
  struct item_text*const                io_text)
{

  item_common_box_normalize(&(*io_text).m_box);

  return;
}

extern void
item_text_set_position(
  struct item_text*const                io_text,
  double const                          i_pos_x,
  double const                          i_pos_y)
{

  (*io_text).m_box.m_pos_x= i_pos_x;
  (*io_text).m_box.m_pos_y= i_pos_y;

  return;
}

extern void
item_text_set_rotation(
  struct item_text*const                io_text,
  double                                i_rotation)
{

  (*io_text).m_transform.m_rotation= i_rotation;

  return;
}

extern void
item_text_set_size(
  struct item_text*const                io_text,
  double                                i_len_x,
  double                                i_len_y)
{

  (*io_text).m_box.m_len_x= i_len_x;
  (*io_text).m_box.m_len_y= i_len_y;

  return;
}

extern void
item_text_set_shear(
  struct item_text*const                io_text,
  double                                i_shear_x,
  double                                i_shear_y)
{

  (*io_text).m_transform.m_shear_x= i_shear_x;
  (*io_text).m_transform.m_shear_y= i_shear_y;

  return;
}

extern int
item_text_read(
  GError**                              o_error,
  struct item_text*const                io_text,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  int                                   l_rc;
  GdkRGBA                               l_rgba;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_rect:
          item_common_box_read(&l_error, &(*io_text).m_box, l_xml);
          break;
        case element_tag_transform:
          item_common_read_transform(&l_error, &(*io_text).m_transform, l_xml);
          break;
        case element_tag_font:
          (*io_text).m_font= pango_font_description_from_string(l_value);
          break;
        case element_tag_color:
          l_rc= gdk_rgba_parse(&l_rgba, l_value);
          if (l_rc)
          {
            (*io_text).m_color= l_rgba;
          }
          break;
        case element_tag_text:
          (*io_text).m_text= g_strdup((*l_xml).m_string->str);
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_text_resize(
  struct item_text*const                io_text,
  struct item_resize_event const*const  i_ctx)
{

  item_common_box_resize(&(*io_text).m_box, i_ctx, &(*io_text).m_transform);

  return;
}

extern void
item_text_rotate(
  struct item_text*const                io_text,
  double const                          i_rotation)
{

  (*io_text).m_transform.m_rotation= i_rotation;

  return;
}
  
extern void
item_text_shear(
  struct item_text*const                io_text,
  double const                          i_delta_x,
  double const                          i_delta_y)
{

  (*io_text).m_transform.m_shear_x+= i_delta_x;
  (*io_text).m_transform.m_shear_y+= i_delta_y;

  return; 
}




extern int
item_text_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_text const*const          i_text)
{
  int                                   l_exit;
  char*                                 l_text;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t<item name=\"text\">\n");
  item_common_box_write(o_error, io_fp, &(*i_text).m_box);
  item_common_write_transform(o_error, io_fp, &(*i_text).m_transform);
  l_text= pango_font_description_to_string((*i_text).m_font);
  fprintf(io_fp, "\t\t\t\t<font>%s</font>\n", l_text);
  g_free(l_text);
  l_text= gdk_rgba_to_string(&(*i_text).m_color);
  fprintf(io_fp, "\t\t\t\t<color>%s</color>\n", l_text);
  g_free(l_text);
  fprintf(io_fp, "\t\t\t\t<autowrap on=\"%s\"/>\n", ((*i_text).m_autowrap ? "true" : "false"));

  if ((*i_text).m_text && (*i_text).m_text[0])
  {
    fprintf(io_fp, "\t\t\t\t<text>%s</text>\n", (*i_text).m_text);
  }

  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
