/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "export_spec.h"

extern void
export_spec_assign(
  struct exports_spec*const             o_export)
{

  memset(o_export, 0, sizeof(*o_export));
  export_config_assign(&(*o_export).m_config);
  database_aspect_assign(&(*o_export).m_aspect);
  query_assign(&(*o_export).m_query);

  return;
}

extern void
export_spec_discharge(
  struct exports_spec*const             io_export)
{
  unsigned                              l_slot;

  export_config_discharge(&(*io_export).m_config);
  database_aspect_discharge(&(*io_export).m_aspect);
  query_discharge(&(*io_export).m_query);

  for(l_slot= 0; (*io_export).m_field_slots > l_slot; l_slot++)
  {
    g_free((*io_export).m_field[l_slot]);
  }

  g_free((*io_export).m_field);

  memset(io_export, 0, sizeof(*io_export));

  return;
}

static int
export_spec_read_tags(
  GError**                              o_error,
  struct exports_spec*const             io_export,
  struct xml const*                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  GList*                                l_node;
  char const*                           l_value;
  struct xml*                           l_xml;
  int                                   l_rc;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (struct xml*)(*l_node).data;
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    l_rc= strcmp("tag", (*l_xml).m_tag);

    if (0 == l_rc)
    {
      (*io_export).m_field= (char**)realloc(
        (*io_export).m_field,
        ((1+(*io_export).m_field_slots) * sizeof(char*)));

      (*io_export).m_field[(*io_export).m_field_slots]= g_strdup(l_value);

      (*io_export).m_field_slots++;
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
export_spec_read(
  GError**                              o_error,
  struct exports_spec*const             io_export,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_has_aspect;
  int                                   l_has_config;
  int                                   l_has_query;
  int                                   l_has_tags;
  GList*                                l_node;
  int                                   l_rc;
  struct xml*                           l_xml;

  l_error= 0;
  l_has_aspect= 0;
  l_has_config= 0;
  l_has_query= 0;
  l_has_tags= 0;
  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    do
    {

      l_rc= strcmp("aspect", (*l_xml).m_tag);
  
      if (0 == l_rc)
      {
        if (l_has_aspect)
        {
          l_error= g_error_new(
            EXPORT_SPEC,
            EXPORT_SPEC_XML_DUPLICATE_ASPECT,
            "Duplicate 'aspect' element");
          l_exit= -1;
          _error_log(l_error);
          break; 
        }
        l_has_aspect= 1;
        l_exit= database_aspect_read(&l_error, &(*io_export).m_aspect, l_xml);
        break;
      }

      l_rc= strcmp("query", (*l_xml).m_tag);

      if (0 == l_rc)
      {
        if (l_has_query)
        {
          l_error= g_error_new(
            EXPORT_SPEC,
            EXPORT_SPEC_XML_DUPLICATE_QUERY,
            "Duplicate 'query' element");
          _error_log(l_error);
          l_exit= -1;
          break; 
        }
        l_has_query= 1;
        l_exit= query_read(&l_error, &(*io_export).m_query, l_xml);
        break;
      }

      l_rc= strcmp("config", (*l_xml).m_tag);
  
      if (0 == l_rc)
      {
        if (l_has_config)
        {
          l_error= g_error_new(
            EXPORT_SPEC,
            EXPORT_SPEC_XML_DUPLICATE_CONFIG,
            "Duplicate 'config' element");
          l_exit= -1;
          _error_log(l_error);
          break; 
        }
        l_has_config= 1;
        l_exit= export_config_read(&l_error, &(*io_export).m_config, l_xml);
        break;
      }

      l_rc= strcmp("tags", (*l_xml).m_tag);

      if (0 == l_rc)
      {
        if (l_has_tags)
        {
          l_error= g_error_new(
            EXPORT_SPEC,
            EXPORT_SPEC_XML_DUPLICATE_TAGS,
            "Duplicate 'tags' element");
          _error_log(l_error);
          l_exit= -1;
          break; 
        }
        l_has_tags= 1;
        l_exit= export_spec_read_tags(&l_error, io_export, l_xml);
        break;
      }

      g_warning("Unkown element ('%s')", (*l_xml).m_tag);

    }while(0);

    if (l_exit)
    {
      break;
    }
  
    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}
