/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "layout_spec.h"

extern void
layout_spec_assign(
  struct layout_spec*const            o_layout)
{

  memset(o_layout, 0, sizeof(*o_layout));

  database_aspect_assign(&(*o_layout).m_aspect);
  query_assign(&(*o_layout).m_query);
  paper_assign(&(*o_layout).m_paper);

  return;
}

static void
layout_spec_division_sync_paper(
  struct layout_spec_division*const     io_division,
  struct paper const*const              i_paper)
{
  gdouble                               l_height;

  l_height= (*io_division).m_paper.m_height;
  (*io_division).m_paper= (*i_paper);
  (*io_division).m_paper.m_height= l_height;

  return;
}

static void
layout_spec_sync_paper(
  struct layout_spec*const              io_layout)
{
  unsigned                              l_slot;

  layout_spec_division_sync_paper(&(*io_layout).m_cover.m_header, &(*io_layout).m_paper);
  layout_spec_division_sync_paper(&(*io_layout).m_cover.m_footer, &(*io_layout).m_paper);
  layout_spec_division_sync_paper(&(*io_layout).m_report.m_header, &(*io_layout).m_paper);
  layout_spec_division_sync_paper(&(*io_layout).m_report.m_detail, &(*io_layout).m_paper);
  layout_spec_division_sync_paper(&(*io_layout).m_report.m_footer, &(*io_layout).m_paper);

  for (l_slot= 0; (*io_layout).m_group_slots > l_slot; l_slot++)
  {
    layout_spec_division_sync_paper(&(*io_layout).m_group[l_slot].m_header, &(*io_layout).m_paper);
    layout_spec_division_sync_paper(&(*io_layout).m_group[l_slot].m_footer, &(*io_layout).m_paper);
  }

  return;
}

static void
layout_spec_division_discharge(
  struct layout_spec_division*const     o_div)
{

  if (o_div && (*o_div).m_item)
  {
    item_discharge((*o_div).m_item);
    g_free((*o_div).m_item);
    paper_discharge(&(*o_div).m_paper);
  }

  memset(o_div, 0, sizeof(*o_div));

  return;
}

extern void
layout_spec_discharge(
  struct layout_spec*const              io_layout)
{
  unsigned                              l_slot;

  database_aspect_discharge(&(*io_layout).m_aspect);
  query_discharge(&(*io_layout).m_query);
  paper_discharge(&(*io_layout).m_paper);

  layout_spec_division_discharge(&(*io_layout).m_cover.m_header); 
  layout_spec_division_discharge(&(*io_layout).m_cover.m_footer); 

  layout_spec_division_discharge(&(*io_layout).m_report.m_header); 
  layout_spec_division_discharge(&(*io_layout).m_report.m_detail); 
  layout_spec_division_discharge(&(*io_layout).m_report.m_footer); 

  for (l_slot= 0; (*io_layout).m_group_slots > l_slot; l_slot++)
  {
    layout_spec_division_discharge(&(*io_layout).m_group[l_slot].m_header);
    layout_spec_division_discharge(&(*io_layout).m_group[l_slot].m_footer);
  }

  memset(io_layout, 0, sizeof(*io_layout));

  return;
}

extern int
layout_spec_read_section(
  GError**                              o_error,
  struct layout_spec_division*const     io_division,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  GList*                                l_node;
  int                                   l_rc;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;
  l_node= (*i_xml).m_children;

  (*io_division).m_item= g_new0(struct item, 1);
  paper_assign(&(*io_division).m_paper);

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (struct xml*)(*l_node).data;

    l_rc= strcmp("item", (*l_xml).m_tag);

    if (0 == l_rc)
    {
      l_exit= item_read(&l_error, (*io_division).m_item, l_xml);
    }
    else
    {
      l_rc= strcmp("paper", (*l_xml).m_tag);

      if (0 == l_rc)
      {
        l_exit= paper_read(&l_error, &(*io_division).m_paper, l_xml);
      }
      else
      {
        l_error= g_error_new(
          LAYOUT,
          LAYOUT_SPEC_XML_UNKNOWN_ELEMENT,
          "Unkown element ('%s')",
          (*l_xml).m_tag);
        _error_log(l_error);
        l_exit= -1;
      }
    }

    if (l_exit)
    {
      break;
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
layout_spec_read_division(
  GError**                              o_error,
  struct layout_spec*const              io_layout,
  struct xml *const                     i_xml)
{
  char const*                           l_attr_name;
  char const*                           l_attr_type;
  struct layout_spec_division*          l_division;
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  int                                   l_rc;
  enum element_tag_type                 l_type;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;
  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (struct xml*)(*l_node).data;
    l_rc= strcmp("section", (*l_xml).m_tag);

    if (l_rc)
    {
      l_error= g_error_new(
        LAYOUT,
        LAYOUT_SPEC_XML_UNKNOWN_ELEMENT,
        "Unknown element ('%s')\n",
        (*l_xml).m_tag);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_attr_name= xml_lookup_attribute_value(l_xml, "name");

    if (0 == l_attr_name)
    {
      l_error= g_error_new(
        LAYOUT,
        LAYOUT_SPEC_XML_MISSING_ATTRIBUTE,
        "Missing 'name' attribute\n");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_attr_type= xml_lookup_attribute_value(l_xml, "type");
  
    if (0 == l_attr_type)
    {
      l_error= g_error_new(
        LAYOUT,
        LAYOUT_SPEC_XML_MISSING_ATTRIBUTE,
        "Missing type attribute\n");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    item_common_lookup(&l_found, &l_type, l_attr_type);
    l_division= 0;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_cover_header:
          l_division= &(*io_layout).m_cover.m_header;
          (*l_division).m_type= section_cover_header;
          if ((*l_division).m_item)
          {
            l_error= g_error_new(
              LAYOUT,
              LAYOUT_SPEC_XML_DUPLICATE_SECTION,
              "Duplicate cover header");
            _error_log(l_error);
            l_exit= -1;
          }
          break;
        case element_tag_cover_footer:
          l_division= &(*io_layout).m_cover.m_footer;
          (*l_division).m_type= section_cover_footer;
          if ((*l_division).m_item)
          {
            l_error= g_error_new(
              LAYOUT,
              LAYOUT_SPEC_XML_DUPLICATE_SECTION,
              "Duplicate cover footer");
            _error_log(l_error);
            l_exit= -1;
          }
          break;
        case element_tag_report_header:
          l_division= &(*io_layout).m_report.m_header;
          (*l_division).m_type= section_report_header;
          if ((*l_division).m_item)
          {
            l_error= g_error_new(
              LAYOUT,
              LAYOUT_SPEC_XML_DUPLICATE_SECTION,
              "Duplicate report header");
            _error_log(l_error);
            l_exit= -1;
          }
          break;
        case element_tag_report_detail:
          l_division= &(*io_layout).m_report.m_detail;
          (*l_division).m_type= section_report_detail;
          if ((*l_division).m_item)
          {
            l_error= g_error_new(
              LAYOUT,
              LAYOUT_SPEC_XML_DUPLICATE_SECTION,
              "Duplicate report detail");
            _error_log(l_error);
            l_exit= -1;
          }
          break;
        case element_tag_report_footer:
          l_division= &(*io_layout).m_report.m_footer;
          (*l_division).m_type= section_report_footer;
          if ((*l_division).m_item)
          {
            l_error= g_error_new(
              LAYOUT,
              LAYOUT_SPEC_XML_DUPLICATE_SECTION,
              "Duplicate report detail");
            _error_log(l_error);
            l_exit= -1;
          }
          break;
        default:
          break;
      }
    }

    if (l_exit)
    {
      break;
    }

    if (0 == l_division)
    {
      l_error= g_error_new(
        LAYOUT,
        LAYOUT_SPEC_XML_UNKNOWN_ATTRIBUTE,
        "Unkown section ('%s')\n",
        l_attr_type);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    g_strlcpy((*l_division).m_tag, l_attr_name, sizeof((*l_division).m_tag));

    l_exit= layout_spec_read_section(&l_error, l_division, l_xml);

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
layout_spec_read_group(
  GError**                              o_error,
  struct layout_spec_division_group*    io_group,
  struct xml const*const                i_xml)
{
  struct layout_spec_division*          l_division;
  char const*                           l_attr_name;
  enum element_tag_type                 l_type;
  char const*                           l_attr_type;
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (struct xml*)(*l_node).data;

    l_attr_name= xml_lookup_attribute_value(l_xml, "name");

    if (0 == l_attr_name)
    {
      l_error= g_error_new(
        LAYOUT,
        LAYOUT_SPEC_XML_MISSING_ATTRIBUTE,
        "Missing 'name' attribute\n");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_attr_type= xml_lookup_attribute_value(l_xml, "type");

    if (0 == l_attr_type)
    {
      l_error= g_error_new(
        LAYOUT,
        LAYOUT_SPEC_XML_MISSING_ATTRIBUTE,
        "Missing 'type' attribute\n");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    item_common_lookup(&l_found, &l_type, l_attr_type);
    l_division= 0;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_group_header:
          l_division= &(*io_group).m_header;
          (*l_division).m_type= section_group_header;
          if ((*l_division).m_item)
          {
            l_error= g_error_new(
              LAYOUT,
              LAYOUT_SPEC_XML_DUPLICATE_SECTION,
              "Duplicate group header");
            _error_log(l_error);
            l_exit= -1;
          } 
          break;
        case element_tag_group_footer:
          l_division= &(*io_group).m_footer;
          (*l_division).m_type= section_group_footer;
          if ((*l_division).m_item)
          {
            l_error= g_error_new(
              LAYOUT,
              LAYOUT_SPEC_XML_DUPLICATE_SECTION,
              "Duplicate group footer");
            _error_log(l_error);
            l_exit= -1;
          } 
          break;
        default:
          break;
      }
    }

    if (l_exit)
    {
      break;
    }

    if (0 == l_division)
    {
      l_error= g_error_new(
        LAYOUT,
        LAYOUT_SPEC_XML_UNKNOWN_ATTRIBUTE,
        "Unkown section ('%s')\n",
        l_attr_type);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    g_strlcpy((*l_division).m_tag, l_attr_name, sizeof((*l_division).m_tag));

    l_exit= layout_spec_read_section(&l_error, l_division, l_xml);

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
layout_spec_read_groups(
  GError**                              o_error,
  struct layout_spec*const              io_layout,
  struct xml const*const                i_xml)
{
  char const*                           l_attr_name;
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  struct layout_spec_division_group*    l_group;
  GList*                                l_node;
  int                                   l_rc;
  unsigned                              l_slot;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (struct xml*)(*l_node).data;
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    do
    {

      l_rc= strcmp("page_break", (*l_xml).m_tag);

      if (0 == l_rc)
      {
        g_strlcpy((*io_layout).m_page_break.m_tag, l_value, sizeof((*io_layout).m_page_break.m_tag));
        break;
      }

      l_rc= strcmp("group", (*l_xml).m_tag);

      if (l_rc)
      {
        l_error= g_error_new(
          LAYOUT,
          LAYOUT_SPEC_XML_UNKNOWN_ELEMENT,
          "Unkown element ('%s')",
          (*l_xml).m_tag);
        _error_log(l_error);
        l_exit= -1;
        break;
      }

      l_attr_name= xml_lookup_attribute_value(l_xml, "name");

      if (0 == l_attr_name)
      {
        l_error= g_error_new(
          LAYOUT,
          LAYOUT_SPEC_XML_MISSING_ATTRIBUTE,
          "Missing 'name' attribute\n");
        _error_log(l_error);
        l_exit= -1;
        break;
      }

      l_found= 0;

      for(l_slot= 0; (*io_layout).m_group_slots > l_slot; l_slot++)
      {

        l_found= (0 == strncmp(
          (*io_layout).m_group[l_slot].m_break.m_tag,
          l_attr_name,
          sizeof((*io_layout).m_group[l_slot].m_break.m_tag)));

        if (l_found)
        {
          break;
        }

      }

      if (l_found)
      {
        l_error= g_error_new(
          LAYOUT,
          LAYOUT_SPEC_XML_DUPLICATE_GROUP,
          "Duplicate group ('%s')\n",
          (*l_xml).m_tag);
        _error_log(l_error);
        l_exit= -1;
        break;
      }

      (*io_layout).m_group= (struct layout_spec_division_group*)
        realloc(
          (*io_layout).m_group,
          (1+(*io_layout).m_group_slots) * sizeof(*l_group));

      l_group= &(*io_layout).m_group[(*io_layout).m_group_slots];
      memset(l_group, 0, sizeof(*l_group));
      (*io_layout).m_group_slots++;

      g_strlcpy(
        (*l_group).m_break.m_tag, 
        l_attr_name,
        sizeof((*l_group).m_break.m_tag));

      l_exit= layout_spec_read_group(&l_error, l_group, l_xml);

    }while(0);

    if (l_exit)
    {
      break;
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
layout_spec_read(
  GError**                              o_error,
  struct layout_spec*const              io_layout,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_has_aspect;
  int                                   l_has_division;
  int                                   l_has_group;
  int                                   l_has_paper;
  int                                   l_has_query;
  GList*                                l_node;
  int                                   l_rc;
  struct xml*                           l_xml;

  l_error= 0;
  l_has_aspect= 0;
  l_has_division= 0;
  l_has_group= 0;
  l_has_paper= 0;
  l_has_query= 0;
  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    do
    {

      l_rc= strcmp("aspect", (*l_xml).m_tag);
  
      if (0 == l_rc)
      {
        if (l_has_aspect)
        {
          l_error= g_error_new(
            LAYOUT,
            LAYOUT_SPEC_XML_DUPLICATE_ASPECT,
            "Duplicate 'aspect' element");
          l_exit= -1;
          _error_log(l_error);
          break; 
        }
        l_has_aspect= 1;
        l_exit= database_aspect_read(&l_error, &(*io_layout).m_aspect, l_xml);
        break;
      }

      l_rc= strcmp("query", (*l_xml).m_tag);

      if (0 == l_rc)
      {
        if (l_has_query)
        {
          l_error= g_error_new(
            LAYOUT,
            LAYOUT_SPEC_XML_DUPLICATE_QUERY,
            "Duplicate 'query' element");
          _error_log(l_error);
          l_exit= -1;
          break; 
        }
        l_has_query= 1;
        l_exit= query_read(&l_error, &(*io_layout).m_query, l_xml);
        break;
      }

      l_rc= strcmp("paper", (*l_xml).m_tag);

      if (0 == l_rc)
      {
        if (l_has_paper)
        {
          l_error= g_error_new(
            LAYOUT,
            LAYOUT_SPEC_XML_DUPLICATE_PAPER,
            "Duplicate 'paper' element");
          _error_log(l_error);
          l_exit= -1;
          break; 
        }
        l_has_paper= 1;
        l_exit= paper_read(&l_error, &(*io_layout).m_paper, l_xml);
        break;
      }

      l_rc= strcmp("division", (*l_xml).m_tag);

      if (0 == l_rc)
      {
        if (l_has_division)
        {
          l_error= g_error_new(
            LAYOUT,
            LAYOUT_SPEC_XML_DUPLICATE_DIVISION,
            "Duplicate 'division' element");
          _error_log(l_error);
          l_exit= -1;
          break; 
        }
        l_has_division= 1;
        l_exit= layout_spec_read_division(&l_error, io_layout, l_xml);
        break;
      }

      l_rc= strcmp("groups", (*l_xml).m_tag);

      if (0 == l_rc)
      {
        if (l_has_group)
        {
          l_error= g_error_new(
          LAYOUT,
          LAYOUT_SPEC_XML_DUPLICATE_GROUP,
          "Duplicate 'group' element");
          _error_log(l_error);
          l_exit= -1;
          break;
        }
        l_has_group= 1;
        l_exit= layout_spec_read_groups(&l_error, io_layout, l_xml);
        break;
      }

      g_warning("Unkown element ('%s')", (*l_xml).m_tag);

    }while(0);

    if (l_exit)
    {
      break;
    }
  
    l_node= (*l_node).next;

  }while(1);

  if (0 == l_exit)
  {
    layout_spec_sync_paper(io_layout);
  }

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}
