/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_fnumber.h"
#include "item_fsum.h"

extern void
item_fnumber_assign(
  struct item_fnumber*const             o_fnumber,
  struct item_method*const              o_method)
{

  memset(o_fnumber, 0, sizeof(*o_fnumber));

  (*o_fnumber).m_len_x= 30.0;
  (*o_fnumber).m_len_y= 18.0;
  (*o_fnumber).m_font= pango_font_description_new();
  (*o_fnumber).m_color.alpha= 1.0;
  (*o_fnumber).m_border.m_style= line_style_solid;
  (*o_fnumber).m_border.m_thickness= 1.0;
  (*o_fnumber).m_border.m_color.alpha= 1.0;
  (*o_fnumber).m_fill.m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, fnumber);

  return;
}

extern void
item_fnumber_discharge(
  struct item_fnumber*const             io_fnumber)
{
  if ((*io_fnumber).m_font)
  {
    pango_font_description_free((*io_fnumber).m_font);
  }

  if ((*io_fnumber).m_value)
  {
    g_free((*io_fnumber).m_value);
  }

  memset(io_fnumber, 0, sizeof(*io_fnumber));

  return;
}

static int
item_fnumber_get_text(
  char**                                o_text,
  struct item_fnumber const*const       i_fnumber,
  struct item_draw_context const*const  i_ctx)
{
  int                                   l_exit;
  char                                  l_number[14+DECNUMDIGITS];

  (*o_text)= 0;
  l_exit =0;
  memset(&l_number, 0, sizeof(l_number));
  l_number[0]= '0';

  if ((item_draw_print == (*i_ctx).m_option.m_type) && (*i_fnumber).m_value)
  {
    strncpy(l_number, (*i_fnumber).m_value, sizeof(l_number));
  }

  if (number_float == (*i_fnumber).m_type)
  {
    l_exit= icu_double_to_locale(
      o_text,
      l_number,
      (*i_fnumber).m_locale,
      2,
      (*i_fnumber).m_pattern);
  }
  else if (number_integer == (*i_fnumber).m_type)
  {
    l_exit= icu_integer_to_locale(
      o_text, 
      l_number, 
      (*i_fnumber).m_locale, 
      (*i_fnumber).m_pattern);
  }
  else if (number_monetary == (*i_fnumber).m_type)
  {
    l_exit= icu_monetary_to_locale(o_text, l_number, (*i_fnumber).m_locale);
  }

  return l_exit;
}

extern void
item_fnumber_draw(
  struct item_fnumber*const             i_fnumber,
  struct item_draw_context const*const  i_ctx)
{
  PangoFontDescription*                 l_font;
  PangoLayout*                          l_layout;
  double                                l_pos_x;
  char*                                 l_text;
  int                                   l_text_width;

  l_font= 0;
  l_layout= 0;
  l_text= 0;

  do
  {

    if ((*i_fnumber).m_fill.m_active)
    {
      cairo_set_source_rgba(
        (*i_ctx).m_cr,
        (*i_fnumber).m_fill.m_color.red,
        (*i_fnumber).m_fill.m_color.green,
        (*i_fnumber).m_fill.m_color.blue,
        (*i_fnumber).m_fill.m_color.alpha);

      cairo_rectangle(
        (*i_ctx).m_cr,
        (*i_fnumber).m_pos_x,
        (*i_fnumber).m_pos_y,
        (*i_fnumber).m_len_x,
        (*i_fnumber).m_len_y);

      cairo_fill((*i_ctx).m_cr);
    }

    cairo_save((*i_ctx).m_cr);

    cairo_rectangle(
      (*i_ctx).m_cr,
      (*i_fnumber).m_pos_x,
      (*i_fnumber).m_pos_y,
      (*i_fnumber).m_len_x,
      (*i_fnumber).m_len_y);

    cairo_clip((*i_ctx).m_cr);

    cairo_set_source_rgba(
      (*i_ctx).m_cr,
      (*i_fnumber).m_color.red,
      (*i_fnumber).m_color.green,
      (*i_fnumber).m_color.blue,
      (*i_fnumber).m_color.alpha);

    l_layout= pango_cairo_create_layout((*i_ctx).m_cr);
    pango_layout_set_font_description(l_layout, (*i_fnumber).m_font); 

    item_fnumber_get_text(&l_text, i_fnumber, i_ctx);

    pango_layout_set_text(l_layout, l_text, -1);
    pango_layout_get_pixel_size(l_layout, &l_text_width, 0);
    l_pos_x= ((*i_fnumber).m_pos_x+(*i_fnumber).m_len_x) - l_text_width;
    cairo_translate((*i_ctx).m_cr, l_pos_x, (*i_fnumber).m_pos_y);

    pango_cairo_update_layout((*i_ctx).m_cr, l_layout);
    pango_cairo_show_layout((*i_ctx).m_cr, l_layout);

    cairo_restore((*i_ctx).m_cr);

    if ((*i_fnumber).m_border.m_active)
    {

      item_common_cairo_dash_set((*i_ctx).m_cr, (*i_fnumber).m_border.m_style);
      cairo_set_line_width((*i_ctx).m_cr, (*i_fnumber).m_border.m_thickness);

      cairo_set_source_rgba(
        (*i_ctx).m_cr,
        (*i_fnumber).m_border.m_color.red,
        (*i_fnumber).m_border.m_color.green,
        (*i_fnumber).m_border.m_color.blue,
        (*i_fnumber).m_border.m_color.alpha);

      cairo_rectangle(
        (*i_ctx).m_cr,
        (*i_fnumber).m_pos_x,
        (*i_fnumber).m_pos_y,
        (*i_fnumber).m_len_x,
        (*i_fnumber).m_len_y);

      cairo_stroke((*i_ctx).m_cr);
    }

    if (0 == (*i_ctx).m_selected)
    {
      if ((*i_ctx).m_option.m_highlight &&
        (item_draw_design == (*i_ctx).m_option.m_type))
      {
        item_common_box_draw_hightlight(
          (*i_ctx).m_cr, 
          (struct geom_rect*)i_fnumber,
          1.0);
      }
      break;
    }

    item_common_box_draw_grips((*i_ctx).m_cr, (struct geom_rect*)i_fnumber);

  }while(0);

  if (l_font)
  {
    pango_font_description_free(l_font);
  }

  if (l_layout)
  {
    g_object_unref(l_layout);
  }

  if (l_text)
  {
    g_free(l_text);
  }

  return;
}

extern void
item_fnumber_get_bbox(
  struct geom_rect*const                o_bbox,
  void const*const                      i_object)
{
  (*o_bbox)= *(struct geom_rect*)i_object;
  return;
}

extern void
item_fnumber_in_event(
  enum event_type*const                 o_type,
  struct item_fnumber const*const       i_fnumber,
  struct item_in_event const*const i_ctx)
{

  item_common_box_in_event(o_type, i_ctx, (struct geom_rect*)i_fnumber);

  return;
}

extern void
item_fnumber_move(
  struct item_fnumber*const             io_fnumber,
  struct geom_point const*const         i_event)
{
  item_common_box_move((struct geom_rect*)io_fnumber, i_event);
  return;
}

extern void
item_fnumber_normalize(
  struct item_fnumber*const             io_fnumber,
  double const                          i_bound_width,
  double const                          i_bound_height)
{
  item_common_box_normalize(
    (struct geom_rect*)io_fnumber, 
    i_bound_width,
    i_bound_height);
  return;
}

extern int
item_fnumber_read(
  GError**                              o_error,
  struct item_fnumber*const             io_fnumber,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  int                                   l_rc;
  GdkRGBA                               l_rgba;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_pos_x:
          (*io_fnumber).m_pos_x= xml_convert_double(l_value);
          break;
        case element_tag_pos_y:
          (*io_fnumber).m_pos_y= xml_convert_double(l_value);
          break;
        case element_tag_len_x:
          (*io_fnumber).m_len_x= xml_convert_double(l_value);
          break;
        case element_tag_len_y:
          (*io_fnumber).m_len_y= xml_convert_double(l_value);
          break;
        case element_tag_font:
          (*io_fnumber).m_font= pango_font_description_from_string(l_value);
          break;
        case element_tag_color:
          l_rc= gdk_rgba_parse(&l_rgba, l_value);
          if (l_rc)
          {
            (*io_fnumber).m_color= l_rgba;
          }
          break;
        case element_tag_border:
          l_exit= item_common_read_border_attr(&l_error, &(*io_fnumber).m_border, l_xml);
          break;
        case element_tag_fill:
          l_exit= item_common_read_fill(&l_error, &(*io_fnumber).m_fill, l_xml);
          break;
        case element_tag_type:
          (*io_fnumber).m_type= item_common_get_number(l_value);
          break;
        case element_tag_pattern:
          (*io_fnumber).m_pattern= g_strdup(l_value);
          break;
        case element_tag_locale:
          (*io_fnumber).m_locale= g_strdup(l_value);
          break;
        case element_tag_tag:
          g_strlcpy((*io_fnumber).m_tag, l_value, sizeof((*io_fnumber).m_tag));
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_fnumber_resize(
  struct item_fnumber*const             io_fnumber,
  struct item_resize_event const*const  i_ctx)
{

  item_common_box_resize((struct geom_rect *)io_fnumber, i_ctx);

  return;
}

extern int
item_fnumber_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_fnumber const*const       i_fnumber)
{
  char                                  l_buf[G_ASCII_DTOSTR_BUF_SIZE+1];
  int                                   l_exit;
  char*                                 l_text;

  l_exit= 0;
  l_text= 0;

  fprintf(io_fp, "\t\t\t<item name=\"fnumber\">\n");
  fprintf(io_fp, "\t\t\t\t<pos_x>%s</pos_x>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_fnumber).m_pos_x));
  fprintf(io_fp, "\t\t\t\t<pos_y>%s</pos_y>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_fnumber).m_pos_y));
  fprintf(io_fp, "\t\t\t\t<len_x>%s</len_x>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_fnumber).m_len_x));
  fprintf(io_fp, "\t\t\t\t<len_y>%s</len_y>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_fnumber).m_len_y));
  l_text= pango_font_description_to_string((*i_fnumber).m_font);
  fprintf(io_fp, "\t\t\t\t<font>%s</font>\n", l_text);
  g_free(l_text);
  l_text= gdk_rgba_to_string(&(*i_fnumber).m_color);
  fprintf(io_fp, "\t\t\t\t<color>%s</color>\n", l_text);
  g_free(l_text);
  item_common_write_border_attr(o_error, io_fp, &(*i_fnumber).m_border);
  item_common_write_fill(o_error, io_fp, &(*i_fnumber).m_fill);
  fprintf(io_fp, "\t\t\t\t<type>%s</type>\n", item_common_get_number_text((*i_fnumber).m_type));

  if ((*i_fnumber).m_pattern && (*i_fnumber).m_pattern[0])
  {
    fprintf(io_fp, "\t\t\t\t<pattern>%s</pattern>\n", (*i_fnumber).m_pattern);
  }

  if ((*i_fnumber).m_locale && (*i_fnumber).m_locale[0])
  {
    fprintf(io_fp, "\t\t\t\t<locale>%s</locale>\n", (*i_fnumber).m_locale);
  }

  if ((*i_fnumber).m_tag[0])
  {
    fprintf(io_fp, "\t\t\t\t<tag>%s</tag>\n", (*i_fnumber).m_tag);
  }

  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
