/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_common_box.h"

extern void
item_common_box_normalize(
  struct geom_rect *const               io_box,
  double const                          i_bound_width,
  double const                          i_bound_height)
{
  int                                   l_pos;

  if (0 > (int)(*io_box).m_len_x)
  {
    (*io_box).m_pos_x+= (*io_box).m_len_x;
    (*io_box).m_len_x= -1.0 * (*io_box).m_len_x;
  }

  if (0 > (int)(*io_box).m_len_y)
  {
    (*io_box).m_pos_y+= (*io_box).m_len_y;
    (*io_box).m_len_y= -1.0 * (*io_box).m_len_y;
  }

  if (0 == (int)(*io_box).m_len_x)
  {
    (*io_box).m_len_x= 1.0;
  }

  if (0 == (int)(*io_box).m_len_y)
  {
    (*io_box).m_len_y= 1.0;
  }

  l_pos= (int)(*io_box).m_pos_x+(int)(*io_box).m_len_x;

  if (0 > l_pos)
  {
    (*io_box).m_pos_x= (1.0 - (*io_box).m_len_x);
  }

  l_pos= (int)(*io_box).m_pos_y+(int)(*io_box).m_len_y;
  
  if (0 > l_pos)
  {
    (*io_box).m_pos_y= (1.0 - (*io_box).m_len_y);
  }

  if (i_bound_width < (*io_box).m_pos_x)
  {
    (*io_box).m_pos_x= i_bound_width - 1.0;
  }

  if (i_bound_height < (*io_box).m_pos_y)
  {
    (*io_box).m_pos_y= i_bound_height - 1.0;
  }

  return;
}

extern void
item_common_box_resize(
  struct geom_rect *const               io_box,
  struct item_resize_event const*const  i_ctx)
{

  switch((*i_ctx).m_type)
  {
    case event_resize_east:
      (*io_box).m_len_x+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_north:
      (*io_box).m_pos_y+= (*i_ctx).m_event.m_pos_y;
      (*io_box).m_len_y-= (*i_ctx).m_event.m_pos_y;
      break;
    case event_resize_north_east:
      (*io_box).m_pos_y+= (*i_ctx).m_event.m_pos_y;
      (*io_box).m_len_y-= (*i_ctx).m_event.m_pos_y;
      (*io_box).m_len_x+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_north_west:
      (*io_box).m_pos_y+= (*i_ctx).m_event.m_pos_y;
      (*io_box).m_len_y-= (*i_ctx).m_event.m_pos_y;
      (*io_box).m_pos_x+= (*i_ctx).m_event.m_pos_x;
      (*io_box).m_len_x-= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_south:
      (*io_box).m_len_y+= (*i_ctx).m_event.m_pos_y;
      break;
    case event_resize_south_east:
      (*io_box).m_len_y+= (*i_ctx).m_event.m_pos_y;
      (*io_box).m_len_x+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_south_west:
      (*io_box).m_len_y+= (*i_ctx).m_event.m_pos_y;
      (*io_box).m_pos_x+= (*i_ctx).m_event.m_pos_x;
      (*io_box).m_len_x-= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_west:
      (*io_box).m_pos_x+= (*i_ctx).m_event.m_pos_x;
      (*io_box).m_len_x-= (*i_ctx).m_event.m_pos_x;
      break;
    default:
      break;
  }

  return;
}

static int
item_common_box_in_event_grips(
  enum event_type*const                 o_type,
  struct item_in_event const*const      i_ctx,
  struct geom_rect const*const          i_box)
{
  int                                   l_hit;

  l_hit= 0;

  do
  {
    /* top,left */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x-4.0,
      (*i_box).m_pos_y-4.0,
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_north_west;
      break;
    }

    /* top, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+((*i_box).m_len_x/2)-4.0, 
      (*i_box).m_pos_y-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_north;
      break;
    }

    /* top, right */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+(*i_box).m_len_x-4.0,
      (*i_box).m_pos_y-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_north_east;
      break;
    }

    /* bottom, right */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+(*i_box).m_len_x-4.0, 
      (*i_box).m_pos_y+(*i_box).m_len_y-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_south_east;
      break;
    }

    /* bottom, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+((*i_box).m_len_x/2)-4.0, 
      (*i_box).m_pos_y+(*i_box).m_len_y-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_south;
      break;
    }

    /* bottom, left */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x-4.0, 
      (*i_box).m_pos_y+(*i_box).m_len_y-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_south_west;
      break;
    }

    /* right side, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+(*i_box).m_len_x-4.0, 
      (*i_box).m_pos_y+((*i_box).m_len_y/2)-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_east;
      break;
    }

    /* left side, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x-4.0,
      (*i_box).m_pos_y+((*i_box).m_len_y/2)-4.0,
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_west;
      break;
    }

  }while(0);

  return l_hit;
}

extern void
item_common_box_in_event(
  enum event_type*const                 o_type,
  struct item_in_event const*const      i_ctx,
  struct geom_rect const*const          i_box)
{
  gdouble                               l_adj_len;
  gdouble                               l_adj_pos;
  int                                   l_hit;

  do
  {

    (*o_type)= event_none;

    if (0 == (*i_ctx).m_selected)
    {
      l_adj_pos= 0.0;
      l_adj_len= 0.0;
    }
    else
    {
      l_adj_pos= 4.0;
      l_adj_len= 8.0;
    }

    l_hit= item_common_event_inside(
      i_ctx,
      ((*i_box).m_pos_x-l_adj_pos),
      ((*i_box).m_pos_y-l_adj_pos),
      ((*i_box).m_len_x+l_adj_len),
      ((*i_box).m_len_y+l_adj_len));

    if (0 == l_hit)
    {
      break;
    }

    if ((*i_ctx).m_selected)
    {
      l_hit= item_common_box_in_event_grips(o_type, i_ctx, i_box);
  
      if (l_hit)
      {
        break;
      }
    }

    (*o_type)= event_move;

  }while(0);

  return;
}

extern void
item_common_box_draw_grips(
  cairo_t *const                        io_cr,
  struct geom_rect const*const          i_box)
{
  int                                   l_pos_x;
  int                                   l_pos_y;
  int                                   l_len_x;
  int                                   l_len_y;

  l_pos_x= (*i_box).m_pos_x;
  l_pos_y= (*i_box).m_pos_y;
  l_len_x= (*i_box).m_len_x;
  l_len_y= (*i_box).m_len_y;

  item_common_cairo_dash_set(io_cr, line_style_solid);
  cairo_set_line_width(io_cr, 1.0);
  cairo_set_source_rgb(io_cr, 0.0, 0.0, 0.0); 
  cairo_rectangle(io_cr, l_pos_x, l_pos_y, l_len_x, l_len_y);
  cairo_stroke(io_cr);

  cairo_set_source_rgb(io_cr, 0.0, 1.0, 0.0);

  /* top,left */
  cairo_rectangle(
    io_cr,
    l_pos_x-4.0,
    l_pos_y-4.0,
    8.0,
    8.0); 
  cairo_fill(io_cr);

  /* top, middle */
  cairo_rectangle(
    io_cr,
    l_pos_x+(l_len_x/2)-4.0,
    l_pos_y-4.0,
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* top, right */
  cairo_rectangle(
    io_cr,
    l_pos_x+l_len_x-4.0,
    l_pos_y-4.0,
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* bottom, right */
  cairo_rectangle(
    io_cr,
    l_pos_x+l_len_x-4.0,
    l_pos_y+l_len_y-4.0, 
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* bottom, middle */
  cairo_rectangle(
    io_cr,
    l_pos_x+(l_len_x/2)-4.0,
    l_pos_y+l_len_y-4.0, 
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* bottom, left */
  cairo_rectangle(
    io_cr,
    l_pos_x-4.0,
    l_pos_y+l_len_y-4.0, 
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* right side, middle */
  cairo_rectangle(
    io_cr,
    l_pos_x+l_len_x-4.0,
    l_pos_y+(l_len_y/2)-4.0, 
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* left side, middle */
  cairo_rectangle(
    io_cr,
    l_pos_x-4.0,
    l_pos_y+(l_len_y/2)-4.0, 
    8.0,
    8.0);
  cairo_fill(io_cr);

  return;
}

extern void
item_common_box_draw_hightlight(
  cairo_t *const                        i_cr,
  struct geom_rect const*const          i_box,
  double const                          i_width)
{
  double                                l_width;

  if (1.00 > i_width)
  {
    l_width= 1.00;
  }
  else
  {
    l_width= i_width;
  }

  item_common_cairo_dash_set(i_cr, line_style_solid);
  cairo_set_line_width(i_cr, l_width);
  cairo_set_source_rgb(i_cr, 0.50, 0.50, 0.00);
  cairo_rectangle(
    i_cr, 
    (*i_box).m_pos_x,
    (*i_box).m_pos_y,
    (*i_box).m_len_x,
    (*i_box).m_len_y);
  cairo_stroke(i_cr);

  return;
}

extern void
item_common_box_move(
  struct geom_rect*const                io_box,
  struct geom_point const*const         i_event)
{

  (*io_box).m_pos_x+= (*i_event).m_pos_x;
  (*io_box).m_pos_y+= (*i_event).m_pos_y;

  return;
}
