/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_common.h"

extern int
item_common_event_inside(
  struct item_in_event const*const      i_ctx,
  double const                          i_pos_x,
  double const                          i_pos_y,
  double const                          i_len_x,
  double const                          i_len_y)
{
  int                                   l_hit;
  double                                l_pos_x;
  double                                l_pos_y;
  double                                l_len_x;
  double                                l_len_y;
  double                                l_pos_x1;
  double                                l_pos_x2;
  double                                l_pos_y1;
  double                                l_pos_y2;

  if (0.0 > i_len_x)
  {
    l_pos_x= i_pos_x + i_len_x;
    l_len_x= -1.0 * i_len_x;
  }
  else
  {
    l_pos_x= i_pos_x;
    l_len_x= i_len_x;
  }

  if (0.0 > i_len_y)
  {
    l_pos_y= i_pos_y + i_len_y;
    l_len_y= -1.0 * i_len_y; 
  }
  else
  {
    l_pos_y= i_pos_y;
    l_len_y= i_len_y;
  }

  l_pos_x1= (*i_ctx).m_scale * l_pos_x;
  l_pos_y1= (*i_ctx).m_scale * l_pos_y;
  l_pos_x2= (*i_ctx).m_scale * (l_pos_x + l_len_x);
  l_pos_y2= (*i_ctx).m_scale * (l_pos_y + l_len_y);

  l_hit= ((*i_ctx).m_event.m_pos_x >= l_pos_x1 && 
    (*i_ctx).m_event.m_pos_x <= l_pos_x2 && 
    (*i_ctx).m_event.m_pos_y >= l_pos_y1 && 
    (*i_ctx).m_event.m_pos_y <= l_pos_y2); 

  return l_hit;
}

extern enum date_style_type
item_common_get_datetime_style(
  char const*                           i_style)
{
  int                                   l_rc;
  enum date_style_type                  l_style;

  l_style= date_style_none;

  do
  {

    l_rc= strcmp("short", i_style);

    if (0 == l_rc)
    {
      l_style= date_style_short;
      break;
    }

    l_rc= strcmp("medium", i_style);

    if (0 == l_rc)
    {
      l_style= date_style_medium;
      break;
    }

    l_rc= strcmp("long", i_style);

    if (0 == l_rc)
    {
      l_style= date_style_long;
      break;
    }

    l_rc= strcmp("full", i_style);

    if (0 == l_rc)
    {
      l_style= date_style_full;
      break;
    }

  }while(0);

  return l_style;
}

extern char const*
item_common_get_datetime_style_text(
  enum date_style_type const            i_style)
{
  char const*                           l_text;

  switch(i_style)
  {
    case date_style_none:
      l_text= "none";
      break;
    case date_style_short:
      l_text= "short";
      break;
    case date_style_medium:
      l_text= "medium";
      break;
    case date_style_long:
      l_text= "long";
      break;
    case date_style_full:
      l_text= "full";
      break;
    default:
      l_text= "";
      break;
  }

  return l_text;
}

extern enum line_style_type
item_common_get_line_style(
  char const*                           i_style)
{
  int                                   l_rc;
  enum line_style_type                  l_style;

  l_style= line_style_solid;

  do
  {

    l_rc= strcmp("DASH", i_style);

    if (0 == l_rc)
    {
      l_style= line_style_dash;
      break;
    }

    l_rc= strcmp("DOT", i_style);

    if (0 == l_rc)
    {
      l_style= line_style_dot;
      break;
    }

    l_rc= strcmp("DASHDOT", i_style);

    if (0 == l_rc)
    {
      l_style= line_style_dash_dot;
      break;
    }

  }while(0);

  return l_style;
}

extern char const*
item_common_get_line_style_text(
  enum line_style_type const            i_style)
{
  char const*                           l_text;

  switch(i_style)
  {
  case line_style_solid:
    l_text= "SOLID";
    break;
  case line_style_dash:
    l_text= "DASH";
    break;
  case line_style_dot:
    l_text= "DOT";
    break;
  case line_style_dash_dot:
    l_text= "DASHDOT";
    break;
  default:
    l_text= "";
    break;
  }

  return l_text;
}

extern enum number_type 
item_common_get_number(
  char const*                           i_type)
{
  int                                   l_rc;
  enum number_type                      l_type;

  l_type= number_float;

  do
  {

    l_rc= strcmp("integer", i_type);

    if (0 == l_rc)
    {
      l_type= number_integer;
      break;
    }

    l_rc= strcmp("monetary", i_type);

    if (0 == l_rc)
    {
      l_type= number_monetary;
      break;
    }

  }while(0);

  return l_type;
}

extern char const*
item_common_get_number_text(
  enum number_type const                i_type)
{
  char const*                           l_text;

  switch(i_type)
  {
    case number_float:
      l_text= "float";
      break;
    case number_integer:
      l_text= "integer";
      break;
    case number_monetary:
      l_text= "monetary";
      break;
    default:
      break;
  }

  return l_text;
}

extern PangoWrapMode 
item_common_get_wrapmode(
  char const*                           i_mode)
{
  PangoWrapMode                         l_mode;
  int                                   l_rc;

  l_mode= PANGO_WRAP_WORD;

  do
  {

    l_rc= strcmp("CHAR", i_mode);

    if (0 == l_rc)
    {
      l_mode= PANGO_WRAP_CHAR;
      break;
    }

    l_rc= strcmp("WORDCHAR", i_mode);

    if (0 == l_rc)
    {
      l_mode= PANGO_WRAP_WORD_CHAR;
      break;
    }

  }while(0);

  return l_mode;
}

extern char const*
item_common_get_wrapmode_text(
  PangoWrapMode const                   i_mode)
{
  char const*                           l_text;

  switch(i_mode)
  {
    case PANGO_WRAP_WORD:
      l_text= "WORD";
      break;
    case PANGO_WRAP_CHAR:
      l_text= "CHAR";
      break;
    case PANGO_WRAP_WORD_CHAR:
      l_text= "WORDCHAR";
      break;
    default:
      l_text= "";
      break;
  }

  return l_text;
}

extern void
item_common_cairo_dash_set(
  cairo_t *const                        io_cr,
  enum line_style_type const            i_style)
{
  double                                l_dash[1]= {4.0};
  double                                l_dot[1]= {2.0};
  double                                l_dash_dot[3]= {4.0, 2.0, 2.0};

  do
  {

    cairo_set_line_cap(io_cr, CAIRO_LINE_CAP_BUTT);
    cairo_set_line_join(io_cr, CAIRO_LINE_JOIN_MITER);

    switch(i_style)
    {
      case line_style_solid:
        cairo_set_dash(io_cr, 0, 0, 0);
        break;
      case line_style_dash:
        cairo_set_dash(io_cr, l_dash, G_N_ELEMENTS(l_dash), 0);
        break;
      case line_style_dot:
        cairo_set_dash(io_cr, l_dot, G_N_ELEMENTS(l_dot), 0);
        break;
      case line_style_dash_dot:
        cairo_set_dash(io_cr, l_dash_dot, G_N_ELEMENTS(l_dash_dot), 0);
        break;
      default:
        break;
    }

  }while(0);

  return;
}

extern int
item_common_read_border(
  GError**                              o_error,
  struct item_border*const              io_border,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  int                                   l_rc;
  GdkRGBA                               l_rgba;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_exit= 0;
  l_error= 0;
  
  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_style:
          (*io_border).m_style= item_common_get_line_style(l_value);
          break;
        case element_tag_thickness:
          (*io_border).m_thickness= g_ascii_strtod(l_value, 0);
          break;
        case element_tag_color:
          l_rc= gdk_rgba_parse(&l_rgba, l_value);
          if (l_rc)
          {
            (*io_border).m_color= l_rgba;
          }
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
item_common_read_border_attr(
  GError**                              o_error,
  struct item_border_attr*const         io_border,
  struct xml *const                     i_xml)
{
  int                                   l_exit;
  char const*                           l_value;
  struct item_border                    l_border;

  l_border.m_style= (*io_border).m_style;
  l_border.m_thickness= (*io_border).m_thickness;
  l_border.m_color= (*io_border).m_color;
  l_exit= 0;

  l_value= xml_lookup_attribute_value(i_xml, "on");
  (*io_border).m_active= xml_convert_boolean(l_value);
  item_common_read_border(o_error, &l_border, i_xml);

  (*io_border).m_style= l_border.m_style;
  (*io_border).m_thickness= l_border.m_thickness;
  (*io_border).m_color= l_border.m_color;

  return l_exit;
}

extern int
item_common_read_fill(
  GError**                              o_error,
  struct item_fill*const                io_fill,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  int                                   l_rc;
  GdkRGBA                               l_rgba;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_exit= 0;
  l_error= 0;

  l_value= xml_lookup_attribute_value(i_xml, "on");
  (*io_fill).m_active= xml_convert_boolean(l_value);
  
  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_color:
          l_rc= gdk_rgba_parse(&l_rgba, l_value);
          if (l_rc)
          {
            (*io_fill).m_color= l_rgba;
          }
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
item_common_write_border(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_border const*const        i_border)
{
  char                                  l_buf[G_ASCII_DTOSTR_BUF_SIZE+1];
  int                                   l_exit;
  char*                                 l_text;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t\t<border>\n");
  fprintf(io_fp, "\t\t\t\t\t<style>%s</style>\n", item_common_get_line_style_text((*i_border).m_style));
  fprintf(io_fp, "\t\t\t\t\t<thickness>%s</thickness>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_border).m_thickness));
  l_text= gdk_rgba_to_string(&(*i_border).m_color);
  fprintf(io_fp, "\t\t\t\t\t<color>%s</color>\n", l_text);
  g_free(l_text);
  fprintf(io_fp, "\t\t\t\t</border>\n");

  return l_exit;
}

extern int
item_common_write_border_attr(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_border_attr const*const   i_border)
{
  char                                  l_buf[G_ASCII_DTOSTR_BUF_SIZE+1];
  int                                   l_exit;
  char*                                 l_text;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t\t<border on=\"%s\">\n", ((*i_border).m_active ? "true" : "false"));
  fprintf(io_fp, "\t\t\t\t\t<style>%s</style>\n", item_common_get_line_style_text((*i_border).m_style));
  fprintf(io_fp, "\t\t\t\t\t<thickness>%s</thickness>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_border).m_thickness));
  l_text= gdk_rgba_to_string(&(*i_border).m_color);
  fprintf(io_fp, "\t\t\t\t\t<color>%s</color>\n", l_text);
  g_free(l_text);
  fprintf(io_fp, "\t\t\t\t</border>\n");

  return l_exit;
}

extern int
item_common_write_fill(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_fill const*const          i_fill)
{
  int                                   l_exit;
  char*                                 l_text;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t\t<fill on=\"%s\">\n", ((*i_fill).m_active ? "true" : "false"));
  l_text= gdk_rgba_to_string(&(*i_fill).m_color);
  fprintf(io_fp, "\t\t\t\t\t<color>%s</color>\n", l_text);
  g_free(l_text);
  fprintf(io_fp, "\t\t\t\t</fill>\n");

  return l_exit;
}
