/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "barsym.h"

cairo_t*                                g_cr= 0;
cairo_surface_t*                        g_surface= 0;

static int
surface_create(
  double const                          i_width,
  double const                          i_height)
{
  int                                   l_exit;
  cairo_status_t                        l_status;

  l_exit= 0;

  do
  {

    g_surface= cairo_image_surface_create(CAIRO_FORMAT_ARGB32, i_width, i_height);

    l_status= cairo_surface_status(g_surface);

    if (CAIRO_STATUS_SUCCESS != l_status)
    {
      l_exit= l_status;
      break;
    }

    g_cr= cairo_create(g_surface);

    l_status= cairo_status(g_cr);

    if (CAIRO_STATUS_SUCCESS != l_status)
    {
      l_exit= l_status; 
      break;
    }

  }while(0);

  return l_exit;
}

static void
surface_destroy()
{
  cairo_destroy(g_cr);
  cairo_surface_destroy(g_surface);
  g_cr= 0;
  g_surface= 0;
}

static int
sample_code128(
  unsigned char const*                  i_text,
  char const*                           i_png,
  double const                          i_scale)
{
  int                                   l_exit;
  struct barsym                         l_barsym;
  int                                   l_length;
  double                                l_width;

  l_exit= 0;

  do
  {

    l_length= strlen((char*)i_text);

    l_exit= barsym_code128_verify(i_text, l_length);

    if (l_exit)
    {
      break;
    }

    barsym_code128_default(&l_barsym);
    l_width= barsym_code128_get_width(&l_barsym, i_text, l_length);
    l_exit= surface_create(l_width * i_scale, l_barsym.m_height * i_scale);

    if (l_exit)
    {
      break;
    }

    l_barsym.m_cr= g_cr;

    cairo_scale(l_barsym.m_cr, i_scale, i_scale);
    barsym_set_font(&l_barsym);
    barsym_code128(&l_barsym, i_text, l_length);

    cairo_surface_write_to_png(g_surface, i_png);

  }while(0);

  surface_destroy();

  return l_exit;
}

static int
sample_code39(
  unsigned char const*                  i_text,
  char const*                           i_png,
  double const                          i_scale)
{
  struct barsym                         l_barsym;
  int                                   l_exit;
  unsigned                              l_length;
  double                                l_width;

  l_exit= 0;

  do
  {

    l_length= strlen((char*)i_text);

    l_exit= barsym_code39_verify(i_text, l_length);

    if (l_exit)
    {
      break;
    }

    barsym_code39_default(&l_barsym);
    l_width= barsym_code39_get_width(&l_barsym, i_text, l_length);
    l_exit= surface_create(l_width * i_scale, l_barsym.m_height * i_scale);

    if (l_exit)
    {
      break;
    }

    l_barsym.m_cr= g_cr;

    cairo_scale(l_barsym.m_cr, i_scale, i_scale);
    barsym_set_font(&l_barsym);
    barsym_code39(&l_barsym, i_text, l_length);

    cairo_surface_write_to_png(g_surface, i_png);

  }while(0);

  surface_destroy();

  return l_exit;
}

static int
sample_code93(
  unsigned char const*                  i_text,
  char const*                           i_png,
  double const                          i_scale)
{
  struct barsym                         l_barsym;
  int                                   l_exit;
  unsigned                              l_length;
  double                                l_width;

  l_exit= 0;

  do
  {

    l_length= strlen((char*)i_text);

    l_exit= barsym_code93_verify(i_text, l_length);

    if (l_exit)
    {
      break;
    }

    barsym_code93_default(&l_barsym);
    l_width= barsym_code93_get_width(&l_barsym, i_text, l_length);
    l_exit= surface_create(l_width * i_scale, l_barsym.m_height * i_scale);

    if (l_exit)
    {
      break;
    }

    l_barsym.m_cr= g_cr;

    cairo_scale(l_barsym.m_cr, i_scale, i_scale);
    barsym_set_font(&l_barsym);
    barsym_code93(&l_barsym, i_text, l_length);

    cairo_surface_write_to_png(g_surface, i_png);

  }while(0);

  surface_destroy();

  return l_exit;
}

static int
sample_codabar(
  unsigned char const*                  i_text,
  char const*                           i_png,
  double const                          i_scale)
{
  struct barsym                         l_barsym;
  int                                   l_exit;
  unsigned                              l_length;
  double                                l_width;

  l_exit= 0;

  do
  {

    l_length= strlen((char*)i_text);

    l_exit= barsym_codabar_verify(i_text, l_length);

    if (l_exit)
    {
      break;
    }

    barsym_codabar_default(&l_barsym);
    l_width= barsym_codabar_get_width(&l_barsym, i_text, l_length);
    l_exit= surface_create(l_width * i_scale, l_barsym.m_height * i_scale);

    if (l_exit)
    {
      break;
    }

    l_barsym.m_cr= g_cr;

    cairo_scale(l_barsym.m_cr, i_scale, i_scale);
    barsym_set_font(&l_barsym);
    barsym_codabar(&l_barsym, i_text, l_length);

    cairo_surface_write_to_png(g_surface, i_png);

  }while(0);

  surface_destroy();

  return l_exit;
}

static int
sample_ean2(
  char const                            i_digits[2],
  char const*                           i_png,
  double const                          i_scale)
{
  int                                   l_exit;
  struct barsym                         l_barsym;
  double                                l_width;

  l_exit= 0;

  do
  {

    l_exit= barsym_ean2_verify(i_digits);

    if (l_exit)
    {
      break;
    }

    barsym_ean2_default(&l_barsym);
    l_width= barsym_ean2_get_width(&l_barsym, i_digits);
    l_exit= surface_create(l_width * i_scale, l_barsym.m_height * i_scale);

    if (l_exit)
    {
      break;
    }

    l_barsym.m_cr= g_cr;

    cairo_scale(l_barsym.m_cr, i_scale, i_scale);
    barsym_set_font(&l_barsym);
    barsym_ean2(&l_barsym, i_digits);

    cairo_surface_write_to_png(g_surface, i_png);

  }while(0);

  surface_destroy();

  return l_exit;
}

static int
sample_ean5(
  char const                            i_digits[5],
  char const*                           i_png,
  double const                          i_scale)
{
  int                                   l_exit;
  struct barsym                         l_barsym;
  double                                l_width;

  l_exit= 0;

  do
  {

    l_exit= barsym_ean5_verify(i_digits);

    if (l_exit)
    {
      break;
    }

    barsym_ean5_default(&l_barsym);
    l_width= barsym_ean5_get_width(&l_barsym, i_digits);
    l_exit= surface_create(l_width * i_scale, l_barsym.m_height * i_scale);

    if (l_exit)
    {
      break;
    }

    l_barsym.m_cr= g_cr;

    cairo_scale(l_barsym.m_cr, i_scale, i_scale);
    barsym_set_font(&l_barsym);
    barsym_ean5(&l_barsym, i_digits);

    cairo_surface_write_to_png(g_surface, i_png);

  }while(0);

  surface_destroy();

  return l_exit;
}

static int
sample_ean8(
  char const                            i_digits[8],
  char const*                           i_png,
  double const                          i_scale)
{
  int                                   l_exit;
  struct barsym                         l_barsym;
  double                                l_width;

  l_exit= 0;

  do
  {

    l_exit= barsym_ean8_verify(i_digits);

    if (l_exit)
    {
      break;
    }

    barsym_ean8_default(&l_barsym);
    l_width= barsym_ean8_get_width(&l_barsym, i_digits);
    l_exit= surface_create(l_width * i_scale, l_barsym.m_height * i_scale);

    if (l_exit)
    {
      break;
    }

    l_barsym.m_cr= g_cr;

    cairo_scale(l_barsym.m_cr, i_scale, i_scale);
    barsym_set_font(&l_barsym);
    barsym_ean8(&l_barsym, i_digits);

    cairo_surface_write_to_png(g_surface, i_png);

  }while(0);

  surface_destroy();

  return l_exit;
}

static int
sample_ean13(
  char const                            i_digits[13],
  char const*                           i_png,
  double const                          i_scale)
{
  int                                   l_exit;
  struct barsym                         l_barsym;
  double                                l_width;

  l_exit= 0;

  do
  {

    l_exit= barsym_ean13_verify(i_digits);

    if (l_exit)
    {
      break;
    }

    barsym_ean13_default(&l_barsym);
    l_width= barsym_ean13_get_width(&l_barsym, i_digits);
    l_exit= surface_create(l_width * i_scale, l_barsym.m_height * i_scale);

    if (l_exit)
    {
      break;
    }

    l_barsym.m_cr= g_cr;

    cairo_scale(l_barsym.m_cr, i_scale, i_scale);
    barsym_set_font(&l_barsym);
    barsym_ean13(&l_barsym, i_digits);

    cairo_surface_write_to_png(g_surface, i_png);

  }while(0);

  surface_destroy();

  return l_exit;
}

static int
sample_interleaved2of5(
  char const*                           i_digits,
  char const*                           i_png,
  double const                          i_scale)
{
  int                                   l_exit;
  struct barsym                         l_barsym;
  double                                l_width;

  l_exit= 0;

  do
  {

    l_exit= barsym_interleaved2of5_verify(i_digits);

    if (l_exit)
    {
      break;
    }

    barsym_interleaved2of5_default(&l_barsym);
    l_width= barsym_interleaved2of5_get_width(&l_barsym, i_digits);
    l_exit= surface_create(l_width * i_scale, l_barsym.m_height * i_scale);

    if (l_exit)
    {
      break;
    }

    l_barsym.m_cr= g_cr;

    cairo_scale(l_barsym.m_cr, i_scale, i_scale);
    barsym_set_font(&l_barsym);
    barsym_interleaved2of5(&l_barsym, i_digits);

    cairo_surface_write_to_png(g_surface, i_png);

  }while(0);

  surface_destroy();

  return l_exit;
}

static int
sample_standard2of5(
  char const*                           i_digits,
  char const*                           i_png,
  double const                          i_scale)
{
  int                                   l_exit;
  struct barsym                         l_barsym;
  double                                l_width;

  l_exit= 0;

  do
  {

    l_exit= barsym_standard2of5_verify(i_digits);

    if (l_exit)
    {
      break;
    }

    barsym_standard2of5_default(&l_barsym);
    l_width= barsym_standard2of5_get_width(&l_barsym, i_digits);
    l_exit= surface_create(l_width * i_scale, l_barsym.m_height * i_scale);

    if (l_exit)
    {
      break;
    }

    l_barsym.m_cr= g_cr;

    cairo_scale(l_barsym.m_cr, i_scale, i_scale);
    barsym_set_font(&l_barsym);
    barsym_standard2of5(&l_barsym, i_digits);

    cairo_surface_write_to_png(g_surface, i_png);

  }while(0);

  surface_destroy();

  return l_exit;
}

int
main(
  int                                   argc, 
  char*                                 argv[])
{
  int                                   l_exit;

  l_exit= 0;
  
  sample_code128((unsigned char*)"Wikipedia", "code128_Wikipedia.png", 2.0);
  sample_code39((unsigned char*)"CODE39", "code39_CODE39.png", 2.0);
  sample_code93((unsigned char*)"WIKIPEDIA", "code93_WIKIPEDIA.png", 2.0);
  sample_codabar((unsigned char*)"A40156B", "codabar_A40156B.png", 2.0);
  sample_ean2("34", "ean2_34.png", 2.0);
  sample_ean5("54495", "ean5_54495.png", 2.0);
  sample_ean8("96385074", "ean8_96385074.png", 2.0);
  sample_ean13("5901234123457", "ean13_5901234123457.png", 2.0);
  sample_interleaved2of5("12345670", "2of5i_12345670.png", 2.0);
  sample_standard2of5("12345670", "2of5_12345670.png", 2.0);

  return l_exit;
}
