/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "barsym.h"

enum parity_type
{
  ODD= 0,
  EVEN= 1
};

static enum parity_type const g_parity2[4][2]= {
/* Remainder */ /*parity */
/*         0 */ /*    LL */{ODD,  ODD},
/*         1 */ /*    LG */{ODD,  EVEN},
/*         2 */ /*    GL */{EVEN, ODD},
/*         3 */ /*    GG */{EVEN, EVEN}
};

static enum parity_type const g_parity5[10][5]= {
/* Checksum */ /*	parity  */
/*        0 */ /* "GGLLL" */{EVEN, EVEN, ODD,  ODD,  ODD},
/*        1 */ /* "GLGLL" */{EVEN, ODD,  EVEN, ODD,  ODD},
/*        2 */ /* "GLLGL" */{EVEN, ODD,  ODD,  EVEN, ODD},
/*        3 */ /* "GLLLG" */{EVEN, ODD,  ODD,  ODD,  EVEN},
/*        4 */ /* "LGGLL" */{ODD,  EVEN, EVEN, ODD,  ODD},
/*        5 */ /* "LLGGL" */{ODD,  ODD,  EVEN, EVEN, ODD},
/*        6 */ /* "LLLGG" */{ODD,  ODD,  ODD,  EVEN, EVEN},
/*        7 */ /* "LGLGL" */{ODD,  EVEN, ODD,  EVEN, ODD},
/*        8 */ /* "LGLLG" */{ODD,  EVEN, ODD,  ODD,  EVEN},
/*        9 */ /* "LLGLG" */{ODD,  ODD,  EVEN, ODD,  EVEN}
};

static enum parity_type const g_parity[10][6]= {
  /* digit */ /* parity   */
  /*    0  */	/* "LLLLLL" */{ODD, ODD,  ODD,  ODD,  ODD,  ODD},
  /*    1  */	/* "LLGLGG"	*/{ODD, ODD,  EVEN, ODD,  EVEN, EVEN},
  /*    2  */	/* "LLGGLG" */{ODD, ODD,  EVEN, EVEN, ODD,  EVEN},
  /*    3  */	/* "LLGGGL"	*/{ODD, ODD,  EVEN, EVEN, EVEN, ODD},
  /*    4  */	/* "LGLLGG" */{ODD, EVEN, ODD,  ODD,  EVEN, EVEN},
  /*    5  */	/* "LGGLLG" */{ODD, EVEN, EVEN, ODD,  ODD,  EVEN},
  /*    6  */	/* "LGGGLL" */{ODD, EVEN, EVEN, EVEN, ODD,  ODD},
  /*    7  */	/* "LGLGLG" */{ODD, EVEN, ODD,  EVEN, ODD,  EVEN},
  /*    8  */	/* "LGLGGL" */{ODD, EVEN, ODD,  EVEN, EVEN, ODD},
  /*    9  */	/* "LGGLGL" */{ODD, EVEN, EVEN, ODD,  EVEN, ODD}
};

static char const* g_encode_left[10][2]={
   /*digit*/ /*L-code*/ /*G-code*/
  {/*   0 */ "0001101", "0100111"},
  {/*   1 */ "0011001", "0110011"},
  {/*   2 */ "0010011", "0011011"},
  {/*   3 */ "0111101", "0100001"},
  {/*   4 */ "0100011", "0011101"},
  {/*   5 */ "0110001", "0111001"},
  {/*   6 */ "0101111", "0000101"},
  {/*   7 */ "0111011", "0010001"},
  {/*   8 */ "0110111", "0001001"},
  {/*   9 */ "0001011", "0010111"}
};

static char const* g_encode_right[10]={
   /*digit*/ /*R-code*/
  /*   0 */ "1110010",
  /*   1 */ "1100110",
  /*   2 */ "1101100",
  /*   3 */ "1000010",
  /*   4 */ "1011100",
  /*   5 */ "1001110",
  /*   6 */ "1010000",
  /*   7 */ "1000100",
  /*   8 */ "1001000",
  /*   9 */ "1110100"
};

static unsigned
barsym_ean2_get_remainder(
  char const                            i_digits[2])
{
  unsigned                              l_digit1;
  unsigned                              l_digit2;
  unsigned                              l_value;
  unsigned                              l_rem;

  l_digit1= (i_digits[0] - 0x30) * 10;
  l_digit2= (i_digits[1] - 0x30);
  l_value= l_digit1 + l_digit2;
  l_rem= l_value % 4;

  return l_rem;
}

extern void
barsym_ean2(
  struct barsym*const                   io_barsym,
  char const                            i_digits[2])
{
  unsigned                              l_digit;
  char const*                           l_encode;
  enum parity_type                      l_parity;
  double                                l_pos;
  unsigned                              l_remainder;
  unsigned                              l_slot;
  char                                  l_text[3];

  l_remainder= barsym_ean2_get_remainder(i_digits);

  (*io_barsym).m_pos_x+= (*io_barsym).m_quiet_zone_left;

  barsym_write_bar(io_barsym, "01011", 0);
  l_pos= (*io_barsym).m_pos_x;

  l_slot= 0;

  do
  {

    l_digit= i_digits[l_slot] - 0x30;    
    l_parity= g_parity2[l_remainder][l_slot];
    l_encode= g_encode_left[l_digit][l_parity];
    barsym_write_bar(io_barsym, l_encode, 0);

    if (1 == l_slot)
    {
      break;
    }

    barsym_write_bar(io_barsym, "01", 0);

    l_slot++;

  }while(1);

  cairo_move_to((*io_barsym).m_cr, l_pos, POINTS_PER_INCH * 0.10);
  memcpy(l_text, i_digits, 2);
  l_text[2]= 0;
  cairo_show_text((*io_barsym).m_cr, l_text);

  (*io_barsym).m_pos_x+= (*io_barsym).m_quiet_zone_right;

  return;
}

extern double
barsym_ean2_get_width(
  struct barsym*const                   io_barsym,
  char const                            i_digits[2])
{
  double                                l_width;

  l_width= (*io_barsym).m_quiet_zone_left;
  l_width+= (*io_barsym).m_quiet_zone_right;
  l_width+= ((2+2) * (*io_barsym).m_bar_width * 7.0);

  return l_width;
}

extern int
barsym_ean2_verify(
  char const                            i_digits[2])
{
  int                                   l_exit;
  
  l_exit= barsym_verify_digits(i_digits, 2);

  return l_exit;
}

static unsigned
barsym_ean5_get_checksum(
  char const                            i_digits[5])
{
  unsigned                              l_check[5];
  unsigned                              l_sum;
  unsigned                              l_rem;

  l_check[0]= (i_digits[0] - 0x30) * 3;
  l_check[1]= (i_digits[1] - 0x30) * 9;
  l_check[2]= (i_digits[2] - 0x30) * 3;
  l_check[3]= (i_digits[3] - 0x30) * 9;
  l_check[4]= (i_digits[4] - 0x30) * 3;
  l_sum= l_check[0] + l_check[1] + l_check[2] + l_check[3] + l_check[4];
  l_rem= l_sum % 10;

  return l_rem;
}

extern void
barsym_ean5(
  struct barsym*const                   io_barsym,
  char const                            i_digits[5])
{
  unsigned                              l_checksum;
  unsigned                              l_digit;
  char const*                           l_encode;
  enum parity_type                      l_parity;
  double                                l_pos;
  unsigned                              l_slot;
  char                                  l_text[6];

  (*io_barsym).m_pos_x+= (*io_barsym).m_quiet_zone_left;

  barsym_write_bar(io_barsym, "01011", 0);
  l_pos= (*io_barsym).m_pos_x;

  l_checksum= barsym_ean5_get_checksum(i_digits);

  l_slot= 0;

  do
  {

    l_digit= i_digits[l_slot] - 0x30;    
    l_parity= g_parity5[l_checksum][l_slot];
    l_encode= g_encode_left[l_digit][l_parity];
    barsym_write_bar(io_barsym, l_encode, 0);

    if (4 == l_slot)
    {
      break;
    }

    barsym_write_bar(io_barsym, "01", 0);
    l_slot++;

  }while(1);

  cairo_move_to((*io_barsym).m_cr, l_pos, POINTS_PER_INCH * .1);
  memcpy(l_text, i_digits, 5);
  l_text[5]= 0;
  cairo_show_text((*io_barsym).m_cr, l_text);

  (*io_barsym).m_pos_x+= (*io_barsym).m_quiet_zone_right;

  return;
}

extern double
barsym_ean5_get_width(
  struct barsym*const                   io_barsym,
  char const                            i_digits[5])
{
  double                                l_width;

  l_width= (*io_barsym).m_quiet_zone_left;
  l_width+= (*io_barsym).m_quiet_zone_right;
  l_width+= ((2+5) * (*io_barsym).m_bar_width * 7.0);

  return l_width;
}

extern int
barsym_ean5_verify(
  char const                            i_digits[5])
{
  int                                   l_exit;
  
  l_exit= barsym_verify_digits(i_digits, 5);

  return l_exit;
}

extern void
barsym_ean8(
  struct barsym*const                   io_barsym,
  char const                            i_digits[8])
{
  unsigned                              l_digit;
  char const*                           l_encode;
  double                                l_pos[2];
  unsigned                              l_slot;
  char                                  l_text[5];

  (*io_barsym).m_pos_x+= (*io_barsym).m_quiet_zone_left;
  barsym_write_bar(io_barsym, "101", BARSYM_GUARD);
  l_pos[0]= (*io_barsym).m_pos_x;

  for (l_slot= 0; 4 > l_slot; l_slot++)
  {
    l_digit= i_digits[l_slot] - 0x30;
    l_encode= g_encode_left[l_digit][ODD];
    barsym_write_bar(io_barsym, l_encode, 0);
  }

  barsym_write_bar(io_barsym, "01010", BARSYM_GUARD);

  l_pos[1]= (*io_barsym).m_pos_x;

  for (l_slot= 0; 4 > l_slot; l_slot++)
  {
    l_digit= i_digits[4+l_slot] - 0x30;
    l_encode= g_encode_right[l_digit];
    barsym_write_bar(io_barsym, l_encode, 0);
  }

  barsym_write_bar(io_barsym, "101", BARSYM_GUARD);

  memset(l_text, 0, sizeof(l_text));

  cairo_move_to((*io_barsym).m_cr, l_pos[0], (*io_barsym).m_height);
  memcpy(l_text, i_digits, 4);
  cairo_show_text((*io_barsym).m_cr, l_text);
    
  cairo_move_to((*io_barsym).m_cr, l_pos[1], (*io_barsym).m_height);
  memcpy(l_text, i_digits+4, 4);
  cairo_show_text((*io_barsym).m_cr, l_text);

  (*io_barsym).m_pos_x+= (*io_barsym).m_quiet_zone_right;

  return;
}

extern double
barsym_ean8_get_width(
  struct barsym*const                   io_barsym,
  char const                            i_digits[8])
{
  double                                l_width;

  l_width= (*io_barsym).m_quiet_zone_left;
  l_width+= (*io_barsym).m_quiet_zone_right;
  l_width+= ((3+8) * (*io_barsym).m_bar_width * 7.0);

  return l_width;
}

extern int
barsym_ean8_verify(
  char const                            i_digits[8])
{
  int                                   l_exit;
  
  l_exit= barsym_verify_digits(i_digits, 8);

  return l_exit;
}

extern void
barsym_ean13(
  struct barsym*const                   io_barsym,
  char const                            i_digits[13])
{
  unsigned                              l_digit;
  char const*                           l_encode;
  enum parity_type                      l_parity;
  unsigned                              l_parity_slot;
  double                                l_pos[3];
  unsigned                              l_slot;
  char                                  l_text[7];

  l_pos[0]= (*io_barsym).m_pos_x;
  (*io_barsym).m_pos_x+= (*io_barsym).m_quiet_zone_left;

  barsym_write_bar(io_barsym, "101", BARSYM_GUARD);

  l_pos[1]= (*io_barsym).m_pos_x;

  l_parity_slot= i_digits[0] - 0x30;
  for (l_slot= 0; 6 > l_slot; l_slot++)
  {
    l_digit= i_digits[1+l_slot] - 0x30;
    l_parity= g_parity[l_parity_slot][l_slot];
    l_encode= g_encode_left[l_digit][l_parity];
    barsym_write_bar(io_barsym, l_encode, 0);
  }

  barsym_write_bar(io_barsym, "01010", BARSYM_GUARD);

  l_pos[2]= (*io_barsym).m_pos_x;

  for (l_slot= 0; 6 > l_slot; l_slot++)
  {
    l_digit= i_digits[7+l_slot] - 0x30;
    l_encode= g_encode_right[l_digit];
    barsym_write_bar(io_barsym, l_encode, 0);
  }

  barsym_write_bar(io_barsym, "101", BARSYM_GUARD);

  memset(l_text, 0, sizeof(l_text));

  cairo_move_to((*io_barsym).m_cr, l_pos[0], (*io_barsym).m_height);
  l_text[0]= i_digits[0];
  cairo_show_text((*io_barsym).m_cr, l_text);
    
  cairo_move_to((*io_barsym).m_cr, l_pos[1], (*io_barsym).m_height);
  memcpy(l_text, i_digits+1, 6);
  cairo_show_text((*io_barsym).m_cr, l_text);
    
  cairo_move_to((*io_barsym).m_cr, l_pos[2], (*io_barsym).m_height);
  memcpy(l_text, i_digits+7, 6);
  cairo_show_text((*io_barsym).m_cr, l_text);

  (*io_barsym).m_pos_x+= (*io_barsym).m_quiet_zone_right;

  return;
}

extern double
barsym_ean13_get_width(
  struct barsym*const                   io_barsym,
  char const                            i_digits[13])
{
  double                                l_width;

  l_width= (*io_barsym).m_quiet_zone_left;
  l_width+= (*io_barsym).m_quiet_zone_right;
  l_width+= ((3+13) * (*io_barsym).m_bar_width * 7.0);

  return l_width;
}

extern int
barsym_ean13_verify(
  char const                            i_digits[13])
{
  int                                   l_exit;
  
  l_exit= barsym_verify_digits(i_digits, 13);

  return l_exit;
}
