/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "barsym_code93.h"

enum ascii_ctl
{
  NUL = 0x00,
  SOH,
  STX,
  ETX,
  EOT,
  ENQ,
  ACK,
  BEL,
  BS,
  HT,
  LF,
  VT,
  FF,
  CR,
  SO,
  SI,
  DLE,
  DC1,
  DC2,
  DC3,
  DC4,
  NAK,
  SYN,
  ETB,
  CAN,
  EM,
  SUB,
  ESC,
  FS,
  GS,
  RS,
  YS,
  DEL=0x7F
};

enum barsym_code39_shift
{
  shift_none= 0,
  shift_dollar= 43,
  shift_percent= 44, 
  shift_slash= 45,
  shift_plus= 46
};

#define SHIFT_DOLLAR                    "100100110" /* $ */
#define SHIFT_SLASH                     "111010110" /* / */
#define SHIFT_PLUS                      "100110010" /* + */
#define SHIFT_PERCENT                   "111011010" /* % */

char const* g_code93_encode[47]={
/* IDX  CHR */
/*  0 	'0' */  "100010100",
/*  1 	'1' */  "101001000",
/*  2 	'2' */  "101000100",
/*  3 	'3' */  "101000010",
/*  4 	'4' */  "100101000",
/*  5 	'5' */  "100100100",
/*  6 	'6' */  "100100010",
/*  7 	'7' */  "101010000",
/*  8 	'8' */  "100010010",
/*  9 	'9' */  "100001010",
/* 10 	'A' */  "110101000",
/* 11 	'B' */  "110100100",
/* 12 	'C' */  "110100010",
/* 13 	'D' */  "110010100",
/* 14 	'E' */  "110010010",
/* 15 	'F' */  "110001010",
/* 16 	'G' */  "101101000",
/* 17 	'H' */  "101100100",
/* 18 	'I' */  "101100010",
/* 19 	'J' */  "100110100",
/* 20 	'K' */  "100011010",
/* 21 	'L' */  "101011000",
/* 22 	'M' */  "101001100",
/* 23 	'N' */  "101000110",
/* 24 	'O' */  "100101100",
/* 25 	'P' */  "100010110",
/* 26 	'Q' */  "110110100",
/* 27 	'R' */  "110110010",
/* 28 	'S' */  "110101100",
/* 29 	'T' */  "110100110",
/* 30 	'U' */  "110010110",
/* 31 	'V' */  "110011010",
/* 32 	'W' */  "101101100",
/* 33 	'X' */  "101100110",
/* 34 	'Y' */  "100110110",
/* 35 	'Z' */  "100111010",
/* 36 	'-' */  "100101110",
/* 37 	'.' */  "111010100",
/* 38 	' ' */  "111010010",
/* 39 	'$' */  "111001010",
/* 40 	'/' */  "101101110",
/* 41 	'+' */  "101110110",
/* 42 	'%' */  "110101110",
/* 43 	($) */  "100100110",
/* 44 	(%) */  "111011010",
/* 45 	(/) */  "111010110",
/* 46 	(+) */  "100110010"
};

static int
barsym_code39_encode_ext(
  struct barsym_code39_encode*const     o_encode,
  unsigned char const                   i_code)
{
  int                                   l_exit;

  l_exit= 0;

  switch(i_code)
  {
  case NUL:
    /* %U */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 30;
    (*o_encode).m_tuple[1].m_encode= "110010110"; 
    (*o_encode).m_count= 2;
    break;
  case SOH:
    /* $A */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 10;
    (*o_encode).m_tuple[1].m_encode= "110101000";
    (*o_encode).m_count= 2;
    break;
  case STX:
    /* $B */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 11;
    (*o_encode).m_tuple[1].m_encode= "110100100";
    (*o_encode).m_count= 2;
    break;
  case ETX:
    /* $C */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 12;
    (*o_encode).m_tuple[1].m_encode= "110100010";
    (*o_encode).m_count= 2;
    break;
  case EOT:
    /* $D */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 13;
    (*o_encode).m_tuple[1].m_encode= "110010100";
    (*o_encode).m_count= 2;
    break;
  case ENQ:
    /* $E */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 14;
    (*o_encode).m_tuple[1].m_encode= "110010010";
    (*o_encode).m_count= 2;
    break;
  case ACK:
    /* $F */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 15;
    (*o_encode).m_tuple[1].m_encode= "110001010";
    (*o_encode).m_count= 2;
    break;
  case BEL:
    /* $G */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 16;
    (*o_encode).m_tuple[1].m_encode= "101101000";
    (*o_encode).m_count= 2;
    break;
  case BS:
    /* $H */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 17;
    (*o_encode).m_tuple[1].m_encode= "101100100";
    (*o_encode).m_count= 2;
    break;
  case HT:
    /* $I */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 18;
    (*o_encode).m_tuple[1].m_encode= "101100010";
    (*o_encode).m_count= 2;
    break;
  case LF:
    /* $J */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 19;
    (*o_encode).m_tuple[1].m_encode= "100110100";
    (*o_encode).m_count= 2;
    break;
  case VT:
    /* $K */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 20;
    (*o_encode).m_tuple[1].m_encode= "100011010";
    (*o_encode).m_count= 2;
    break;
  case FF:
    /* $L */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 21;
    (*o_encode).m_tuple[1].m_encode= "101011000";
    (*o_encode).m_count= 2;
    break;
  case CR:
    /* $M */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 22;
    (*o_encode).m_tuple[1].m_encode= "101001100";
    (*o_encode).m_count= 2;
    break;
  case SO:
    /* $N */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 23;
    (*o_encode).m_tuple[1].m_encode= "101000110";
    (*o_encode).m_count= 2;
    break;
  case SI:
    /* $O */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 24;
    (*o_encode).m_tuple[1].m_encode= "100101100";
    (*o_encode).m_count= 2;
    break;
  case DLE:
    /* $P */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 25;
    (*o_encode).m_tuple[1].m_encode= "100010110";
    (*o_encode).m_count= 2;
    break;
  case DC1:
    /* $Q */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 26;
    (*o_encode).m_tuple[1].m_encode= "110110100";
    (*o_encode).m_count= 2;
    break;
  case DC2:
    /* $R */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 27;
    (*o_encode).m_tuple[1].m_encode= "110110010";
    (*o_encode).m_count= 2;
    break;
  case DC3:
    /* $S */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 28;
    (*o_encode).m_tuple[1].m_encode= "110101100";
    (*o_encode).m_count= 2;
    break;
  case DC4:
    /* $T */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 29;
    (*o_encode).m_tuple[1].m_encode= "110100110";
    (*o_encode).m_count= 2;
    break;
  case NAK:
    /* $U */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 30;
    (*o_encode).m_tuple[1].m_encode= "110010110";
    (*o_encode).m_count= 2;
    break;
  case SYN:
    /* $V */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 31;
    (*o_encode).m_tuple[1].m_encode= "110011010";
    (*o_encode).m_count= 2;
    break;
  case ETB:
    /* $W */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 32;
    (*o_encode).m_tuple[1].m_encode= "101101100";
    (*o_encode).m_count= 2;
    break;
  case CAN:
    /* $X */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 33;
    (*o_encode).m_tuple[1].m_encode= "101100110";
    (*o_encode).m_count= 2;
    break;
  case EM:
    /* $Y */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 34;
    (*o_encode).m_tuple[1].m_encode= "100110110";
    (*o_encode).m_count= 2;
    break;
  case SUB:
    /* $Z */
    (*o_encode).m_tuple[0].m_value= shift_dollar;
    (*o_encode).m_tuple[0].m_encode= SHIFT_DOLLAR;
    (*o_encode).m_tuple[1].m_value= 35;
    (*o_encode).m_tuple[1].m_encode= "100111010";
    (*o_encode).m_count= 2;
    break;
  case ESC:
    /* %A */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 10;
    (*o_encode).m_tuple[1].m_encode= "110101000";
    (*o_encode).m_count= 2;
    break;
  case FS:
    /* %B */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 11;
    (*o_encode).m_tuple[1].m_encode= "110100100";
    (*o_encode).m_count= 2;
    break;
  case GS:
    /* %C */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 12;
    (*o_encode).m_tuple[1].m_encode= "110100010"; 
    (*o_encode).m_count= 2;
    break;
  case RS:
    /* %D */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 13;
    (*o_encode).m_tuple[1].m_encode= "110010100";
    (*o_encode).m_count= 2;
    break;
  case YS:
    /* %E */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 14;
    (*o_encode).m_tuple[1].m_encode= "110010010";
    (*o_encode).m_count= 2;
    break;
  case '!':
    /* /A */
    (*o_encode).m_tuple[0].m_value= shift_slash;
    (*o_encode).m_tuple[0].m_encode= SHIFT_SLASH;
    (*o_encode).m_tuple[1].m_value= 10;
    (*o_encode).m_tuple[1].m_encode= "110101000";
    (*o_encode).m_count= 2;
    break;
  case '"': 	
    /* /B */
    (*o_encode).m_tuple[0].m_value= shift_slash;
    (*o_encode).m_tuple[0].m_encode= SHIFT_SLASH;
    (*o_encode).m_tuple[1].m_value= 11;
    (*o_encode).m_tuple[1].m_encode= "110100100";
    (*o_encode).m_count= 2;
    break;
  case '#':
    /* /C */
    (*o_encode).m_tuple[0].m_value= shift_slash;
    (*o_encode).m_tuple[0].m_encode= SHIFT_SLASH;
    (*o_encode).m_tuple[1].m_value= 12;
    (*o_encode).m_tuple[1].m_encode= "110100010";
    (*o_encode).m_count= 2;
    break;
  case '&':
    /* /F */
    (*o_encode).m_tuple[0].m_value= shift_slash;
    (*o_encode).m_tuple[0].m_encode= SHIFT_SLASH;
    (*o_encode).m_tuple[1].m_value= 15;
    (*o_encode).m_tuple[1].m_encode= "110001010";
    (*o_encode).m_count= 2;
    break;
  case '\'':
    /* /G */
    (*o_encode).m_tuple[0].m_value= shift_slash;
    (*o_encode).m_tuple[0].m_encode= SHIFT_SLASH;
    (*o_encode).m_tuple[1].m_value= 16;
    (*o_encode).m_tuple[1].m_encode= "101101000";
    (*o_encode).m_count= 2;
    break;
  case '(':
    /* /H */
    (*o_encode).m_tuple[0].m_value= shift_slash;
    (*o_encode).m_tuple[0].m_encode= SHIFT_SLASH;
    (*o_encode).m_tuple[1].m_value= 17;
    (*o_encode).m_tuple[1].m_encode= "101100100";
    (*o_encode).m_count= 2;
    break;
  case ')':
    /* /I */
    (*o_encode).m_tuple[0].m_value= shift_slash;
    (*o_encode).m_tuple[0].m_encode= SHIFT_SLASH;
    (*o_encode).m_tuple[1].m_value= 18;
    (*o_encode).m_tuple[1].m_encode= "101100010";
    (*o_encode).m_count= 2;
    break;
  case '*':
    /* /J */
    (*o_encode).m_tuple[0].m_value= shift_slash;
    (*o_encode).m_tuple[0].m_encode= SHIFT_SLASH;
    (*o_encode).m_tuple[1].m_value= 19;
    (*o_encode).m_tuple[1].m_encode= "100110100";
    (*o_encode).m_count= 2;
    break;
  case ',':
    /* /L */
    (*o_encode).m_tuple[0].m_value= shift_slash;
    (*o_encode).m_tuple[0].m_encode= SHIFT_SLASH;
    (*o_encode).m_tuple[1].m_value= 21;
    (*o_encode).m_tuple[1].m_encode= "101011000";
    (*o_encode).m_count= 2;
    break;
  case ':':
    /* /Z */
    (*o_encode).m_tuple[0].m_value= shift_slash;
    (*o_encode).m_tuple[0].m_encode= SHIFT_SLASH;
    (*o_encode).m_tuple[1].m_value= 35;
    (*o_encode).m_tuple[1].m_encode= "100111010";
    (*o_encode).m_count= 2;
    break;
  case ';':
    /* %F */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 15;
    (*o_encode).m_tuple[1].m_encode= "110001010";
    (*o_encode).m_count= 2;
    break;
  case '<':
    /* %G */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 16;
    (*o_encode).m_tuple[1].m_encode= "101101000";
    (*o_encode).m_count= 2;
    break;
  case '=':
    /* %H */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 17;
    (*o_encode).m_tuple[1].m_encode= "101100100";
    (*o_encode).m_count= 2;
    break;
  case '>':
    /* %I */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 18;
    (*o_encode).m_tuple[1].m_encode= "101100010";
    (*o_encode).m_count= 2;
    break;
  case '?':
    /* %J */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 19;
    (*o_encode).m_tuple[1].m_encode= "100110100";
    (*o_encode).m_count= 2;
    break;
  case '@':
    /* %V */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 31;
    (*o_encode).m_tuple[1].m_encode= "110011010";
    (*o_encode).m_count= 2;
    break;
  case '[':
    /* %K */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 20;
    (*o_encode).m_tuple[1].m_encode= "100011010";
    (*o_encode).m_count= 2;
    break;
  case '\\':
    /* %L */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 21;
    (*o_encode).m_tuple[1].m_encode= "101011000";
    (*o_encode).m_count= 2;
    break;
  case ']':
    /* %M */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 22;
    (*o_encode).m_tuple[1].m_encode= "101001100";
    (*o_encode).m_count= 2;
    break;
  case '^':
    /* %N */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 23;
    (*o_encode).m_tuple[1].m_encode= "101000110";
    (*o_encode).m_count= 2;
    break;
  case '_':
    /* %O */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 24;
    (*o_encode).m_tuple[1].m_encode= "100101100";
    (*o_encode).m_count= 2;
    break;
  case '`':
    /* %W */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 32;
    (*o_encode).m_tuple[1].m_encode= "101101100";
    (*o_encode).m_count= 2;
    break;
  case 'a':
    /* +A */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 10;
    (*o_encode).m_tuple[1].m_encode= "110101000";
    (*o_encode).m_count= 2;
    break;
  case 'b':
    /* +B */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 11;
    (*o_encode).m_tuple[1].m_encode= "110100100";
    (*o_encode).m_count= 2;
    break;
  case 'c':
    /* +C */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 12;
    (*o_encode).m_tuple[1].m_encode= "110100010";
    (*o_encode).m_count= 2;
    break;
  case 'd':
    /* +D */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 13;
    (*o_encode).m_tuple[1].m_encode= "110010100";
    (*o_encode).m_count= 2;
    break;
  case 'e':
    /* +E */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 14;
    (*o_encode).m_tuple[1].m_encode= "110010010";
    (*o_encode).m_count= 2;
    break;
  case 'f':
    /* +F */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 15;
    (*o_encode).m_tuple[1].m_encode= "110001010";
    (*o_encode).m_count= 2;
    break;
  case 'g':
    /* +G */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 16;
    (*o_encode).m_tuple[1].m_encode= "101101000";
    (*o_encode).m_count= 2;
    break;
  case 'h':
    /* +H */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 17;
    (*o_encode).m_tuple[1].m_encode= "101100100";
    (*o_encode).m_count= 2;
    break;
  case 'i':
    /* +I */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 18;
    (*o_encode).m_tuple[1].m_encode= "101100010";
    (*o_encode).m_count= 2;
    break;
  case 'j':
    /* +J */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 19;
    (*o_encode).m_tuple[1].m_encode= "100110100";
    (*o_encode).m_count= 2;
    break;
  case 'k':
    /* +K */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 20;
    (*o_encode).m_tuple[1].m_encode= "100011010";
    (*o_encode).m_count= 2;
    break;
  case 'l':
    /* +L */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 21;
    (*o_encode).m_tuple[1].m_encode= "101011000";
    (*o_encode).m_count= 2;
    break;
  case 'm':
    /* +M */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 22;
    (*o_encode).m_tuple[1].m_encode= "101001100";
    (*o_encode).m_count= 2;
    break;
  case 'n':
    /* +N */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 23;
    (*o_encode).m_tuple[1].m_encode= "101000110";
    (*o_encode).m_count= 2;
    break;
  case 'o':
    /* +O */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 24;
    (*o_encode).m_tuple[1].m_encode= "100101100";
    (*o_encode).m_count= 2;
    break;
  case 'p':
    /* +P */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 25;
    (*o_encode).m_tuple[1].m_encode= "100010110";
    (*o_encode).m_count= 2;
    break;
  case 'q':
    /* +Q */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 26;
    (*o_encode).m_tuple[1].m_encode= "110110100";
    (*o_encode).m_count= 2;
    break;
  case 'r':
    /* +R */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 27;
    (*o_encode).m_tuple[1].m_encode= "110110010";
    (*o_encode).m_count= 2;
    break;
  case 's':
    /* +S */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 28;
    (*o_encode).m_tuple[1].m_encode= "110101100";
    (*o_encode).m_count= 2;
    break;
  case 't':
    /* +T */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 29;
    (*o_encode).m_tuple[1].m_encode= "110100110";
    (*o_encode).m_count= 2;
    break;
  case 'u':
    /* +U */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 30;
    (*o_encode).m_tuple[1].m_encode= "110010110";
    (*o_encode).m_count= 2;
    break;
  case 'v':
    /* +V */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 31;
    (*o_encode).m_tuple[1].m_encode= "110011010";
    (*o_encode).m_count= 2;
    break;
  case 'w':
    /* +W */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 32;
    (*o_encode).m_tuple[1].m_encode= "101101100";
    (*o_encode).m_count= 2;
    break;
  case 'x':
    /* +X */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 33;
    (*o_encode).m_tuple[1].m_encode= "101100110";
    (*o_encode).m_count= 2;
    break;
  case 'y':
    /* +Y */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 34;
    (*o_encode).m_tuple[1].m_encode= "100110110";
    (*o_encode).m_count= 2;
    break;
  case 'z':
    /* +Z */
    (*o_encode).m_tuple[0].m_value= shift_plus;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PLUS;
    (*o_encode).m_tuple[1].m_value= 35;
    (*o_encode).m_tuple[1].m_encode= "100111010";
    (*o_encode).m_count= 2;
    break;
  case '{':
    /* %P */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 25;
    (*o_encode).m_tuple[1].m_encode= "100010110";
    (*o_encode).m_count= 2;
    break;
  case '|':
    /* %Q */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 26;
    (*o_encode).m_tuple[1].m_encode= "110110100";
    (*o_encode).m_count= 2;
    break;
  case '}':
    /* %R */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 27;
    (*o_encode).m_tuple[1].m_encode= "110110010";
    (*o_encode).m_count= 2;
    break;
  case '~':
    /* %S */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    (*o_encode).m_tuple[1].m_value= 28;
    (*o_encode).m_tuple[1].m_encode= "110101100";
    (*o_encode).m_count= 2;
    break;
  case DEL:
    /* %T, %X, %Y, %Z */
    /* % */
    (*o_encode).m_tuple[0].m_value= shift_percent;
    (*o_encode).m_tuple[0].m_encode= SHIFT_PERCENT;
    /* T */
    (*o_encode).m_tuple[1].m_value= 29;
    (*o_encode).m_tuple[1].m_encode= "110100110";
    /* % */
    (*o_encode).m_tuple[2].m_value= shift_percent;
    (*o_encode).m_tuple[2].m_encode= SHIFT_PERCENT;
    /* X */
    (*o_encode).m_tuple[3].m_value= 33;
    (*o_encode).m_tuple[3].m_encode= "101100110";
    /* % */
    (*o_encode).m_tuple[4].m_value= shift_percent;
    (*o_encode).m_tuple[4].m_encode= SHIFT_PERCENT;
    /* Y */
    (*o_encode).m_tuple[5].m_value= 34;
    (*o_encode).m_tuple[5].m_encode= "100110110";
    /* % */
    (*o_encode).m_tuple[6].m_value= shift_percent;
    (*o_encode).m_tuple[6].m_encode= SHIFT_PERCENT;
    /* Z */
    (*o_encode).m_tuple[7].m_value= 35;
    (*o_encode).m_tuple[7].m_encode= "100111010";
    (*o_encode).m_count= 8;
    break;
  default:
    l_exit= -1;
    break;
  }

  return l_exit;
}

extern int
barsym_code93_encode(
  struct barsym_code39_encode*const     o_encode,
  unsigned char const                   i_code)
{
  int                                   l_exit;

  memset(o_encode, 0, sizeof(*o_encode));
  l_exit= 0;

  switch(i_code)
  {
  case '0':
    (*o_encode).m_tuple[0].m_value= 0;
    (*o_encode).m_tuple[0].m_encode= "100010100";
    (*o_encode).m_count= 1;
    break;
  case '1':
    (*o_encode).m_tuple[0].m_value= 1;
    (*o_encode).m_tuple[0].m_encode= "101001000";
    (*o_encode).m_count= 1;
    break;
  case '2':
    (*o_encode).m_tuple[0].m_value= 2;
    (*o_encode).m_tuple[0].m_encode= "101000100";
    (*o_encode).m_count= 1;
    break;
  case '3':
    (*o_encode).m_tuple[0].m_value= 3;
    (*o_encode).m_tuple[0].m_encode= "101000010";
    (*o_encode).m_count= 1;
    break;
  case '4':
    (*o_encode).m_tuple[0].m_value= 4;
    (*o_encode).m_tuple[0].m_encode= "100101000";
    (*o_encode).m_count= 1;
    break;
  case '5':
    (*o_encode).m_tuple[0].m_value= 5;
    (*o_encode).m_tuple[0].m_encode= "100100100";
    (*o_encode).m_count= 1;
    break;
  case '6':
    (*o_encode).m_tuple[0].m_value= 6;
    (*o_encode).m_tuple[0].m_encode= "100100010";
    (*o_encode).m_count= 1;
    break;
  case '7':
    (*o_encode).m_tuple[0].m_value= 7;
    (*o_encode).m_tuple[0].m_encode= "101010000";
    (*o_encode).m_count= 1;
    break;
  case '8':
    (*o_encode).m_tuple[0].m_value= 8;
    (*o_encode).m_tuple[0].m_encode= "100010010";
    (*o_encode).m_count= 1;
    break;
  case '9':
    (*o_encode).m_tuple[0].m_value= 9;
    (*o_encode).m_tuple[0].m_encode= "100001010";
    (*o_encode).m_count= 1;
    break;
  case 'A':
    (*o_encode).m_tuple[0].m_value= 10;
    (*o_encode).m_tuple[0].m_encode= "110101000";
    (*o_encode).m_count= 1;
    break;
  case 'B':
    (*o_encode).m_tuple[0].m_value= 11;
    (*o_encode).m_tuple[0].m_encode= "110100100";
    (*o_encode).m_count= 1;
    break;
  case 'C':
    (*o_encode).m_tuple[0].m_value= 12;
    (*o_encode).m_tuple[0].m_encode= "110100010";
    (*o_encode).m_count= 1;
    break;
  case 'D':
    (*o_encode).m_tuple[0].m_value= 13;
    (*o_encode).m_tuple[0].m_encode= "110010100";
    (*o_encode).m_count= 1;
    break;
  case 'E':
    (*o_encode).m_tuple[0].m_value= 14;
    (*o_encode).m_tuple[0].m_encode= "110010010";
    (*o_encode).m_count= 1;
    break;
  case 'F':
    (*o_encode).m_tuple[0].m_value= 15;
    (*o_encode).m_tuple[0].m_encode= "110001010";
    (*o_encode).m_count= 1;
    break;
  case 'G':
    (*o_encode).m_tuple[0].m_value= 16;
    (*o_encode).m_tuple[0].m_encode= "101101000";
    (*o_encode).m_count= 1;
    break;
  case 'H':
    (*o_encode).m_tuple[0].m_value= 17;
    (*o_encode).m_tuple[0].m_encode= "101100100";
    (*o_encode).m_count= 1;
    break;
  case 'I':
    (*o_encode).m_tuple[0].m_value= 18;
    (*o_encode).m_tuple[0].m_encode= "101100010";
    (*o_encode).m_count= 1;
    break;
  case 'J':
    (*o_encode).m_tuple[0].m_value= 19;
    (*o_encode).m_tuple[0].m_encode= "100110100";
    (*o_encode).m_count= 1;
    break;
  case 'K':
    (*o_encode).m_tuple[0].m_value= 20;
    (*o_encode).m_tuple[0].m_encode= "100011010";
    (*o_encode).m_count= 1;
    break;
  case 'L':
    (*o_encode).m_tuple[0].m_value= 21;
    (*o_encode).m_tuple[0].m_encode= "101011000";
    (*o_encode).m_count= 1;
    break;
  case 'M':
    (*o_encode).m_tuple[0].m_value= 22;
    (*o_encode).m_tuple[0].m_encode= "101001100";
    (*o_encode).m_count= 1;
    break;
  case 'N':
    (*o_encode).m_tuple[0].m_value= 23;
    (*o_encode).m_tuple[0].m_encode= "101000110";
    (*o_encode).m_count= 1;
    break;
  case 'O':
    (*o_encode).m_tuple[0].m_value= 24;
    (*o_encode).m_tuple[0].m_encode= "100101100";
    (*o_encode).m_count= 1;
    break;
  case 'P':
    (*o_encode).m_tuple[0].m_value= 25;
    (*o_encode).m_tuple[0].m_encode= "100010110";
    (*o_encode).m_count= 1;
    break;
  case 'Q':
    (*o_encode).m_tuple[0].m_value= 26;
    (*o_encode).m_tuple[0].m_encode= "110110100";
    (*o_encode).m_count= 1;
    break;
  case 'R':
    (*o_encode).m_tuple[0].m_value= 27;
    (*o_encode).m_tuple[0].m_encode= "110110010";
    (*o_encode).m_count= 1;
    break;
  case 'S':
    (*o_encode).m_tuple[0].m_value= 28;
    (*o_encode).m_tuple[0].m_encode= "110101100";
    (*o_encode).m_count= 1;
    break;
  case 'T':
    (*o_encode).m_tuple[0].m_value= 29;
    (*o_encode).m_tuple[0].m_encode= "110100110";
    (*o_encode).m_count= 1;
    break;
  case 'U':
    (*o_encode).m_tuple[0].m_value= 30;
    (*o_encode).m_tuple[0].m_encode= "110010110";
    (*o_encode).m_count= 1;
    break;
  case 'V':
    (*o_encode).m_tuple[0].m_value= 31;
    (*o_encode).m_tuple[0].m_encode= "110011010";
    (*o_encode).m_count= 1;
    break;
  case 'W':
    (*o_encode).m_tuple[0].m_value= 32;
    (*o_encode).m_tuple[0].m_encode= "101101100";
    (*o_encode).m_count= 1;
    break;
  case 'X':
    (*o_encode).m_tuple[0].m_value= 33;
    (*o_encode).m_tuple[0].m_encode= "101100110";
    (*o_encode).m_count= 1;
    break;
  case 'Y':
    (*o_encode).m_tuple[0].m_value= 34;
    (*o_encode).m_tuple[0].m_encode= "100110110";
    (*o_encode).m_count= 1;
    break;
  case 'Z':
    (*o_encode).m_tuple[0].m_value= 35;
    (*o_encode).m_tuple[0].m_encode= "100111010";
    (*o_encode).m_count= 1;
    break;
  case '-':
    (*o_encode).m_tuple[0].m_value= 36;
    (*o_encode).m_tuple[0].m_encode= "100101110";
    (*o_encode).m_count= 1;
    break;
  case '.':
    (*o_encode).m_tuple[0].m_value= 37;
    (*o_encode).m_tuple[0].m_encode= "111010100";
    (*o_encode).m_count= 1;
    break;
  case ' ':
    (*o_encode).m_tuple[0].m_value= 38;
    (*o_encode).m_tuple[0].m_encode= "111010010";
    (*o_encode).m_count= 1;
    break;
  case '$':
    (*o_encode).m_tuple[0].m_value= 39;
    (*o_encode).m_tuple[0].m_encode= "111001010";
    (*o_encode).m_count= 1;
    break;
  case '/':
    (*o_encode).m_tuple[0].m_value= 40;
    (*o_encode).m_tuple[0].m_encode= "101101110";
    (*o_encode).m_count= 1;
    break;
  case '+':
    (*o_encode).m_tuple[0].m_value= 41;
    (*o_encode).m_tuple[0].m_encode= "101110110";
    (*o_encode).m_count= 1;
    break;
  case '%':
    (*o_encode).m_tuple[0].m_value= 42;
    (*o_encode).m_tuple[0].m_encode= "110101110";
    (*o_encode).m_count= 1;
    break;
  default:
    l_exit= barsym_code39_encode_ext(o_encode, i_code);
    break;
  }

  return l_exit;
}
