(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005-2008  Slawomir Kolodynski

    This program is free software Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES LOSS OF USE, DATA, OR PROFITS OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

section \<open>ZF set theory basics\<close>

theory ZF1 imports ZF.equalities

begin

text\<open>Standard Isabelle distribution contains lots of facts about basic set
  theory. This theory file adds some more.\<close>

subsection\<open>Lemmas in Zermelo-Fraenkel set theory\<close>

text\<open>Here we put lemmas from the set theory that we could not find in 
the standard Isabelle distribution.\<close>

text\<open>If one collection is contained in another, then we can say the same
  abot their unions.\<close>

lemma collection_contain: assumes "A\<subseteq>B" shows "\<Union>A \<subseteq> \<Union>B"
proof
  fix x assume "x \<in> \<Union>A"
  then obtain X where "x\<in>X" and "X\<in>A" by auto
  with assms show "x \<in> \<Union>B" by auto
qed


text\<open>If all sets of a nonempty collection are the same, then its union 
  is the same.\<close>

lemma ZF1_1_L1: assumes "C\<noteq>0" and "\<forall>y\<in>C. b(y) = A" 
  shows "(\<Union>y\<in>C. b(y)) = A" using assms by blast
  
text\<open>The union af all values of a constant meta-function belongs to 
the same set as the constant.\<close>

lemma ZF1_1_L2: assumes A1:"C\<noteq>0" and A2: "\<forall>x\<in>C. b(x) \<in> A" 
  and A3: "\<forall>x y. x\<in>C \<and> y\<in>C \<longrightarrow> b(x) = b(y)"
  shows "(\<Union>x\<in>C. b(x))\<in>A"
proof -
  from A1 obtain x where D1: "x\<in>C" by auto
  with A3 have "\<forall>y\<in>C. b(y) = b(x)" by blast
  with A1 have "(\<Union>y\<in>C. b(y)) = b(x)" 
    using ZF1_1_L1 by simp
  with D1 A2 show ?thesis by simp
qed

text\<open>If two meta-functions are the same on a cartesian product,
  then the subsets defined by them are the same. I am surprised Isabelle
  can not handle this automatically.\<close>

lemma ZF1_1_L4: assumes A1: "\<forall>x\<in>X.\<forall>y\<in>Y. a(x,y) = b(x,y)"
  shows "{a(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y} = {b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y}"
proof
  show "{a(x, y). \<langle>x,y\<rangle> \<in> X \<times> Y} \<subseteq> {b(x, y). \<langle>x,y\<rangle> \<in> X \<times> Y}"
  proof
    fix z assume "z \<in> {a(x, y) . \<langle>x,y\<rangle> \<in> X \<times> Y}"
    with A1 show  "z \<in> {b(x,y).\<langle>x,y\<rangle> \<in> X\<times>Y}" by auto   
  qed
  show "{b(x, y). \<langle>x,y\<rangle> \<in> X \<times> Y} \<subseteq> {a(x, y). \<langle>x,y\<rangle> \<in> X \<times> Y}"
  proof
    fix z assume "z \<in> {b(x, y). \<langle>x,y\<rangle> \<in> X \<times> Y}"
    with A1 show "z \<in> {a(x,y).\<langle>x,y\<rangle> \<in> X\<times>Y}" by auto
  qed
qed

text\<open>If two meta-functions are the same on a cartesian product,
  then the subsets defined by them are the same. 
  This is similar to \<open>ZF1_1_L4\<close>, except that
  the set definition varies over \<open>p\<in>X\<times>Y\<close> rather than 
  \<open>\<langle> x,y\<rangle>\<in>X\<times>Y\<close>.\<close>

lemma ZF1_1_L4A: assumes A1: "\<forall>x\<in>X.\<forall>y\<in>Y. a(\<langle> x,y\<rangle>) = b(x,y)"
  shows "{a(p). p \<in> X\<times>Y} = {b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y}"
proof
  { fix z assume "z \<in> {a(p). p\<in>X\<times>Y}"
    then obtain p where D1: "z=a(p)" "p\<in>X\<times>Y" by auto
    let ?x = "fst(p)" let ?y = "snd(p)"
    from A1 D1 have "z \<in> {b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y}" by auto
  } then show "{a(p). p \<in> X\<times>Y} \<subseteq> {b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y}" by blast
next 
  { fix z assume "z \<in> {b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y}"
    then obtain x y where D1: "\<langle>x,y\<rangle> \<in> X\<times>Y" "z=b(x,y)" by auto
    let ?p = "\<langle> x,y\<rangle>" 
    from A1 D1 have "?p\<in>X\<times>Y" "z = a(?p)" by auto
    then have "z \<in> {a(p). p \<in> X\<times>Y}" by auto
  } then show "{b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y} \<subseteq> {a(p). p \<in> X\<times>Y}" by blast
qed

text\<open>A lemma about inclusion in cartesian products. Included here to remember
  that we need the $U\times V \neq \emptyset$ assumption.\<close>

lemma prod_subset: assumes "U\<times>V\<noteq>0" "U\<times>V \<subseteq> X\<times>Y" shows "U\<subseteq>X" and "V\<subseteq>Y"
  using assms by auto

text\<open>A technical lemma about sections in cartesian products.\<close>

lemma section_proj: assumes "A \<subseteq> X\<times>Y" and "U\<times>V \<subseteq> A" and "x \<in> U"  "y \<in> V"
  shows "U \<subseteq> {t\<in>X. \<langle>t,y\<rangle> \<in> A}" and "V \<subseteq> {t\<in>Y. \<langle>x,t\<rangle> \<in> A}"
  using assms by auto

text\<open>If two meta-functions are the same on a set, then they define the same
  set by separation.\<close>

lemma ZF1_1_L4B: assumes "\<forall>x\<in>X. a(x) = b(x)"
  shows "{a(x). x\<in>X} = {b(x). x\<in>X}"
  using assms by simp

text\<open>A set defined by a constant meta-function is a singleton.\<close>

lemma ZF1_1_L5: assumes "X\<noteq>0" and "\<forall>x\<in>X. b(x) = c"
  shows "{b(x). x\<in>X} = {c}" using assms by blast

text\<open>Most of the time, \<open>auto\<close> does this job, but there are strange 
  cases when the next lemma is needed.\<close>

lemma subset_with_property: assumes "Y = {x\<in>X. b(x)}"
  shows "Y \<subseteq> X" 
  using assms by auto

text\<open>We can choose an element from a nonempty set.\<close>

lemma nonempty_has_element: assumes "X\<noteq>0" shows "\<exists>x. x\<in>X"
  using assms by auto

(*text{*If after removing an element from a set we get an empty set,
  then this set must be a singleton.*}

lemma rem_point_empty: assumes "a\<in>A" and "A-{a} = 0"
  shows "A = {a}" using assms by auto; *)

text\<open>In Isabelle/ZF the intersection of an empty family is 
  empty. This is exactly lemma \<open>Inter_0\<close> from Isabelle's
  \<open>equalities\<close> theory. We repeat this lemma here as it is very
  difficult to find. This is one reason we need comments before every 
  theorem: so that we can search for keywords.\<close>

lemma inter_empty_empty: shows "\<Inter>0 = 0" by (rule Inter_0)

text\<open>If an intersection of a collection is not empty, then the collection is
  not empty. We are (ab)using the fact the the intesection of empty collection 
  is defined to be empty.\<close>

lemma inter_nempty_nempty: assumes "\<Inter>A \<noteq> 0" shows "A\<noteq>0"
  using assms by auto

text\<open>For two collections $S,T$ of sets we define the product collection
  as the collections of cartesian products $A\times B$, where $A\in S, B\in T$.\<close>

definition
  "ProductCollection(T,S) \<equiv> \<Union>U\<in>T.{U\<times>V. V\<in>S}"

text\<open>The union of the product collection of collections $S,T$ is the 
  cartesian product of $\bigcup S$ and  $\bigcup T$.\<close>

lemma ZF1_1_L6: shows "\<Union> ProductCollection(S,T) = \<Union>S \<times> \<Union>T"
  using ProductCollection_def by auto

text\<open>An intersection of subsets is a subset.\<close>

lemma ZF1_1_L7: assumes A1: "I\<noteq>0" and A2: "\<forall>i\<in>I. P(i) \<subseteq> X"
  shows "( \<Inter>i\<in>I. P(i) ) \<subseteq> X"
proof -
  from A1 obtain i\<^sub>0 where "i\<^sub>0 \<in> I" by auto
  with A2 have "( \<Inter>i\<in>I. P(i) ) \<subseteq> P(i\<^sub>0)" and "P(i\<^sub>0) \<subseteq> X"
    by auto
  thus "( \<Inter>i\<in>I. P(i) ) \<subseteq> X" by auto
qed

text\<open>Isabelle/ZF has a "THE" construct that allows to define an element
  if there is only one such that is satisfies given predicate.
  In pure ZF we can express something similar using the indentity proven below.\<close>

lemma ZF1_1_L8: shows "\<Union> {x} = x" by auto

text\<open>Some properties of singletons.\<close>

lemma ZF1_1_L9: assumes A1: "\<exists>! x. x\<in>A \<and> \<phi>(x)"
  shows 
  "\<exists>a. {x\<in>A. \<phi>(x)} = {a}"
  "\<Union> {x\<in>A. \<phi>(x)} \<in> A"
  "\<phi>(\<Union> {x\<in>A. \<phi>(x)})"
proof -
  from A1 show "\<exists>a. {x\<in>A. \<phi>(x)} = {a}" by auto
  then obtain a where I: "{x\<in>A. \<phi>(x)} = {a}" by auto
  then have "\<Union> {x\<in>A. \<phi>(x)} = a" by auto
  moreover
  from I have "a \<in> {x\<in>A. \<phi>(x)}" by simp
  hence "a\<in>A" and "\<phi>(a)" by auto
  ultimately show "\<Union> {x\<in>A. \<phi>(x)} \<in> A" and "\<phi>(\<Union> {x\<in>A. \<phi>(x)})"
    by auto
qed

text\<open>A simple version of \<open> ZF1_1_L9\<close>.\<close>

corollary sigleton_extract: assumes  "\<exists>! x. x\<in>A"
  shows "(\<Union> A) \<in> A"
proof -
  from assms have "\<exists>! x. x\<in>A \<and> True" by simp
  then have "\<Union> {x\<in>A. True} \<in> A" by (rule ZF1_1_L9)
  thus "(\<Union> A) \<in> A" by simp
qed

text\<open>A criterion for when a set defined by comprehension is a singleton.\<close>

lemma singleton_comprehension: 
  assumes A1: "y\<in>X" and A2: "\<forall>x\<in>X. \<forall>y\<in>X. P(x) = P(y)"
  shows "(\<Union>{P(x). x\<in>X}) = P(y)"
proof - 
  let ?A = "{P(x). x\<in>X}"
  have "\<exists>! c. c \<in> ?A"
  proof
    from A1 show "\<exists>c. c \<in> ?A" by auto
  next
    fix a b assume "a \<in> ?A" and "b \<in> ?A"
    then obtain x t where 
      "x \<in> X" "a = P(x)" and "t \<in> X" "b = P(t)"
      by auto
    with A2 show "a=b" by blast
  qed
  then have "(\<Union>?A) \<in> ?A" by (rule sigleton_extract)
  then obtain x where "x \<in> X" and "(\<Union>?A) = P(x)"
    by auto
  from A1 A2 \<open>x \<in> X\<close> have "P(x) = P(y)"
    by blast
  with \<open>(\<Union>?A) = P(x)\<close> show "(\<Union>?A) = P(y)" by simp
qed

text\<open>Adding an element of a set to that set does not change the set.\<close>

lemma set_elem_add: assumes "x\<in>X" shows "X \<union> {x} = X" using assms 
  by auto

text\<open>Here we define a restriction of a collection of sets to a given set. 
  In romantic math this is typically denoted $X\cap M$ and means 
  $\{X\cap A : A\in M \} $. Note there is also restrict$(f,A)$ 
  defined for relations in ZF.thy.\<close>

definition
  RestrictedTo (infixl "{restricted to}" 70) where
  "M {restricted to} X \<equiv> {X \<inter> A . A \<in> M}"

text\<open>A lemma on a union of a restriction of a collection
  to a set.\<close>

lemma union_restrict: 
  shows "\<Union>(M {restricted to} X) = (\<Union>M) \<inter> X"
  using RestrictedTo_def by auto

text\<open>Next we show a technical identity that is used to prove sufficiency 
  of some condition for a collection of sets to be a base for a topology.\<close>

lemma ZF1_1_L10: assumes A1: "\<forall>U\<in>C. \<exists>A\<in>B. U = \<Union>A" 
  shows "\<Union>\<Union> {\<Union>{A\<in>B. U = \<Union>A}. U\<in>C} = \<Union>C"
proof
  show "\<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A}) \<subseteq> \<Union>C" by blast
  show "\<Union>C \<subseteq> \<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A})"
  proof
    fix x assume "x \<in> \<Union>C" 
    show "x \<in> \<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A})"
    proof -
      from \<open>x \<in> \<Union>C\<close> obtain U where "U\<in>C \<and> x\<in>U" by auto
      with A1 obtain A where "A\<in>B \<and> U = \<Union>A" by auto
      from \<open>U\<in>C \<and> x\<in>U\<close> \<open>A\<in>B \<and> U = \<Union>A\<close> show "x\<in> \<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A})" 
	by auto
    qed
  qed
qed

text\<open>Standard Isabelle uses a notion of \<open>cons(A,a)\<close> that can be thought 
  of as $A\cup \{a\}$.\<close>

lemma consdef: shows "cons(a,A) = A \<union> {a}"
  using cons_def by auto

text\<open>If a difference between a set and a sigleton is empty, then
  the set is empty or it is equal to the sigleton.\<close>

lemma singl_diff_empty: assumes "A - {x} = 0"
  shows "A = 0 \<or> A = {x}"
  using assms by auto

text\<open>If a difference between a set and a sigleton is the set, 
  then the only element of the singleton is not in the set.\<close>

lemma singl_diff_eq: assumes A1: "A - {x} = A"
  shows "x \<notin> A"
proof -
  have "x \<notin> A - {x}" by auto
  with A1 show "x \<notin> A" by simp
qed

text\<open>A basic property of sets defined by comprehension.
  This is one side of standard Isabelle's \<open>separation\<close> 
  that is in the simp set but somehow not always used by simp.\<close>

lemma comprehension: assumes "a \<in> {x\<in>X. p(x)}"
  shows "a\<in>X" and "p(a)" using assms by auto

end
