(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2012 Daniel de la Concepcion

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. *)

header{*\isaheader{Cardinal\_ZF.thy}*}

theory Cardinal_ZF imports CardinalArith func1

begin

text{*This theory file deals with results on cardinal and sets*}

section{*Some new ideas on cardinals*}

text{*All the results of this section are done without assuming
the \emph{Axiom of Choice}. With the \emph{Axiom of Choice} in play, the proofs become easier
and some of the assumptions may be dropped.*}

text{*Since General Topology Theory is closely related to Set Theory, it is very interesting
to make use of all the possibilities of Set Theory to try to classify homeomorphic
topological spaces. These ideas are generally used to prove that two topological
spaces are not homeomorphic.*}

subsection{*cases-type results*}

text{*There exist cardinals which are the successor of another cardinal,
but; as happens with ordinals, there are cardinals which are limit cardinal.*}

definition
    "LimitC(i)    \<equiv> Card(i) & 0<i & (\<forall>y. (y<i\<and>Card(y)) \<longrightarrow> csucc(y)<i)"

text{*There are three types of cardinals, the zero one, the succesors
of other cardinals and the limit cardinals.*}

lemma Card_cases_disj: 
  assumes "Card(i)" 
  shows "i=0 | (\<exists>j. Card(j) & i=csucc(j)) | LimitC(i)"
proof-
  from assms have D:"Ord(i)" using Card_is_Ord by auto
  {
    assume F:"i\<noteq>0"
    assume False:"~LimitC(i)"
    from F D have "0<i" using Ord_0_lt by auto
    with False assms have "\<exists>y. y < i \<and> Card(y) \<and> \<not> csucc(y) < i" 
      using LimitC_def by blast
    then obtain y where " y < i \<and> Card(y) \<and> \<not> csucc(y) < i"  by blast
    with D have " y < i"" i\<le>csucc(y)" and O:"Card(y)"
      using not_lt_imp_le lt_Ord Card_csucc Card_is_Ord
      by auto
    with assms have "csucc(y)\<le>i""i\<le>csucc(y)" using csucc_le by auto
    then have "i=csucc(y)" using le_anti_sym by auto
    with O have "\<exists>j. Card(j) & i=csucc(j)" by auto
  }
  then show ?thesis by auto
qed

lemma Card_cases: 
  assumes "Card (Q)"
  obtains ("0") "Q=0" | (csucc) T where "Card(T)"  "Q=csucc(T)" | (limit) "LimitC(Q)"
  by (insert Card_cases_disj assms, blast)

text{*Given an ordinal bounded by a cardinal in ordinal order, we can change
to the order of sets.*}

lemma le_imp_lesspoll:
  assumes "Card(Q)"
  shows "A \<le> Q \<Longrightarrow> A \<lesssim> Q"
proof-
  assume "A \<le> Q"
  then have "A<Q\<or>A=Q" using le_iff by auto
  then have "A\<approx>Q\<or>A< Q" using eqpoll_refl by auto
  with assms have "A\<approx>Q\<or>A\<prec> Q" using lt_Card_imp_lesspoll by auto
  then show "A\<lesssim>Q" using lesspoll_def eqpoll_imp_lepoll by auto
qed

text{*There are two types of infinite cardinals, the natural numbers
and those that have at least one infinite strictly smaller cardinal.*}

lemma InfCard_cases_disj:
  assumes "InfCard(Q)"
  shows "Q=nat \<or> (\<exists>j. csucc(j)\<lesssim>Q & InfCard(j))"
proof-
  {
    assume "\<forall>j. \<not> csucc(j) \<lesssim> Q \<or> \<not> InfCard(j)"
    then have D:"\<not> csucc(nat) \<lesssim> Q" using InfCard_nat by auto
    with D assms have "\<not>(csucc(nat) \<le> Q)" using le_imp_lesspoll InfCard_is_Card by auto
    with assms have "Q<(csucc(nat))" using not_le_iff_lt Card_is_Ord Card_csucc Card_is_Ord
      Card_is_Ord InfCard_is_Card Card_nat by auto  
    with assms have "Q\<le>nat" using Card_lt_csucc_iff InfCard_is_Card Card_nat by auto
    with assms have "Q=nat" using InfCard_def le_anti_sym by auto
  }
  then show ?thesis by auto
qed

lemma InfCard_cases: 
  assumes "InfCard (Q)"
  obtains ("nat") "Q=nat"| predecesor j where "csucc(j)\<lesssim>Q \<and> InfCard(j)"
  by (insert InfCard_cases_disj assms,blast)

subsection{*Relations between a cardinal and its successor*}

text{*A set is injective and not bijective to the successor of a cardinal
if and only if it is injective and possibly bijective to the cardinal. *}

lemma Card_less_csucc_eq_le: 
  assumes "Card(m)"
  shows "A \<prec> csucc(m) \<longleftrightarrow> A \<lesssim> m"
proof
  have S:"Ord(csucc(m))" using Card_csucc Card_is_Ord assms by auto
  {
    assume A:"A \<prec> csucc(m)"
    with S have "|A|\<approx>A" using lesspoll_imp_eqpoll by auto
    also with A have "\<dots>\<prec> csucc(m)" by auto
    finally have "|A|\<prec> csucc(m)" by auto
    then have "|A|\<lesssim>csucc(m)""~(|A|\<approx>csucc(m))" using lesspoll_def by auto
    with S have "||A||\<le>csucc(m)""|A|\<noteq>csucc(m)" using lepoll_cardinal_le by auto
    then have "|A|\<le>csucc(m)""|A|\<noteq>csucc(m)" using Card_def Card_cardinal by auto
    then have "~(csucc(m)<|A|)" "|A|\<noteq>csucc(m)" using le_imp_not_lt by auto
    then have C:"csucc(m)<|A| \<longrightarrow> |A|<csucc(m)""|A|=csucc(m) \<longrightarrow> |A|<csucc(m)""|A|<csucc(m) \<longrightarrow> |A|<csucc(m)"
      by auto
    with S have "Ord(|A|)""Ord(csucc(m))" using  Card_cardinal Card_is_Ord by auto
    with C have "|A|<csucc(m)" using Ord_linear_lt[where thesis="|A|<csucc(m)"] by auto
    with assms have "|A|\<le>m" using Card_lt_csucc_iff Card_cardinal
      by auto
    then have "|A|=m\<or> |A|< m" using le_iff by auto
    then have "|A|\<approx>m\<or>|A|< m" using eqpoll_refl by auto
    then have "|A|\<approx>m\<or>|A|\<prec> m" using lt_Card_imp_lesspoll assms by auto
    then have T:"|A|\<lesssim>m" using lesspoll_def eqpoll_imp_lepoll by auto
    from A S have "A\<approx>|A|" using lesspoll_imp_eqpoll eqpoll_sym by auto
    also with T have "\<dots>\<lesssim>m" by auto
    finally show "A\<lesssim>m".
  }
  {
    assume A:"A\<lesssim>m"
    from assms have "m\<prec>csucc(m)" using lt_Card_imp_lesspoll Card_csucc Card_is_Ord
      lt_csucc by auto
    with A show "A\<prec>csucc(m)" using lesspoll_trans1 by auto
  }
qed

text{*If the successor of a cardinal is infinite, so is the original
cardinal.*}

lemma csucc_inf_imp_inf:
  assumes "Card(j)" and "InfCard(csucc(j))"
  shows "InfCard(j)"
proof-
  {
    assume f:"Finite (j)"
    then obtain n where "n\<in>nat" "j\<approx>n" using Finite_def by auto
    with assms(1) have TT: "j=n" "n\<in>nat" 
	using cardinal_cong nat_into_Card Card_def by auto
    then have Q:"succ(j)\<in>nat" using nat_succI by auto
    with f TT have T:"Finite(succ(j))""Card(succ(j))" 
      using nat_into_Card nat_succI by (simp,blast)
    from T(2) have "Card(succ(j))\<and> j\<le>j" using Card_is_Ord by auto
      moreover
    then have "Ord(succ(j))" using Card_is_Ord by auto
      moreover
      {
        fix x
        assume A:"x\<le>j"
        {
          assume "Card(succ(j))\<and> j<x"
          with A have "False" using lt_trans1 by auto
        }
        then have "~(Card(succ(j))\<and> j<x)" by auto
      }
      ultimately have "(LEAST L. Card(L) \<and> j < L)=succ(j) "
        using Least_equality[where i="succ(j)"] by auto
    then have "csucc(j)=succ(j)" using csucc_def by auto
    with Q have "csucc(j)\<in>nat" by auto
    then have "csucc(j)<nat" using lt_def Card_nat Card_is_Ord by auto
    with assms(2) have "False" using InfCard_def lt_trans2 by auto
  }
  then have "~(Finite (j))" by auto
  with assms(1) show ?thesis using Inf_Card_is_InfCard by auto
qed

text{*Since all the cardinals previous to @{text "nat"}
are finite, it cannot be a successor cardinal; hence it is 
a @{text "LimitC"} cardinal.*}

corollary LimitC_nat:
  shows "LimitC(nat)"
proof-
  note Card_nat
  moreover
  have "0<nat" using lt_def by auto
  moreover
  {
    fix y
    assume AS:"y<nat""Card(y)"
    then have ord:"Ord(y)" unfolding lt_def by auto
    then have Cacsucc:"Card(csucc(y))" using Card_csucc by auto
    {
      assume "nat\<le>csucc(y)"
      with Cacsucc have "InfCard(csucc(y))" using InfCard_def by auto
      with AS(2) have "InfCard(y)" using csucc_inf_imp_inf by auto
      then have "nat\<le>y" using InfCard_def by auto
      with AS(1) have "False" using lt_trans2 by auto
    }
    then have "~(nat\<le>csucc(y))" by auto
    then have "csucc(y)<nat" using not_le_iff_lt Ord_nat Cacsucc Card_is_Ord by auto
  }
  ultimately show ?thesis using LimitC_def by auto
qed

subsection{*Main result on cardinals (without the \emph{Axiom of Choice})*}

text{*If two sets are strictly injective to an infinite cardinal,
then so is its union. For the case of successor cardinal, this
theorem is done in the isabelle library in a more general setting;
 but that theorem is of not
use in the case where @{prop "LimitC(Q)"} and it also makes use
of the \emph{Axiom of Choice}. The mentioned theorem is in the
theory file @{text "Cardinal_AC.thy"}*}

text{*Note that if $Q$ is finite and different from $1$, let's assume $Q=n$,
then the union of $A$ and $B$ is not bounded by $Q$.
Counterexample: two disjoint sets of $n-1$ elements each
have a union of $2n-2$ elements which are more than $n$.*}

text{*Note also that if $Q=1$ then $A$ and $B$ must be empty
and the union is then empty too; and $Q$ cannot be @{text "0"} because
no set is injective and not bijective to @{text "0"}.*}

text{* The proof is divided in two parts, first the case when
both sets $A$ and $B$ are finite; and second,
the part when at least one of them is infinite.
In the first part, it is used the fact that a finite
union of finite sets is finite. In the second part it
is used the linear order on cardinals (ordinals).
This proof can not be generalized to a setting with 
an infinite union easily.*}


lemma less_less_imp_un_less:
  assumes "A\<prec>Q" and "B\<prec>Q" and "InfCard(Q)"
  shows "A \<union> B\<prec>Q"
proof-
{
  assume "Finite (A) & Finite(B)"
  then have "Finite(A \<union> B)" using Finite_Un by auto
  then obtain n where R:"A \<union> B \<approx>n"  "n\<in>nat" using Finite_def
    by auto
  then have "|A \<union> B|<nat" using lt_def  cardinal_cong
    nat_into_Card  Card_def  Card_nat Card_is_Ord by auto
  with assms(3) have T:"|A \<union> B|<Q" using InfCard_def lt_trans2 by auto
  from R have "Ord(n)""A \<union> B \<lesssim> n" using nat_into_Card Card_is_Ord eqpoll_imp_lepoll by auto
  then have "A \<union> B\<approx>|A \<union> B|" using lepoll_Ord_imp_eqpoll eqpoll_sym by auto
  also with T assms(3) have "\<dots>\<prec>Q" using lt_Card_imp_lesspoll InfCard_is_Card
    by auto
  finally have "A \<union> B\<prec>Q". 
}
moreover
{
  assume "~(Finite (A) & Finite(B))"
  then have A:"~Finite (A) \<or> ~Finite(B)" by auto
  from assms have B:"|A|\<approx>A""|B|\<approx>B" using lesspoll_imp_eqpoll lesspoll_imp_eqpoll
    InfCard_is_Card Card_is_Ord by auto
  from B(1) have Aeq:"\<forall>x. (|A|\<approx>x) \<longrightarrow> (A\<approx>x)"
    using eqpoll_sym eqpoll_trans by blast
  from B(2) have Beq:"\<forall>x. (|B|\<approx>x) \<longrightarrow> (B\<approx>x)"
    using eqpoll_sym eqpoll_trans by blast
  with A Aeq have "~Finite(|A|)\<or> ~Finite(|B|)" using Finite_def
    by auto
  then have D:"InfCard(|A|)\<or>InfCard(|B|)" using Inf_Card_is_InfCard Inf_Card_is_InfCard  Card_cardinal by blast
  {
    assume AS:"|A| < |B|"
    {
      assume "~InfCard(|A|)"
      with D have "InfCard(|B|)" by auto
    }
    moreover
    {
      assume "InfCard(|A|)"
      then have "nat\<le>|A|" using InfCard_def by auto
      with AS have "nat<|B|" using lt_trans1 by auto
      then have "nat\<le>|B|" using leI by auto
      then have "InfCard(|B|)" using InfCard_def Card_cardinal by auto
    }
    ultimately have INFB:"InfCard(|B|)" by auto
    then have "2<|B|" using InfCard_def lt_trans2[where i="2"]
      lt_def by auto
    then have AG:"2\<lesssim>|B|" using lt_Card_imp_lesspoll Card_cardinal lesspoll_def
      by auto
    from B(2) have "|B|\<approx>B" .
    also with assms(2) have "\<dots>\<prec>Q" by auto
    finally have TTT:"|B|\<prec>Q".
    from B(1) have "Card(|B|)""A \<lesssim>|A|" using eqpoll_sym Card_cardinal eqpoll_imp_lepoll by auto
    with AS have "A\<prec>|B|" using  lt_Card_imp_lesspoll lesspoll_trans1 by auto
    then have I1:"A\<lesssim>|B|" using lesspoll_def by auto
    from B(2) have I2:"B\<lesssim>|B|" using  eqpoll_sym eqpoll_imp_lepoll by auto
    have "A \<union> B\<lesssim>A+B" using Un_lepoll_sum by auto
    also with I1 I2 have "\<dots>\<lesssim> |B| + |B|" using sum_lepoll_mono by auto
    also with AG have "\<dots>\<lesssim>|B| * |B|" using sum_lepoll_prod by auto
    also from assms(3) INFB have "\<dots>\<approx>|B|" using InfCard_square_eqpoll
      by auto
    finally have "A \<union> B\<lesssim>|B|".
    also with TTT have "\<dots>\<prec>Q" by auto
    finally have "A \<union> B\<prec>Q".
  }
  moreover
  {
    assume AS:"|B| < |A|"
    {
      assume "~InfCard(|B|)"
      with D have "InfCard(|A|)" by auto}
      moreover
      {
        assume "InfCard(|B|)"
        then have "nat\<le>|B|" using InfCard_def by auto
        with AS have "nat<|A|" using lt_trans1 by auto
        then have "nat\<le>|A|" using leI by auto
        then have "InfCard(|A|)" using InfCard_def Card_cardinal by auto
      }
      ultimately have INFB:"InfCard(|A|)" by auto
      then have "2<|A|" unfolding InfCard_def using lt_trans2[where i= "2"]
        using lt_def by auto
      then have AG:"2\<lesssim>|A|" using lt_Card_imp_lesspoll[OF Card_cardinal] lesspoll_def
        by auto
      from B(1) have "|A|\<approx>A" .
      also with assms(1) have "\<dots>\<prec>Q" by auto
      finally have TTT:"|A|\<prec>Q".
      from B(2) have "Card(|A|)""B \<lesssim>|B|" using eqpoll_sym Card_cardinal eqpoll_imp_lepoll by auto
      with AS have "B\<prec>|A|" using  lt_Card_imp_lesspoll lesspoll_trans1 by auto
      then have I1:"B\<lesssim>|A|" using lesspoll_def by auto
      from B(1) have I2:"A\<lesssim>|A|" using eqpoll_sym eqpoll_imp_lepoll by auto
      have "A \<union> B\<lesssim>A+B" using Un_lepoll_sum by auto
      also with I1 I2 have "\<dots>\<lesssim> |A| + |A|" using sum_lepoll_mono by auto
      also with AG have "\<dots>\<lesssim>|A| * |A|" using sum_lepoll_prod by auto
      also from INFB assms(3) have "\<dots>\<approx>|A|" using InfCard_square_eqpoll
        by auto
      finally have "A \<union> B\<lesssim>|A|".
      also with TTT have "\<dots>\<prec>Q" by auto
      finally have "A \<union> B\<prec>Q".
    }
    moreover
    {
      assume AS:"|A|=|B|"
      with D have INFB:"InfCard(|A|)" by auto
      then have "2<|A|" using InfCard_def  lt_trans2[where i="2"]
        using lt_def by auto
      then have AG:"2\<lesssim>|A|" using lt_Card_imp_lesspoll Card_cardinal using lesspoll_def
        by auto
      from B(1) have "|A|\<approx>A".
      also with assms(1) have "\<dots>\<prec>Q" by auto
      finally have TTT:"|A|\<prec>Q".
      from AS B have I1:"A\<lesssim>|A|"and I2:"B\<lesssim>|A|" using eqpoll_refl eqpoll_imp_lepoll
        eqpoll_sym by auto
      have "A \<union> B\<lesssim>A+B" using Un_lepoll_sum by auto
      also with I1 I2 have "\<dots>\<lesssim> |A| + |A|" using sum_lepoll_mono by auto
      also with AG have "\<dots>\<lesssim>|A| * |A|" using sum_lepoll_prod  by auto
      also from assms(3) INFB have "\<dots>\<approx>|A|" using InfCard_square_eqpoll
        by auto
      finally have "A \<union> B\<lesssim>|A|".
      also with TTT have "\<dots>\<prec>Q" by auto
      finally have "A \<union> B\<prec>Q".
    }
    ultimately have "A \<union> B\<prec>Q" using Ord_linear_lt[where thesis= "A \<union> B\<prec>Q"] Card_cardinal Card_is_Ord by auto
  }
  ultimately show "A \<union> B\<prec>Q" by auto
qed

section{*Choice axioms*}

definition
  AxiomCardinalChoice ("{the axiom of}_{choice holds for subsets}_") where
  "{the axiom of} Q {choice holds for subsets}K \<equiv> Card(Q) \<and> (\<forall> M N. (M \<lesssim>Q \<and>  (\<forall>t\<in>M. N`t\<noteq>0 \<and> N`t\<subseteq>K)) \<longrightarrow> (\<exists>f. f:Pi(M,\<lambda>t. N`t) \<and> (\<forall>t\<in>M. f`t\<in>N`t)))"

definition
  AxiomCardinalChoiceGen ("{the axiom of}_{choice holds}") where
  "{the axiom of} Q {choice holds} \<equiv> Card(Q) \<and> (\<forall> M N. (M \<lesssim>Q \<and>  (\<forall>t\<in>M. N`t\<noteq>0)) \<longrightarrow> (\<exists>f. f:Pi(M,\<lambda>t. N`t) \<and> (\<forall>t\<in>M. f`t\<in>N`t)))"

text{*The axiom of finite choice always holds.*}

theorem finite_choice:
  assumes "n\<in>nat"
  shows "{the axiom of} n {choice holds}"
proof(rule nat_induct)
  from assms show "n\<in>nat".
next  
  {
    fix M N assume "M\<lesssim>0" "\<forall>t\<in>M. N`t\<noteq>0"
    then have "M=0" using lepoll_0_is_0 by auto
    then have "{\<langle>t,0\<rangle>. t\<in>M}:Pi(M,\<lambda>t. N`t)" unfolding Pi_def domain_def function_def Sigma_def by auto
    moreover from `M=0` have "\<forall>t\<in>M. {\<langle>t,0\<rangle>. t\<in>M}`t\<in>N`t" by auto
    ultimately have "(\<exists>f. f:Pi(M,\<lambda>t. N`t) \<and> (\<forall>t\<in>M. f`t\<in>N`t))" by auto
  }
  then have "(\<forall> M N. (M \<lesssim>0 \<and>  (\<forall>t\<in>M. N`t\<noteq>0)) \<longrightarrow> (\<exists>f. f:Pi(M,\<lambda>t. N`t) \<and> (\<forall>t\<in>M. f`t\<in>N`t)))" by auto
  then show "{the axiom of} 0 {choice holds}" unfolding AxiomCardinalChoiceGen_def using nat_into_Card by auto
next
  fix x
  assume as:"x\<in>nat" "{the axiom of} x {choice holds}"
  {
    fix M N assume ass:"M\<lesssim>succ(x)" "\<forall>t\<in>M. N`t\<noteq>0"
    {
      assume "M\<lesssim>x"
      from as(2) ass(2) have "(M \<lesssim> x \<and> (\<forall>t\<in>M. N ` t \<noteq> 0)) \<longrightarrow> (\<exists>f. f \<in> Pi(M,\<lambda>t. N ` t) \<and> (\<forall>t\<in>M. f ` t \<in> N ` t))" unfolding AxiomCardinalChoiceGen_def by auto
      with `M\<lesssim>x` ass(2) have "(\<exists>f. f \<in> Pi(M,\<lambda>t. N ` t) \<and> (\<forall>t\<in>M. f ` t \<in> N ` t))" by auto
    }
    moreover
    {
      assume "M\<approx>succ(x)"
      then obtain f where f:"f\<in>bij(succ(x),M)" using eqpoll_sym eqpoll_def by blast moreover
      have "x\<in>succ(x)" unfolding succ_def by auto
      ultimately have "restrict(f,succ(x)-{x})\<in>bij(succ(x)-{x},M-{f`x})" using bij_restrict_rem by auto moreover
      have "x\<notin>x" using mem_not_refl by auto
      then have "succ(x)-{x}=x" unfolding succ_def by auto
      ultimately have "restrict(f,x)\<in>bij(x,M-{f`x})" by auto
      then have "x\<approx>M-{f`x}" unfolding eqpoll_def by auto
      then have "M-{f`x}\<approx>x" using eqpoll_sym by auto
      then have "M-{f`x}\<lesssim>x" using eqpoll_imp_lepoll by auto
      with as(2) ass(2) have "(\<exists>g. g \<in> Pi(M-{f`x},\<lambda>t. N ` t) \<and> (\<forall>t\<in>M-{f`x}. g ` t \<in> N ` t))" unfolding AxiomCardinalChoiceGen_def
        by auto
      then obtain g where g:"g\<in> Pi(M-{f`x},\<lambda>t. N ` t)" "\<forall>t\<in>M-{f`x}. g ` t \<in> N ` t" by auto
      from f have ff:"f`x\<in>M" unfolding bij_def inj_def using apply_funtype by auto
      with ass(2) have "N`(f`x)\<noteq>0" by auto
      then obtain y where y:"y\<in>N`(f`x)" by auto
      from g(1) have gg:"g\<subseteq>Sigma(M-{f`x},op `(N))" unfolding Pi_def by auto
      with y ff have "g \<union>{\<langle>f`x, y\<rangle>}\<subseteq>Sigma(M, op `(N))" unfolding Sigma_def by (safe,auto) moreover
      from g(1) have dom:"M-{f`x}\<subseteq>domain(g)" unfolding Pi_def by auto
      then have "M\<subseteq>domain(g \<union>{\<langle>f`x, y\<rangle>})" unfolding domain_def by auto moreover
      from gg have noe:"~(\<exists>t. \<langle>f`x,t\<rangle>\<in>g)" unfolding domain_def Pi_def Sigma_def by auto
      from g(1) have "function(g)" unfolding Pi_def by auto
      with dom have "function(g \<union>{\<langle>f`x, y\<rangle>})" unfolding function_def apply safe apply auto using noe by auto
      ultimately have PP:"g \<union>{\<langle>f`x, y\<rangle>}\<in>Pi(M,\<lambda>t. N ` t)" unfolding Pi_def by auto
      then have "(g \<union>{\<langle>f`x, y\<rangle>})`(f`x)=y" using apply_equality[of "f`x""y" "g \<union>{\<langle>f`x, y\<rangle>}"] by auto
      with y have "(g \<union>{\<langle>f`x, y\<rangle>})`(f`x)\<in>N`(f`x)" by auto moreover
      {
        fix t assume A:"t\<in>M-{f`x}"
        then have "\<langle>t,g`t\<rangle>\<in>g" using apply_Pair g(1) by auto
        then have "\<langle>t,g`t\<rangle>\<in>(g \<union>{\<langle>f`x, y\<rangle>})" by auto
        then have "(g \<union>{\<langle>f`x, y\<rangle>})`t=g`t" using apply_equality PP by auto
        with A have "(g \<union>{\<langle>f`x, y\<rangle>})`t\<in>N`t" using g(2) by auto
      }
      ultimately have "\<forall>t\<in>M. (g \<union>{\<langle>f`x, y\<rangle>})`t\<in>N`t" by auto
      with PP have "\<exists>g. g\<in>Pi(M,\<lambda>t. N ` t) \<and> (\<forall>t\<in>M. g`t\<in>N`t)" by auto
    }
    ultimately have "\<exists>g. g \<in> Pi(M, \<lambda>t. N`t) \<and> (\<forall>t\<in>M. g ` t \<in> N ` t)" using as(1) ass(1)
      lepoll_succ_disj by auto
  }
  then have "\<forall>M N. M \<lesssim> succ(x)\<and>(\<forall>t\<in>M. N`t\<noteq>0)\<longrightarrow>(\<exists>g. g \<in> Pi(M,\<lambda>t. N ` t) \<and> (\<forall>t\<in>M. g ` t \<in> N ` t))" by auto
  then show "{the axiom of}succ(x){choice holds}" unfolding AxiomCardinalChoiceGen_def using nat_into_Card
    as(1) nat_succI by auto
qed

text{*The axiom of choice holds if and only if the @{text "AxiomCardinalChoice"}
holds for every couple of a cardinal @{text "Q"} and a set @{text "K"}.*}

lemma choice_subset_imp_choice:
  shows "{the axiom of} Q {choice holds} \<longleftrightarrow> (ALL K. {the axiom of} Q {choice holds for subsets}K)"
unfolding AxiomCardinalChoice_def AxiomCardinalChoiceGen_def by blast

text{*A choice axiom for greater cardinality implies one for 
smaller cardinality*}

lemma greater_choice_imp_smaller_choice:
  assumes "Q\<lesssim>Q1" "Card(Q)"
  shows "{the axiom of} Q1 {choice holds} \<longrightarrow> ({the axiom of} Q {choice holds})" using assms
  unfolding AxiomCardinalChoiceGen_def using lepoll_trans by blast

text{*If we have a surjective function from a set which is injective to a set 
of ordinals, then we can find an injection which goes the other way.*}

lemma surj_fun_inv:
  assumes "f:surj(A,B)""A\<subseteq>Q""Ord(Q)"
  shows "B\<lesssim>A"
proof-
  let ?g="{\<langle>m,LEAST j. j\<in>A\<and> f`j=m\<rangle>. m\<in>B}"
  have "relation(?g)" unfolding relation_def by auto
  moreover
  have "function(?g)" unfolding function_def by auto
  moreover
  have "domain(?g)=B" by auto
  moreover
  have "range(?g)=?g``B" by auto
  ultimately
  have fun:"?g:B\<rightarrow>?g``B" using function_imp_Pi[of "?g"] by auto
  from assms(2,3) have OA:"!!j. j\<in>A \<Longrightarrow> Ord(j)" unfolding lt_def using Ord_in_Ord by auto
  {
    fix x
    assume "x\<in>?g``B"
    then have "x\<in>range(?g)" "\<exists>y\<in>B. \<langle>y,x\<rangle>\<in>?g" by auto
    then obtain y where T:"x=(LEAST j. j\<in>A\<and> f`j=y)""y\<in>B" by auto
    then obtain z where "f`z=y""z\<in>A""Ord(z)" using assms(1) OA unfolding surj_def by auto
    then have "(LEAST j. j\<in>A\<and> f`j=y)\<in>A" using LeastI[where i="z" and P="\<lambda>z. z\<in>A\<and>f`z=y"] by auto
    then have "x\<in>A" using T by auto
  }
  then have "?g``B\<subseteq>A" by auto
  with fun have fun2:"?g:B\<rightarrow>A" using fun_weaken_type by auto
  then have "?g\<in>inj(B,A)" unfolding inj_def
  proof
    {
      fix w x
      assume AS:"?g`w=?g`x""w\<in>B""x\<in>B"
      from AS(2,3) obtain wz xz where "f`wz=w""f`xz=x""wz\<in>A""xz\<in>A""Ord(wz)""Ord(xz)" using assms(1) OA unfolding
      surj_def by blast
      then have R:"f`(LEAST j. j\<in>A\<and> f`j=w)=w""f`(LEAST j. j\<in>A\<and> f`j=x)=x" using LeastI[where i="wz" and P="\<lambda>z. z\<in>A\<and>f`z=w"]
      LeastI[where i="xz" and P="\<lambda>z. z\<in>A\<and>f`z=x"] by auto
      from AS have "(LEAST j. j\<in>A\<and> f`j=w)=(LEAST j. j\<in>A\<and> f`j=x)" using apply_equality fun2 by auto
      then have "f`(LEAST j. j\<in>A\<and> f`j=w)=f`(LEAST j. j\<in>A\<and> f`j=x)" by auto
      with R(1) have "w=f`(LEAST j. j\<in>A\<and> f`j=x)" by auto
      with R(2) have "w=x" by auto
    }
    then show "\<forall>w\<in>B. \<forall>x\<in>B. ({\<langle>m, LEAST j. (j \<in> A \<and> f ` j = m)\<rangle> . m \<in> B} ` w = {\<langle>m, LEAST j. (j \<in> A \<and> f ` j = m)\<rangle> . m \<in> B} ` x) \<longrightarrow> w = x" by auto
  qed
  then show ?thesis unfolding lepoll_def by auto
qed

text{*The difference with the previous result is that in this one
@{text "A"} is not a subset of an ordinal, it is only injective
with one.*}

theorem surj_fun_inv_2:
  assumes "f:surj(A,B)""A\<lesssim>Q""Ord(Q)"
  shows "B\<lesssim>A"
proof-
  from assms(2) obtain h where h_def:"h\<in>inj(A,Q)" using lepoll_def by auto
  then have bij:"h\<in>bij(A,range(h))" using inj_bij_range by auto
  then obtain h1 where "h1\<in>bij(range(h),A)" using bij_converse_bij by auto
  then have "h1\<in>surj(range(h),A)" using bij_def by auto
  then have "(f O h1)\<in>surj(range(h),B)" using assms(1) comp_surj by auto
  moreover
  {
    fix x
    assume p:"x\<in>range(h)"
    from bij have "h\<in>surj(A,range(h))" using bij_def by auto
    with p obtain q where "q\<in>A" and "h`q=x" using surj_def by auto
    then have "x\<in>Q" using h_def inj_def by auto
  }
  then have "range(h)\<subseteq>Q" by auto
  ultimately have "B\<lesssim>range(h)" using surj_fun_inv[of "f O h1"] assms(3) by auto
  moreover
  have "range(h)\<approx>A" using bij eqpoll_def eqpoll_sym by blast
  ultimately show "B\<lesssim>A" using lepoll_eq_trans by auto
qed

end
