(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2009  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES LOSS OF USE, DATA, OR PROFITS OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header{*\isaheader{TopologicalGroup\_ZF.thy}*}

theory TopologicalGroup_ZF imports Topology_ZF_2 Group_ZF_1

begin

text{*This theory is about the first subject of algebraic topology:
  topological groups.*}

section{*Topological group: definition and notation*}

text{*Topological group is a group that is a topological space 
  at the same time. This means that a topological group is a triple of sets, 
  say $(G,f,\tau)$ such that $\tau$ is a topology on $G$, $f$ is a group 
  operation on $G$ and both $f$ and the operation of taking inverse in $G$ 
  are continuous. Since IsarMathLib defines topology without using the carrier,
  (see @{text "Topology_ZF"}), in our setup we just use $\bigcup \tau$ instead
  of $G$ and say that the  pair of sets $(\bigcup \tau , f)$ is a group.
  This way our definition of being a topological group is a statement about two
  sets: the topology $\tau$ and the group operation $f$ on $\bigcup \tau$.
  Since the domain of the group operation is $G\times G$, the pair of 
  topologies in which $f$ is supposed to be continuous is $\tau$ and
  the product topology on $G\times G$.*}

text{*This way we arrive at the following definition of a predicate that
  states that pair of sets is a topological group.*}

definition
  "IsAtopologicalGroup(\<tau>,f) \<equiv> (\<tau> {is a topology}) \<and> IsAgroup(\<Union>\<tau>,f) \<and>
  IsContinuous(ProductTopology(\<tau>,\<tau>),\<tau>,f) \<and> 
  IsContinuous(\<tau>,\<tau>,GroupInv(\<Union>\<tau>,f))"

text{*We will inherit notation from the @{text "topology0"} locale. That locale assumes that
  $T$ is a topology. For convenience we will denote $G=\bigcup T$ and $\tau$ to be 
  the product topology on $G\times G$. To that we add some
  notation specific to groups. We will use additive notation
  for the group operation, even though we don't assume that the group is abelian.
  The notation $g+A$ will mean the left translation of the set $A$ by element $g$, i.e.
  $g+A=\{g+a|a\in A\}$.
  The group operation $G$ induces a natural operation on the subsets of $G$
  defined as $\langle A,B\rangle \mapsto \{x+y | x\in A, y\in B \}$.
  Such operation has been considered in @{text "func_ZF"} and called 
  $f$ ''lifted to subsets of'' $G$. We will denote the value of such operation 
  on sets $A,B$ as $A+B$.
  The set of neigboorhoods of zero (denoted @{text "\<N>\<^isub>0"}) is the 
  collection of (not necessarily open) sets whose interior contains the neutral
  element of the group.*}

locale topgroup = topology0 +

  fixes G
  defines G_def [simp]: "G \<equiv> \<Union>T"

  fixes prodtop ("\<tau>")
  defines prodtop_def [simp]: "\<tau> \<equiv> ProductTopology(T,T)"

  fixes f

  assumes Ggroup: "IsAgroup(G,f)"

  assumes fcon: "IsContinuous(\<tau>,T,f)"

  assumes inv_cont: "IsContinuous(T,T,GroupInv(G,f))"

  fixes grop (infixl "\<ra>" 90)
  defines grop_def [simp]: "x\<ra>y \<equiv> f`\<langle>x,y\<rangle>"

  fixes grinv ("\<rm> _" 89)
  defines grinv_def [simp]: "(\<rm>x) \<equiv> GroupInv(G,f)`(x)"

  fixes grsub (infixl "\<rs>" 90)
  defines grsub_def [simp]: "x\<rs>y \<equiv> x\<ra>(\<rm>y)"

  fixes setinv ("\<sm> _" 72)
  defines setninv_def [simp]: "\<sm>A \<equiv> GroupInv(G,f)``(A)"

  fixes ltrans (infix "\<ltr>" 73)
  defines ltrans_def [simp]: "x \<ltr> A \<equiv> LeftTranslation(G,f,x)``(A)"

  fixes rtrans (infix "\<rtr>" 73)
  defines rtrans_def [simp]: "A \<rtr> x \<equiv> RightTranslation(G,f,x)``(A)"

  fixes setadd (infixl "\<sad>" 71)
  defines setadd_def [simp]: "A\<sad>B \<equiv> (f {lifted to subsets of} G)`\<langle>A,B\<rangle>"

  fixes gzero ("\<zero>")
  defines gzero_def [simp]: "\<zero> \<equiv> TheNeutralElement(G,f)"

  fixes zerohoods ("\<N>\<^isub>0")
  defines zerohoods_def [simp]: "\<N>\<^isub>0 \<equiv> {A \<in> Pow(G). \<zero> \<in> int(A)}"

text{*The first lemma states that we indeeed talk about topological group
  in the context of @{text "topgroup"} locale.*}

lemma (in topgroup) topGroup: shows "IsAtopologicalGroup(T,f)"
  using topSpaceAssum Ggroup fcon inv_cont IsAtopologicalGroup_def 
  by simp;

text{* If a pair of sets $(T,f)$ forms a topological group, then
 all theorems proven in the @{text "topgroup"} context are valid as applied to 
 $(T,f)$.*}

lemma topGroupLocale: assumes "IsAtopologicalGroup(T,f)"
  shows "topgroup(T,f)"
  using assms IsAtopologicalGroup_def topgroup_def 
    topgroup_axioms.intro topology0_def by simp;

text{*We can use the @{text "group0"} locale in the context of @{text "topgroup"}.*}

lemma (in topgroup) group0_valid_in_tgroup: shows "group0(G,f)"
  using Ggroup group0_def by simp

text{*We can use the @{text "prod_top_spaces0"} locale in the context of @{text "topgroup"}.*}

lemma (in topgroup) prod_top_spaces0_valid: shows "prod_top_spaces0(T,T,T)"
  using topSpaceAssum prod_top_spaces0_def by simp

text{*Negative of a group element is in group.*}

lemma (in topgroup) neg_in_tgroup: assumes "g\<in>G" shows "(\<rm>g) \<in> G"
proof -
  from assms have "GroupInv(G,f)`(g) \<in> G" 
    using group0_valid_in_tgroup group0.inverse_in_group by blast
  thus ?thesis by simp
qed

text{*Zero is in the group.*}

lemma (in topgroup) zero_in_tgroup: shows "\<zero>\<in>G"
proof -
  have "TheNeutralElement(G,f) \<in> G" 
    using group0_valid_in_tgroup group0.group0_2_L2 by blast
  then show "\<zero>\<in>G" by simp
qed

text{*Of course the product topology is a topology (on $G\times G$).*}

lemma (in topgroup) prod_top_on_G:
  shows "\<tau> {is a topology}" and "\<Union>\<tau> = G\<times>G"
  using topSpaceAssum Top_1_4_T1 by auto;

text{*Let's recall that $f$ is a binary operation on $G$ in this context.*}

lemma (in topgroup) topgroup_f_binop: shows "f : G\<times>G \<rightarrow> G"
  using Ggroup group0_def group0.group_oper_assocA by simp;

text{*A subgroup of a topological group is a topological group 
  with relative topology
  and restricted operation. Relative topology is the same
  as @{text "T {restricted to} H"}
  which is defined to be $\{V \cap H: V\in T\}$ in @{text "ZF1"} theory.*}

lemma (in topgroup) top_subgroup: assumes A1: "IsAsubgroup(H,f)"
  shows "IsAtopologicalGroup(T {restricted to} H,restrict(f,H\<times>H))"
proof -
  let ?\<tau>\<^isub>0 = "T {restricted to} H"
  let ?f\<^isub>H = "restrict(f,H\<times>H)"
  have "\<Union>?\<tau>\<^isub>0 = G \<inter> H" using union_restrict by simp;
  also from A1 have "\<dots> = H" 
    using group0_valid_in_tgroup group0.group0_3_L2 by blast;
  finally have "\<Union>?\<tau>\<^isub>0 = H" by simp;
  have "?\<tau>\<^isub>0 {is a topology}" using Top_1_L4 by simp;
  moreover from A1 `\<Union>?\<tau>\<^isub>0 = H` have "IsAgroup(\<Union>?\<tau>\<^isub>0,?f\<^isub>H)"
    using IsAsubgroup_def by simp;
  moreover have "IsContinuous(ProductTopology(?\<tau>\<^isub>0,?\<tau>\<^isub>0),?\<tau>\<^isub>0,?f\<^isub>H)"
  proof -
    have "two_top_spaces0(\<tau>, T,f)"
      using topSpaceAssum prod_top_on_G topgroup_f_binop prod_top_on_G
	two_top_spaces0_def by simp;
    moreover 
    from A1 have "H \<subseteq> G" using group0_valid_in_tgroup group0.group0_3_L2
      by simp;
    then have "H\<times>H \<subseteq> \<Union>\<tau>" using prod_top_on_G by auto;
    moreover have "IsContinuous(\<tau>,T,f)" using fcon by simp;
    ultimately have 
      "IsContinuous(\<tau> {restricted to} H\<times>H, T {restricted to} ?f\<^isub>H``(H\<times>H),?f\<^isub>H)"
      using two_top_spaces0.restr_restr_image_cont by simp;
    moreover have
      "ProductTopology(?\<tau>\<^isub>0,?\<tau>\<^isub>0) = \<tau> {restricted to} H\<times>H"
      using topSpaceAssum prod_top_restr_comm by simp;
    moreover from A1 have "?f\<^isub>H``(H\<times>H) = H" using image_subgr_op
      by simp;
    ultimately show ?thesis by simp;
  qed 
  moreover have "IsContinuous(?\<tau>\<^isub>0,?\<tau>\<^isub>0,GroupInv(\<Union>?\<tau>\<^isub>0,?f\<^isub>H))"
  proof -
    let ?g = "restrict(GroupInv(G,f),H)"
    have "GroupInv(G,f) : G \<rightarrow> G"
      using Ggroup group0_2_T2 by simp;
    then have "two_top_spaces0(T,T,GroupInv(G,f))"
      using topSpaceAssum two_top_spaces0_def by simp;
    moreover from A1 have "H \<subseteq> \<Union>T" 
      using group0_valid_in_tgroup group0.group0_3_L2
      by simp;
    ultimately have 
      "IsContinuous(?\<tau>\<^isub>0,T {restricted to} ?g``(H),?g)"
      using inv_cont two_top_spaces0.restr_restr_image_cont
      by simp;
    moreover from A1 have "?g``(H) = H"
      using group0_valid_in_tgroup group0.restr_inv_onto
      by simp;  
    moreover
    from A1 have "GroupInv(H,?f\<^isub>H) = ?g"
      using group0_valid_in_tgroup group0.group0_3_T1
      by simp;
    with `\<Union>?\<tau>\<^isub>0 = H` have "?g = GroupInv(\<Union>?\<tau>\<^isub>0,?f\<^isub>H)" by simp;
    ultimately show ?thesis by simp;
  qed
  ultimately show ?thesis unfolding IsAtopologicalGroup_def by simp;
qed

section{*Interval arithmetic, translations and inverse of set*}

text{*In this section we list some properties of operations of translating a
  set and reflecting it around the neutral element of the group. Many of the results
  are proven in other theories, here we just collect them and rewrite in notation
  specific to the @{text "topgroup"} context.*}

text{*Different ways of looking at adding sets.*}

lemma (in topgroup) interval_add: assumes "A\<subseteq>G" "B\<subseteq>G" shows
  "A\<sad>B \<subseteq> G" and "A\<sad>B = f``(A\<times>B)"  "A\<sad>B = (\<Union>x\<in>A. x\<ltr>B)"
proof -
  from assms show "A\<sad>B \<subseteq> G" and "A\<sad>B = f``(A\<times>B)" 
    using topgroup_f_binop lift_subsets_explained by auto
  from assms show "A\<sad>B = (\<Union>x\<in>A. x\<ltr>B)"
    using group0_valid_in_tgroup group0.image_ltrans_union by simp
qed

text{*Right and left translations are continuous.*}

lemma (in topgroup) trans_cont: assumes "g\<in>G" shows
  "IsContinuous(T,T,RightTranslation(G,f,g))" and
  "IsContinuous(T,T,LeftTranslation(G,f,g))"
using assms group0_valid_in_tgroup group0.trans_eq_section
  topgroup_f_binop fcon prod_top_spaces0_valid 
  prod_top_spaces0.fix_1st_var_cont prod_top_spaces0.fix_2nd_var_cont
  by auto

text{*Left and right translations of an open set are open.*}

lemma (in topgroup) open_tr_open: assumes "g\<in>G" and "V\<in>T"
  shows "g\<ltr>V \<in> T" and  "V\<rtr>g \<in> T"
  using assms neg_in_tgroup trans_cont IsContinuous_def 
    group0_valid_in_tgroup group0.trans_image_vimage by auto

text{*Right and left translations are homeomorphisms.*}

lemma (in topgroup) tr_homeo: assumes "g\<in>G" shows
  "IsAhomeomorphism(T,T,RightTranslation(G,f,g))" and
  "IsAhomeomorphism(T,T,LeftTranslation(G,f,g))"
  using assms group0_valid_in_tgroup group0.trans_bij trans_cont open_tr_open
    bij_cont_open_homeo by auto

text{*Translations preserve interior.*}

lemma (in topgroup) trans_interior: assumes A1: "g\<in>G" and A2: "A\<subseteq>G" 
  shows "g \<ltr> int(A) = int(g\<ltr>A)"
proof -
  from assms have "A \<subseteq> \<Union>T" and "IsAhomeomorphism(T,T,LeftTranslation(G,f,g))"
    using tr_homeo by auto
  then show ?thesis using int_top_invariant by simp
qed
(*
text{*Translating by an inverse and then by an element cancels out.*}

lemma (in topgroup) trans_inverse_elem: assumes "g\<in>G" and "A\<subseteq>G"
  shows "g\<ltr>((\<rm>g)\<ltr>A) = A"
using assms neg_in_tgroup group0_valid_in_tgroup group0.trans_comp_image
  group0.group0_2_L6 group0.trans_neutral image_id_same by simp*)

text{*Inverse of an open set is open.*}

lemma (in topgroup) open_inv_open: assumes "V\<in>T" shows "\<sm>V \<in> T"
  using assms group0_valid_in_tgroup group0.inv_image_vimage
    inv_cont IsContinuous_def by simp

text{*Inverse is a homeomorphism.*}

lemma (in topgroup) inv_homeo: shows "IsAhomeomorphism(T,T,GroupInv(G,f))"
  using group0_valid_in_tgroup group0.group_inv_bij inv_cont open_inv_open
  bij_cont_open_homeo by simp

text{*Taking negative preserves interior.*}

lemma (in topgroup) int_inv_inv_int: assumes "A \<subseteq> G" 
  shows "int(\<sm>A) = \<sm>(int(A))"
  using assms inv_homeo int_top_invariant by simp

section{*Neighborhoods of zero*}

text{*Zero neighborhoods are (not necessarily open) sets whose interior
  contains the neutral element of the group. In the @{text "topgroup"} locale
  the collection of neighboorhoods of zero is denoted @{text "\<N>\<^isub>0"}. *}

text{*The whole sppace is a neighborhood of zero.*}

lemma (in topgroup) zneigh_not_empty: shows "G \<in> \<N>\<^isub>0"
  using topSpaceAssum IsATopology_def Top_2_L3 zero_in_tgroup
  by simp

text{*Any element belongs to the interior of any neighboorhood of zero
  translated by that element.*}

lemma (in topgroup) elem_in_int_trans:
  assumes A1: "g\<in>G" and A2: "H \<in> \<N>\<^isub>0"
  shows "g \<in> int(g\<ltr>H)"
proof -
  from A2 have "\<zero> \<in> int(H)" and "int(H) \<subseteq> G" using Top_2_L2 by auto
  with A1 have "g \<in> g \<ltr> int(H)"
    using group0_valid_in_tgroup group0.neut_trans_elem by simp
  with assms show ?thesis using trans_interior by simp
qed

text{*Negative of a neighborhood of zero is a neighborhood of zero.*}

lemma (in topgroup) neg_neigh_neigh: assumes "H \<in> \<N>\<^isub>0"
  shows "\<sm>H \<in> \<N>\<^isub>0"
proof -
  from assms have "int(H) \<subseteq> G" and "\<zero> \<in> int(H)" using Top_2_L1 by auto
  with assms have "\<zero> \<in> int(\<sm>H)" using group0_valid_in_tgroup group0.neut_inv_neut
    int_inv_inv_int by simp
  moreover
  have "GroupInv(G,f):G\<rightarrow>G" using Ggroup group0_2_T2 by simp
  then have "\<sm>H \<subseteq> G" using func1_1_L6 by simp
  ultimately show ?thesis by simp
qed

text{*Translating an open set by a negative of a point that belongs to it
  makes it a neighboorhood of zero.*}

lemma (in topgroup) open_trans_neigh: assumes A1: "U\<in>T" and "g\<in>U"
  shows "(\<rm>g)\<ltr>U \<in> \<N>\<^isub>0"
proof -
  let ?H = "(\<rm>g)\<ltr>U"
  from assms have "g\<in>G" by auto
  then have "(\<rm>g) \<in> G" using neg_in_tgroup by simp
  with A1 have "?H\<in>T" using open_tr_open by simp
  hence "?H \<subseteq> G" by auto
  moreover have "\<zero> \<in> int(?H)"
  proof -
    from assms have "U\<subseteq>G" and "g\<in>U" by auto
    with `?H\<in>T` show "\<zero> \<in> int(?H)" 
      using group0_valid_in_tgroup group0.elem_trans_neut Top_2_L3
        by auto
  qed
  ultimately show ?thesis by simp
qed

section{*Closure in topological groups*}

text{*This section is devoted to a characterization of closure
  in topological groups.*}

text{*Closure of a set is contained in the sum of the set and any
  neighboorhood of zero.*}

lemma (in topgroup) cl_contains_zneigh:
  assumes A1: "A\<subseteq>G" and A2: "H \<in> \<N>\<^isub>0"
  shows "cl(A) \<subseteq> A\<sad>H"
proof
  fix x assume "x \<in> cl(A)"
  from A1 have "cl(A) \<subseteq> G" using Top_3_L11 by simp
  with `x \<in> cl(A)` have "x\<in>G" by auto
  have "int(H) \<subseteq> G" using Top_2_L2 by auto
  let ?V = "int(x \<ltr> (\<sm>H))"
  have "?V = x \<ltr> (\<sm>int(H))"
  proof -
    from A2 `x\<in>G` have "?V = x \<ltr> int(\<sm>H)" 
      using neg_neigh_neigh trans_interior by simp
    with A2 show ?thesis  using int_inv_inv_int by simp
  qed
  have "A\<inter>?V \<noteq> 0"
  proof -
    from A2 `x\<in>G` `x \<in> cl(A)` have "?V\<in>T" and "x \<in> cl(A) \<inter> ?V" 
      using neg_neigh_neigh elem_in_int_trans Top_2_L2 by auto
    with A1 show "A\<inter>?V \<noteq> 0" using cl_inter_neigh by simp
  qed
  then obtain y where "y\<in>A" and "y\<in>?V" by auto
  with `?V = x \<ltr> (\<sm>int(H))` `int(H) \<subseteq> G` `x\<in>G` have "x \<in> y\<ltr>int(H)"
    using group0_valid_in_tgroup group0.ltrans_inv_in by simp
  with `y\<in>A` have "x \<in> (\<Union>y\<in>A. y\<ltr>H)" using Top_2_L1 func1_1_L8 by auto
  with assms show "x \<in> A\<sad>H" using interval_add by simp
qed

text{*The next theorem provides a characterization of closure in topological
  groups in terms of neighborhoods of zero.*}

theorem (in topgroup) cl_topgroup:
  assumes "A\<subseteq>G" shows "cl(A) = (\<Inter>H\<in>\<N>\<^isub>0. A\<sad>H)"
proof
  from assms show "cl(A) \<subseteq> (\<Inter>H\<in>\<N>\<^isub>0. A\<sad>H)" 
    using zneigh_not_empty cl_contains_zneigh by auto
next
  { fix x assume "x \<in> (\<Inter>H\<in>\<N>\<^isub>0. A\<sad>H)"
    then have "x \<in> A\<sad>G" using zneigh_not_empty by auto
    with assms have "x\<in>G" using interval_add by blast
    have "\<forall>U\<in>T. x\<in>U \<longrightarrow> U\<inter>A \<noteq> 0"
    proof -
      { fix U assume "U\<in>T" and "x\<in>U"
        let ?H = "\<sm>((\<rm>x)\<ltr>U)"
        from `U\<in>T` and `x\<in>U` have "(\<rm>x)\<ltr>U \<subseteq> G" and "?H \<in> \<N>\<^isub>0" 
          using open_trans_neigh neg_neigh_neigh by auto
        with `x \<in> (\<Inter>H\<in>\<N>\<^isub>0. A\<sad>H)` have "x \<in> A\<sad>?H" by auto
        with assms `?H \<in> \<N>\<^isub>0` obtain y where "y\<in>A" and "x \<in> y\<ltr>?H"
          using interval_add by auto
        have "y\<in>U"
        proof -
          from assms `y\<in>A` have "y\<in>G" by auto
          with `(\<rm>x)\<ltr>U \<subseteq> G` and `x \<in> y\<ltr>?H` have "y \<in> x\<ltr>((\<rm>x)\<ltr>U)"
            using group0_valid_in_tgroup group0.ltrans_inv_in by simp
          with `U\<in>T` `x\<in>G` show "y\<in>U" 
            using neg_in_tgroup group0_valid_in_tgroup group0.trans_comp_image
              group0.group0_2_L6 group0.trans_neutral image_id_same by auto
        qed
        with `y\<in>A` have "U\<inter>A \<noteq> 0" by auto
      } thus ?thesis by simp
    qed
    with assms `x\<in>G` have "x \<in> cl(A)" using inter_neigh_cl by simp
  } thus "(\<Inter>H\<in>\<N>\<^isub>0. A\<sad>H) \<subseteq> cl(A)" by auto
qed

end
