(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2009  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES LOSS OF USE, DATA, OR PROFITS OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header{*\isaheader{TopologicalGroup\_ZF.thy}*}

theory TopologicalGroup_ZF imports Topology_ZF_2 Group_ZF

begin

text{*This theory is about the first subject of algebraic topology:
  topological groups.*}

section{*Topological group: definition and notation*}

text{*Topological group is a group that is a topological space 
  at the same time. This means that a topological group is a triple of sets, 
  say $(G,f,\tau)$ such that $\tau$ is a topology on $G$, $f$ is a group 
  operation on $G$ and both $f$ and the operation of taking inverse in $G$ 
  are continuous. Since IsarMathLib defines topology without using the carrier,
  (see @{text "Topology_ZF"}), in our setup we just use $\bigcup \tau$ instead
  of $G$ and say that the  pair of sets $(\bigcup \tau , f)$ is a group.
  This way our definition of being a topological group is a statement about two
  sets: the topology $\tau$ and the group operation $f$ on $\bigcup \tau$.
  Since the domain of the group operation is $G\times G$, the pair of 
  topologies in which $f$ is supposed to be continuous is $\tau$ and
  the product topology on $G\times G$.*}

text{*This way we arrive at the following definition of a predicate that
  states that pair of sets is a topological group.*}

definition
  "IsAtopologicalGroup(\<tau>,f) \<equiv> (\<tau> {is a topology}) \<and> IsAgroup(\<Union>\<tau>,f) \<and>
  IsContinuous(ProductTopology(\<tau>,\<tau>),\<tau>,f) \<and> 
  IsContinuous(\<tau>,\<tau>,GroupInv(\<Union>\<tau>,f))"

text{*We will inherit notation from the @{text "topology0"} locale. That locale assumes that
  $T$ is a topology. For convenience we will denote $G=\bigcup T$ and $\tau$ to be 
  the product topology on $G\times G$. To that we add some
  notation specific to groups. We will use additive notation
  for the group operation, even though we don't assume that the group is abelian.*}

locale topgroup = topology0 +

  fixes G
  defines G_def [simp]: "G \<equiv> \<Union>T"

  fixes prodtop ("\<tau>")
  defines prodtop_def [simp]: "\<tau> \<equiv> ProductTopology(T,T)"

  fixes f

  assumes Ggroup: "IsAgroup(G,f)"

  assumes fcon: "IsContinuous(\<tau>,T,f)"

  assumes inv_cont: "IsContinuous(T,T,GroupInv(G,f))"

  fixes grop (infixl "\<ra>" 90)
  defines grop_def [simp]: "x\<ra>y \<equiv> f`\<langle>x,y\<rangle>"

  fixes grinv ("\<rm> _" 89)
  defines grinv_def [simp]: "(\<rm>x) \<equiv> GroupInv(G,f)`(x)"

  fixes grsub (infixl "\<rs>" 90)
  defines grsub_def [simp]: "x\<rs>y \<equiv> x\<ra>(\<rm>y)"

text{*The first lemma states that we indeeed talk about topological group
  in the context of @{text "topgroup"} locale.*}

lemma (in topgroup) topGroup: shows "IsAtopologicalGroup(T,f)"
  using topSpaceAssum Ggroup fcon inv_cont IsAtopologicalGroup_def 
  by simp;

text{* If a pair of sets $(T,f)$ forms a topological group, then
 all theorems proven in the @{text "topgroup"} context are valid as applied to 
 $(T,f)$.*}

lemma topGroupLocale: assumes "IsAtopologicalGroup(T,f)"
  shows "topgroup(T,f)"
  using assms IsAtopologicalGroup_def topgroup_def 
    topgroup_axioms.intro topology0_def by simp;

text{*We can use the @{text "group0"} locale in the context of @{text "topgroup"}.*}

lemma (in topgroup) group0_valid_in_tgroup: shows "group0(G,f)"
  using Ggroup group0_def by simp;

text{*Of course the product topology is a topology (on $G\times G$).*}

lemma (in topgroup) prod_top_on_G:
  shows "\<tau> {is a topology}" and "\<Union>\<tau> = G\<times>G"
  using topSpaceAssum Top_1_4_T1 by auto;

text{*Let's recall that $f$ is a binary operation on $G$ in this context.*}

lemma (in topgroup) topgroup_f_binop: shows "f : G\<times>G \<rightarrow> G"
  using Ggroup group0_def group0.group_oper_assocA by simp;

text{*A subgroup of a topological group is a topological group 
  with relative topology
  and restricted operation.Relative topology is the same 
  as @{text "T {restricted to} H"}
  which is defined to be $\{V \cap H: V\in T\}$ in @{text "ZF1"} theory.*}

lemma (in topgroup) top_subgroup: assumes A1: "IsAsubgroup(H,f)"
  shows "IsAtopologicalGroup(T {restricted to} H,restrict(f,H\<times>H))"
proof -
  let ?\<tau>\<^isub>0 = "T {restricted to} H"
  let ?f\<^isub>H = "restrict(f,H\<times>H)"
  have "\<Union>?\<tau>\<^isub>0 = G \<inter> H" using union_restrict by simp;
  also from A1 have "\<dots> = H" 
    using group0_valid_in_tgroup group0.group0_3_L2 by blast;
  finally have "\<Union>?\<tau>\<^isub>0 = H" by simp;
  have "?\<tau>\<^isub>0 {is a topology}" using Top_1_L4 by simp;
  moreover from A1 `\<Union>?\<tau>\<^isub>0 = H` have "IsAgroup(\<Union>?\<tau>\<^isub>0,?f\<^isub>H)"
    using IsAsubgroup_def by simp;
  moreover have "IsContinuous(ProductTopology(?\<tau>\<^isub>0,?\<tau>\<^isub>0),?\<tau>\<^isub>0,?f\<^isub>H)"
  proof -
    have "two_top_spaces0(\<tau>, T,f)"
      using topSpaceAssum prod_top_on_G topgroup_f_binop prod_top_on_G
	two_top_spaces0_def by simp;
    moreover 
    from A1 have "H \<subseteq> G" using group0_valid_in_tgroup group0.group0_3_L2
      by simp;
    then have "H\<times>H \<subseteq> \<Union>\<tau>" using prod_top_on_G by auto;
    moreover have "IsContinuous(\<tau>,T,f)" using fcon by simp;
    ultimately have 
      "IsContinuous(\<tau> {restricted to} H\<times>H, T {restricted to} ?f\<^isub>H``(H\<times>H),?f\<^isub>H)"
      using two_top_spaces0.restr_restr_image_cont by simp;
    moreover have
      "ProductTopology(?\<tau>\<^isub>0,?\<tau>\<^isub>0) = \<tau> {restricted to} H\<times>H"
      using topSpaceAssum prod_top_restr_comm by simp;
    moreover from A1 have "?f\<^isub>H``(H\<times>H) = H" using image_subgr_op
      by simp;
    ultimately show ?thesis by simp;
  qed 
  moreover have "IsContinuous(?\<tau>\<^isub>0,?\<tau>\<^isub>0,GroupInv(\<Union>?\<tau>\<^isub>0,?f\<^isub>H))"
  proof -
    let ?g = "restrict(GroupInv(G,f),H)"
    have "GroupInv(G,f) : G \<rightarrow> G"
      using Ggroup group0_2_T2 by simp;
    then have "two_top_spaces0(T,T,GroupInv(G,f))"
      using topSpaceAssum two_top_spaces0_def by simp;
    moreover from A1 have "H \<subseteq> \<Union>T" 
      using group0_valid_in_tgroup group0.group0_3_L2
      by simp;
    ultimately have 
      "IsContinuous(?\<tau>\<^isub>0,T {restricted to} ?g``(H),?g)"
      using inv_cont two_top_spaces0.restr_restr_image_cont
      by simp;
    moreover from A1 have "?g``(H) = H"
      using group0_valid_in_tgroup group0.restr_inv_onto
      by simp;  
    moreover
    from A1 have "GroupInv(H,?f\<^isub>H) = ?g"
      using group0_valid_in_tgroup group0.group0_3_T1
      by simp;
    with `\<Union>?\<tau>\<^isub>0 = H` have "?g = GroupInv(\<Union>?\<tau>\<^isub>0,?f\<^isub>H)" by simp;
    ultimately show ?thesis by simp;
  qed
  ultimately show ?thesis unfolding IsAtopologicalGroup_def by simp;
qed

end
