(* 
This file is a part of IsarMathLib - 
a library of formalized mathematics for Isabelle/Isar.

Copyright (C) 2008  Slawomir Kolodynski

This program is free software; Redistribution and use in source and binary forms, 
with or without modification, are permitted provided that the 
following conditions are met:

1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header{*\isaheader{Group\_ZF\_1.thy}*}

theory Group_ZF_1 imports Group_ZF

begin

text{*In this theory we consider right and left translations and odd 
  functions.*}

section{*Translations*}

text{*In this section we consider translations. Translations are maps 
  $T: G\rightarrow G$ of the form $T_g (a) = g\cdot a$ or 
  $T_g (a) = a\cdot g$. We also consider two-dimensional translations
  $T_g : G\times G \rightarrow G\times G$, where 
  $T_g(a,b) = (a\cdot g, b\cdot g)$ or $T_g(a,b) = (g\cdot a, g\cdot b)$. 
  *}

text{*For an element $a\in G$ the right translation is defined 
  a function (set of pairs) such that its value (the second element
  of a pair) is the value of the group operation on the first element
  of the pair and $g$. This looks a bit strange in the raw set notation, 
  when we write a function explicitely as a set of pairs and value of 
  the group operation on the pair $\langle a,b \rangle$ 
  as @{text "P`\<langle>a,b\<rangle>"} instead of the usual infix $a\cdot b$
  or $a + b$. *}

definition
  "RightTranslation(G,P,g) \<equiv> {\<langle> a,b\<rangle> \<in> G\<times>G. P`\<langle>a,g\<rangle> = b}"

text{*A similar definition of the left translation.*}

definition
  "LeftTranslation(G,P,g) \<equiv> {\<langle>a,b\<rangle> \<in> G\<times>G. P`\<langle>g,a\<rangle> = b}"


text{*Translations map $G$ into $G$. Two dimensional translations map
  $G\times G$ into itself.*}

lemma (in group0) group0_5_L1: assumes A1: "g\<in>G"
  shows "RightTranslation(G,P,g) : G\<rightarrow>G"
  "LeftTranslation(G,P,g) : G\<rightarrow>G"
proof -
  from A1 have "\<forall>a\<in>G. a\<cdot>g \<in> G" and "\<forall>a\<in>G. g\<cdot>a \<in> G"
    using group_oper_assocA apply_funtype by auto;
  then show 
    "RightTranslation(G,P,g) : G\<rightarrow>G" 
    "LeftTranslation(G,P,g) : G\<rightarrow>G"
    using RightTranslation_def LeftTranslation_def func1_1_L11A
    by auto;
qed;

text{*The values of the translations are what we expect.*}

lemma (in group0) group0_5_L2: assumes "g\<in>G" "a\<in>G"
  shows 
  "RightTranslation(G,P,g)`(a) = a\<cdot>g"
  "LeftTranslation(G,P,g)`(a) = g\<cdot>a"
  using assms group0_5_L1 RightTranslation_def LeftTranslation_def
    func1_1_L11B by auto;


text{*Composition of left translations is a left translation by the product.*}

lemma (in group0) group0_5_L4: assumes A1: "g\<in>G" "h\<in>G" "a\<in>G" and 
  A2: "T\<^isub>g = LeftTranslation(G,P,g)"  "T\<^isub>h = LeftTranslation(G,P,h)"
  shows 
  "T\<^isub>g`(T\<^isub>h`(a)) = g\<cdot>h\<cdot>a"
  "T\<^isub>g`(T\<^isub>h`(a)) = LeftTranslation(G,P,g\<cdot>h)`(a)"
proof -;
  from A1 have I: "h\<cdot>a\<in>G"  "g\<cdot>h\<in>G"
    using group_oper_assocA apply_funtype by auto;
  with A1 A2 show "T\<^isub>g`(T\<^isub>h`(a)) = g\<cdot>h\<cdot>a"
    using group0_5_L2 group_oper_assoc by simp;
  with A1 A2 I show 
    "T\<^isub>g`(T\<^isub>h`(a)) = LeftTranslation(G,P,g\<cdot>h)`(a)"
    using group0_5_L2 group_oper_assoc by simp;
qed;

text{*Composition of right translations is a right translation by 
  the product.*}

lemma (in group0) group0_5_L5: assumes A1: "g\<in>G" "h\<in>G" "a\<in>G" and 
  A2: "T\<^isub>g = RightTranslation(G,P,g)"  "T\<^isub>h = RightTranslation(G,P,h)"
  shows 
 "T\<^isub>g`(T\<^isub>h`(a)) = a\<cdot>h\<cdot>g"
  "T\<^isub>g`(T\<^isub>h`(a)) = RightTranslation(G,P,h\<cdot>g)`(a)"
proof -
  from A1 have I: "a\<cdot>h\<in>G" "h\<cdot>g \<in>G"
    using group_oper_assocA apply_funtype by auto;
  with A1 A2 show "T\<^isub>g`(T\<^isub>h`(a)) = a\<cdot>h\<cdot>g" 
    using group0_5_L2 group_oper_assoc by simp;
  with A1 A2 I show 
    "T\<^isub>g`(T\<^isub>h`(a)) = RightTranslation(G,P,h\<cdot>g)`(a)"
    using group0_5_L2 group_oper_assoc by simp;
qed;

text{*The image of a set under a composition of translations is the same as
  the image under translation by a product.*}

lemma (in group0) group0_5_L6: 
  assumes A1: "g\<in>G" "h\<in>G" and A2: "A\<subseteq>G" and 
  A3: "T\<^isub>g = RightTranslation(G,P,g)"  "T\<^isub>h = RightTranslation(G,P,h)"
  shows "T\<^isub>g``(T\<^isub>h``(A)) = {a\<cdot>h\<cdot>g. a\<in>A}"
proof -;
  from A2 have "\<forall>a\<in>A. a\<in>G" by auto;
  from A1 A3 have "T\<^isub>g : G\<rightarrow>G"  "T\<^isub>h : G\<rightarrow>G"
    using group0_5_L1 by auto;
  with assms `\<forall>a\<in>A. a\<in>G` show 
    "T\<^isub>g``(T\<^isub>h``(A)) = {a\<cdot>h\<cdot>g. a\<in>A}"
    using func1_1_L15C group0_5_L5 by auto;
qed;

section{*Odd functions*}

text{*This section is about odd functions.*}

text{*Odd functions are those that commute with the group inverse:
  $f(a^{-1}) = (f(a))^{-1}.$*}

definition
  "IsOdd(G,P,f) \<equiv> (\<forall>a\<in>G. f`(GroupInv(G,P)`(a)) = GroupInv(G,P)`(f`(a)) )"

text{*Let's see the definition of an odd function in a more readable 
  notation.*}

lemma (in group0) group0_6_L1: 
  shows "IsOdd(G,P,p) \<longleftrightarrow> ( \<forall>a\<in>G. p`(a\<inverse>) = (p`(a))\<inverse> )"
  using IsOdd_def by simp;

text{*We can express the definition of an odd function in two ways.*}

lemma (in group0) group0_6_L2:
  assumes A1: "p : G\<rightarrow>G" 
  shows 
  "(\<forall>a\<in>G. p`(a\<inverse>) = (p`(a))\<inverse>) \<longleftrightarrow> (\<forall>a\<in>G. (p`(a\<inverse>))\<inverse> = p`(a))"
proof
  assume "\<forall>a\<in>G. p`(a\<inverse>) = (p`(a))\<inverse>"
  with A1 show "\<forall>a\<in>G. (p`(a\<inverse>))\<inverse> = p`(a)"
    using apply_funtype group_inv_of_inv by simp;
next assume A2: "\<forall>a\<in>G. (p`(a\<inverse>))\<inverse> = p`(a)"
  { fix a assume "a\<in>G"
    with A1 A2  have 
      "p`(a\<inverse>) \<in> G" and "((p`(a\<inverse>))\<inverse>)\<inverse> =  (p`(a))\<inverse>"
    using apply_funtype inverse_in_group by auto;
  then have "p`(a\<inverse>) = (p`(a))\<inverse>"
    using group_inv_of_inv by simp;
  } then show "\<forall>a\<in>G. p`(a\<inverse>) = (p`(a))\<inverse>" by simp;
qed;

end
