{- 
	This file is part of isar2html - a tool for rendering IsarMathLib 
	theories in in DHTML.
    Copyright (C) 2008, 2009  Slawomir Kolodynski

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
-}
    
-- | The function export2Html in this module takes a parsed 
-- IsarMathLib theory and produces corresponding body of the HTML
-- document.

module Export2Html
   where

import Data.List(isPrefixOf,intersperse)
import qualified Data.Map as M
import Data.Maybe(isJust, fromJust) 

import IMLP_datatypes
import Utils
import IsarSym2Latex
import ProcessThys
import ExportCommon

-- | converts literal latex: surround with "\( \)"
convLatex :: String -> String
convLatex s = "\\(" ++ s ++ "\\)"

-- | converts literal text: @{text "some text"}
littext :: String -> String
littext s = if (isIdentifier s) then it s else isar2latex s

-- | make text italic
it :: String -> String
it s = "<i>" ++ s ++ "</i>"

-- | makes a string boldface, also surrounds with spaces
bf :: String -> String
bf s = "<b> " ++ s ++ " </b>"

-- | makes a paragraph
par :: String -> String
par s = if "<p>" `isPrefixOf` s then s
        else "<p>" ++ s ++ "</p>"

-- | makes a div to simulate paragraphs.
pd :: String -> String
pd s = "<div id=\"pardiv\" > " ++ s ++ "</div>"

-- | creates a heading
hd :: String -> String
hd s = "<h3>" ++ s ++ "</h3>"

-- | exports informal text
exp2inftext :: String -> String
exp2inftext s = "<div class=\"inftext\">\n<p>" ++
   ( replace "\n\n" "</p>\n<p>" $
     appBetween convLatex "$" "$" $ 
     appBetween littext "@{text \"" "\"}" s ) ++
   "</p>\n</div>\n"

-- | creates a floating slider
floatSlider :: String -> String
floatSlider contents =
   "<span id=\"hintref\">" ++ contents ++ "</span>"
   
-- | creates a slider
slider :: String -> String -> String
slider tittle content = "<span id=\"hstrigger\" class=\"hideshow\" >" ++
   tittle ++ "</span>\n<div id=\"hscontent\" class=\"proof\" >" ++
   content ++ "\n</div>\n"

-- | creates a link
tlink :: String -> String
tlink s = "<a href=\"" ++ s ++"\">" ++ s ++ "</a>"

-- | makes formal
mkformal :: String -> String
mkformal s = "<div class=\"formal\">\n" ++ s ++ "\n</div>\n"

-- | renders a locale name in the theorem
inContext :: String -> String
inContext s = 
   if (not $ null s) then "<b>(in</b> " ++ s ++ "<b>)</b> " else ""

-- | renders a reference hint that is displayed when a reference is clicked
refdiv :: String -> String -> String
refdiv id content = "<div id=\"" ++ id ++ "\"class=\"refhint\">" ++
   content ++ "</div>"


-- | exports FormalItemInfo
exportFormalItemInfo :: FormalItemInfo -> String
exportFormalItemInfo (FormalItemInfo tn (SimpleProp tpe cntxt nme prms clms _)) =
   ( bf tpe) ++ " " ++
   ( inContext cntxt ) ++ nme ++ ": " ++ 
   (if not $ null prms then (bf "assumes ") ++ (exportSimpleClaims prms) else "") ++ 
   ( bf " shows " ) ++ (exportSimpleClaims clms)


exportFormalItemInfo (FormalItemInfo tn ( SimpleDef nm d )) = 
    ("Definition of " ++ nm ++ ":\n" ++ (isar2latex d) )


exportFormalItemInfo (FormalItemInfo tn OtherSimpleItem) = " Other formal item "

-- checks if a reference is on the list of known theorems and and if so creates 
-- a div with the rendered simple form, otherwise returns an empty string
createRefDiv :: M.Map String String -> String -> String
createRefDiv mfii nm = 
   if isJust lookres then refdiv nmAfterDot (fromJust lookres)
   else ""
     where
        nmAfterDot = reverse $ takeWhile (/='.') $ reverse nm
        lookres = (M.lookup nmAfterDot) mfii

-- | exports a list of theories to body html form 
{- exportTheories :: String -> [Theory] -> [(String,String)]
exportTheories templ tt = map expThr tinfos where
   kb =  processTheories tt
   mfii = getRefsInfo $ kbformalitems kb
   tinfos = kbtheories kb
   tlinks = concatMap (thrylink . tiname) tinfos
   expThr tinfo = ("isarhtml/" ++ (tiname tinfo) ++ ".html", thHtml)
      where
         thHtml = replaceAll [ ("this is theory placeholder", exportedTheory)
                             , ("this is links placeholder", tlinks) ] templ
         exportedTheory = exportTheory mfii (tideps tinfo) (titheory tinfo) -}

-- | takes the template and the database of theorems and renders to html 
exportTheories :: String -> KnowledgeBase -> [(String,String)]
exportTheories templ kb = map expThr tinfos where
   mfii = getRefsInfo $ kbformalitems kb
   tinfos = kbtheories kb
   tlinks = concatMap (thrylink . tiname) tinfos
   expThr tinfo = ("isarhtml/" ++ (tiname tinfo) ++ ".html", thHtml)
      where
         thHtml = replaceAll [ ("this is theory placeholder", exportedTheory)
                             , ("this is links placeholder", tlinks) ] templ
         exportedTheory = exportTheory mfii (tideps tinfo) (titheory tinfo)
         

-- | makes a link to a rendered theory for the theory name
thrylink :: String -> String
thrylink thn = 
   "\n<p><a href=\"" ++ thn ++ ".html\">" ++ thn ++ "</a></p>\n"

-- | converts Isar theory to HTML markup, the main function of this module
-- mfii is a map of all known theorems rendered in simplified form,
-- refs - list of all references in all proofs of the theory
-- th - a parsed theory 
exportTheory ::  M.Map String String -> [String] -> Theory -> String
exportTheory mfii refs th = 
   uniquefyids ["hstrigger", "hscontent", "hintref","pardiv"] $
   ( mkformal $ 
   (bf "theory") ++ (name th) ++ (bf "imports") ++ 
   -- ( unwords $ map tlink $ imports th ) ) ++ to do: make theory links
   ( unwords $ imports th ) ) ++
   ( mkformal $ bf "begin\n" ) ++
   ( exp2inftext $ thIntro th ) ++
   ( unlines $ map (exportSection mfii) (thsections th) ) ++
   ( mkformal (bf "end\n") ) ++
   ( unlines $ map (createRefDiv mfii) refs ) 
   
 
-- | exports a section (same)
exportSection :: M.Map String String -> Section ->  String
exportSection m s = 
   ( hd $ sectitle s) ++ "\n" ++
   ( exp2inftext $ secIntro s) ++
   ( unlines $ map (exportItem m) $ items s )


-- | exports an item in a section (same)
exportItem :: M.Map String String -> Item -> String
exportItem mfii it = ( exp2inftext $ description it ) ++ "\n" ++
                ( exportFormalItem mfii $ formalItem it )


-- | exports formal items
exportFormalItem :: M.Map String String -> FormalItem -> String
exportFormalItem mfii (Definition id _ _ df) = mkformal $ ( par $ bf "Definition\n" ) ++ (par $ isar2latex df )
exportFormalItem mfii (Locale nm (parent,vars) itms) = mkformal $ 
   ( par $ (bf "Locale ") ++ nm ++ 
      ( if null parent then "" else " = " ++ parent ++ " " ++ (unwords vars) ++ " +") ) ++
   (unlines $ map exportLocaleItem itms)
exportFormalItem mfii (FormalItem p) = exportProposition mfii p

-- | export locale item
exportLocaleItem :: LocaleItem -> String
exportLocaleItem (Fixes nmnt) = ""

exportLocaleItem (Defines _ ld) =
   par $ ( bf "defines " ) ++ (isar2latex ld)

exportLocaleItem (Assumes clms ) =
   par $ ( bf "assumes " ) ++ (exportClaims clms)

-- | exports a single name with notation, as used in the "fixes"
-- locale item, just write the name, the notation is 
-- evident from the "defines" line (same)
exportNameNotation :: NameNotation -> String
exportNameNotation (NameNotation nm nt) = (isar2latex nm) ++ ", "


-- | exports proposition
exportProposition ::  M.Map String String -> Proposition -> String
exportProposition mfii p = mkformal $
   par ( ( bf $ proptype p ) ++ " " ++
   ( inContext $ context p) ++
   ( propname p ) ++ ":\n") ++
   par ( ( if not $ null $ propprems p then
      ( bf "   assumes " ) ++ ( exportClaims $ propprems p )
   else "" ) ) ++
   ( bf "   shows " ) ++ ( exportClaims $ claims p ) ++
   ( exportProof mfii $ propproof p)

-- | exports a sequence of labelled claims (same)
exportClaims :: [(String,[String])] -> String
exportClaims =  concat . (intersperse (bf "and ") ) . (map exportOneClaim )

-- | exports a single labelled claim (same)
exportOneClaim :: (String,[String]) -> String
exportOneClaim (label,claims) = 
   label ++ (if null label then "" else ": ") ++ 
   ( concat $ append2Init ",  " $ map isar2latex claims )


-- | exports a proof
exportProof :: M.Map String String -> Proof -> String
exportProof mfii ( UsingBy useunf uprops tac ) =
   if not $ null uprops then 
      (bf useunf) ++ " " ++ (unwords $ intersperse ", " $ map (getRefSlider mfii) uprops )
   else " "


exportProof mfii (ByRule s) =
   (bf "  by (rule ") ++ (getRefSlider mfii s) ++ ( bf ")")

exportProof mfii (LongProof d ps) =
   slider "proof" 
      ( rmdnl $ ( unlines $ map ( exportProofStep mfii ) ps ) ++ ( bf "qed" ) )

-- | exports a proof step 
exportProofStep ::  M.Map String String -> ProofStep -> String

exportProofStep mfii (LongReasoning rs mbs) = 
   ( exportReasoning mfii rs ) ++ "\n" ++
   ( sunlines $ map (exportMoreoverBody mfii) mbs)

exportProofStep _ (FixStep v) = pd $
   (bf "fix ") ++ (unwords $ map isar2latex v)

exportProofStep _ (LetStep v d) = pd $
   (bf "let ") ++ ( isar2latex (v ++ " = " ++ d) )

exportProofStep _ Next = pd $ bf "next "


-- | exports a MoreoverBody (same)
exportMoreoverBody :: M.Map String String -> MoreoverBody -> String
exportMoreoverBody mfii mb = pd $
   ( unlines $ map (prepKeyWord . (exportReasoning mfii)) (mrvMrvs mb) ) ++
   ( bf ultfinal ) ++ (exportProofCommand mfii (ultimfinal mb)) ++ "\n" ++
   ( sunlines $ map (exportConnectedStep mfii) (followup mb) )
   where
      s = mrvalso mb
      prepKeyWord = (((bf s) ++ " ") ++)
      ultfinal = if s == "also" then "finally " else "ultimately "


-- | exports a reasoning (same)
exportReasoning :: M.Map String String -> Reasoning -> String
exportReasoning mfii (Reasoning is css ) = 
	( exportInitStep mfii is) ++ "\n" ++
        ( sunlines $ map (exportConnectedStep mfii) css )

-- | exports Init Step
exportInitStep :: M.Map String String  -> InitStep -> String
exportInitStep mfii (InitStep loclabs pc) = pd $ 
   (if not $ null loclabs then 
      ( exportLocRefs "from " loclabs )
   else "") ++
   (exportProofCommand mfii pc)

exportInitStep mfii (StepBlock pss) =
   ( slider "{" ( unlines $ map (exportProofStep mfii) pss ) ) ++
   ( bf "}" )

exportInitStep mfii (Assume clms) = 
   pd $ (bf "assume " ) ++ exportClaims clms

exportInitStep mfii (Note refs) = pd $
   ( bf "note " ) ++ (unwords $ map exportLocRef refs)

-- | exports init or connected proof local references
-- wft can be "with", "from", "then" (same)
exportLocRefs :: String -> [String] -> String
exportLocRefs wft loclabs =
   (bf wft) ++ (concat $ append2Init ", " $ map exportLocRef loclabs) ++ " "


-- | export local reference. It may be a label or latex text in backquotes`
exportLocRef :: String -> String
exportLocRef s = if (head s) == '`' then isar2latex $ tail $ init s
                 else s


-- | exports a connected step (like "with", "then" etc, those that
-- refere to the previous step
exportConnectedStep ::  M.Map String String -> ConnectedStep -> String
exportConnectedStep mfii (WithStep loclabs pc) = pd $
   ( exportLocRefs "with " loclabs ) ++ ( exportProofCommand mfii pc )

exportConnectedStep mfii ( ThenStep pc) = pd $
   (bf "then ") ++ ( exportProofCommand mfii pc)

exportConnectedStep mfii (HenceThus ht clms tac) = pd $
   (bf ht) ++ " " ++ (exportClaims clms)

-- | exports a proof command (same)
exportProofCommand :: M.Map String String -> ProofCommand -> String
exportProofCommand mfii (PChaveShow hs cp) = 
   (bf hs ) ++ " " ++ (exportClaimProof mfii cp)
exportProofCommand mfii (PCbtain vars cp) =
   (bf "obtain ") ++ (unwords $ map isar2latex vars) ++ 
   (bf "where ") ++ (exportClaimProof mfii cp)

-- | exports a claim with the proof (same)
exportClaimProof ::M.Map String String -> ClaimProof -> String
exportClaimProof mfii cp = 
   ( exportClaims (cpclaims cp) ) ++
   ( exportProof mfii (cpproof cp) )

-- | takes the map of formal items and a name of theorem,
-- returns a div that can be clicked on to show the hint 
getRefSlider :: M.Map String String -> String -> String
getRefSlider mfii nm = if isJust lookres then 
                          ( "<span id=\"hintref\">" ++ nmAfterDot ++ "</span>" )
                       else nm
                          where
                             nmAfterDot = reverse $ takeWhile (/='.') $ reverse nm
                             lookres = (M.lookup nmAfterDot) mfii


-- | create a lookup table of descriptions of referenced theorems
getRefsInfo :: [FormalItemInfo] -> (M.Map String String)
getRefsInfo =  M.fromList . (map addKey)
   where addKey fii = ( getSFIname $ fimitem fii, exportFormalItemInfo fii)

-- | exports a single simple (unlabelled) claim (same)
exportOneSimpleClaim :: [String] -> String
exportOneSimpleClaim claims =  
   unwords $ append2Init ",  " $ map isar2latex claims 

-- | exports simple form of claims    
exportSimpleClaims :: [[String]] -> String
-- exportSimpleClaims = (breakClaims 0) . (map exportOneSimpleClaim)
exportSimpleClaims = concat . intersperse (bf " and " ). (map exportOneSimpleClaim)

isar2latex :: String -> String
isar2latex s = remElems "`?" $ "\\( " ++ 
   (replace "\n" "\\)\n\\(" $ replaceAll inner2LatexSym $ convBraces s) ++ " \\)"

-- | makes unique every string occurence, adding the count as a div
-- at the end of the string
uniquefy :: String -> String -> String
uniquefy id htmlstring = st ++ 
   "<div id=\"par_" ++ id ++ "\" style=\" display: none\">" ++ (show count) ++ "</div>\n"  
   where 
      sti = foldr incrcount ("",0) htmlstring
      st = fst sti
      count = snd sti
      incrcount c (s,i) = 
         if id `isPrefixOf` ns then (id ++ (show i) ++ (drop (length id) ns), i+1)
         else (ns,i)
            where ns = c:s

-- composes some number of uniquefy fo a list of id strings
uniquefyids :: [String] -> String -> String
uniquefyids ids = foldr1 (.) (map uniquefy ids)


