{- 
	This file is part of tiddlyisar - a tool for rendering IsarMathLib 
	theories in TiddlyWiki format.
    Copyright (C) 2008  Slawomir Kolodynski

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
-}


module ProcessThys
   where

import qualified Data.Map as M
import IMLP_datatypes

-- | this module deals with processing information obtained 
-- by parsing theory files. For now it extracts the list of
-- definitions and theorems from the theories.


-- | takes a list of formal items info and makes of it a map 
-- indexed by name
infoMap ::  [FormalItemInfo] -> (M.Map String FormalItemInfo)
infoMap = M.fromList . (map addKey)
   where addKey fii = ( getSFIname $ fimitem fii, fii)


-- | gets the name from a formal item in simple form:
getSFIname :: SimpleFormalItem -> String
getSFIname (SimpleProp _ _ nm _ _) = nm
getSFIname (SimpleDef nm _ ) = nm ++ "_def"
getSFIname OtherSimpleItem = "a context"

-- | converts a formal item to a a simplified form.
formal2simple :: FormalItem -> SimpleFormalItem

formal2simple (Definition name spec def) = SimpleDef name def

formal2simple (FormalItem prop) = 
   SimpleProp { sproptype = proptype prop
              , scontext  = context prop
              , spropname = propname prop
              , sproprems= map snd (propprems prop)
              , sclaims   = map snd (claims prop) }

formal2simple _ = OtherSimpleItem 

-- | gets all the formal items in th simple form from a section

getThmsDefsFromSec :: Section -> [SimpleFormalItem]
getThmsDefsFromSec (Section tit intro ims) =
   map (formal2simple . formalItem) ims

-- | gets all formal items from a theory in simple form
getThmsDefsFromThry :: Theory -> [FormalItemInfo]
getThmsDefsFromThry t = 
   map (addTheoryName (name t)) (concat $ map getThmsDefsFromSec (thsections t))

-- | converts a list of simple formal items into a list of 
-- formal item infos by adding the name of theory
addTheoryName :: String -> SimpleFormalItem -> FormalItemInfo
addTheoryName nme item = FormalItemInfo { fimTheoryName = nme, fimitem = item }

-- | gets all formal items in simple form from a list 
-- of theories

getThmsDefsFromTheories :: [Theory] -> [FormalItemInfo]
getThmsDefsFromTheories = concat . map getThmsDefsFromThry


