{- 
	This file is part of tiddlyisar - a tool for rendering IsarMathLib 
	theories in TiddlyWiki format.
    Copyright (C) 2008  Slawomir Kolodynski

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
-}
    
-- | The function exportTheory in this module takes a parsed 
-- IsarMathLib theory and produces corresponding TiddlyWiki markup.


module Export2wiki
   where

import List(intersperse)
import Char(isAlphaNum)
import qualified Data.Map as M
import Data.Maybe(isJust, fromJust) 
import Utils
import IMLP_datatypes
import IsarSym2Latex
import ProcessThys

-- | converts literal LaTeX, just addding $ ... $
convLatex :: String -> String
convLatex s = "$" ++ s ++ "$"

-- | converts literal text: @{text "some text"}, for now it just 
-- treats it as isar text
littext :: String -> String
littext s = if (isIdentifier s) then it s else isar2latex s

-- | checks if the text is an identifier. Those consist of 
-- alphanumeric characters and underscore
isIdentifier :: String -> Bool
isIdentifier = all isallowed 
   where isallowed c = (isAlphaNum c) || (c == '_') || (c == '.') || (c == ' ')

-- | make text italic
it :: String -> String
it s = "//" ++ s ++ "//"

-- | adds new lines at the beginning and end
nl :: String -> String
nl s = "\n" ++ s ++ "\n"

-- | makes a string boldface, also surrounds with spaces
bf :: String -> String
bf s = " ''" ++ s ++ "'' "

-- | creates a heading
hd :: String -> String
hd s = "\n!" ++ s

-- | indents text
indent :: String -> String
indent s = "{{text{" ++ s ++ "}}}"

-- | removes spurious spaces and single newlines, but leaves double lines
cleanlines :: String -> String 
cleanlines = unlines . (map (unwords . words)) . splitList "\n  \n"

-- | exports informal text
exp2inftext :: String -> String
exp2inftext = ("\n" ++) . cleanlines . 
            ( appBetween littext "@{text \"" "\"}" )
            -- ( appBetween convLatex "$" "$" )


-- | creates a floating slider
floatSlider :: String -> String -> String -> String
floatSlider link tooltip contents = 
   "+++^[" ++ link ++ " | " ++ tooltip  ++ " ]... " ++ contents ++ "=== "

-- | creates a slider
slider :: String -> String -> String
slider title contents = 
   "+++[" ++ title ++ " ]>\n" ++ contents ++ "=== "

-- | creates a blockquote. This is in fact a slider, opan by default
-- title is the text that folds the block on click
bquote :: String ->  String -> String
bquote title contents =
   "++++[" ++ title ++ " ]>\n" ++ contents ++ "=== "

-- | surrounds with [[]] to indicate link
tlink :: String -> String
tlink s = "[[" ++ s ++ "]]"


-- | makes the text nowiki
noWiki :: String -> String
noWiki s = "<nowiki>" ++ s ++ "</nowiki>"

inContext :: String -> String
inContext s = if (not $ null s) then "''(in'' " ++ s ++ "'')'' " else ""

-- | exports FormalItemInfo
exportFormalItemInfo :: FormalItemInfo -> String
exportFormalItemInfo (FormalItemInfo tn (SimpleProp tpe cntxt nme prms clms)) =
   floatSlider nme tn ( 
   ( bf tpe) ++ " " ++
   (inContext cntxt) ++
   (noWiki nme) ++ ": " ++ 
   (if not $ null prms then (bf "assumes ") ++ (exportSimpleClaims prms) else "") ++ 
   ( bf " shows " ) ++ (exportSimpleClaims clms) ) 

exportFormalItemInfo (FormalItemInfo tn ( SimpleDef nm d )) = floatSlider 
   (nm ++ "_def")  tn ("Definition of " ++ (noWiki nm) ++ ":\n" ++ isar2latex d)

exportFormalItemInfo (FormalItemInfo tn OtherSimpleItem) = " Other formal item "

-- | converts Isar theory to a blog entry
exportTheory ::  M.Map String String -> Theory -> String
exportTheory mfii th = 
   (bf "theory") ++ (name th) ++ (bf "imports") ++ 
   ( unwords $ map tlink $ imports th ) ++
   "\n\n" ++
   ( bf "begin\n" ) ++
   ( exp2inftext $ thIntro th ) ++
   ( unlines $ map (exportSection mfii) (thsections th) ) ++
   (bf "end\n\n") ++ 
   "+++![Comments on " ++ (name th) ++ 
   "|click to add comment] &lt;html&gt; &lt;div&gt; &lt;iframe style=&quot;width:60%;height:500px&quot; src=&quot;http://www.haloscan.com/comments/slawekk/" ++ 
    (name th) ++ "&quot;&gt;&lt;/iframe&gt; &lt;/div&gt; &lt;/html&gt;\n=== \n"


-- | exports a section
exportSection :: M.Map String String -> Section ->  String
exportSection m s = 
   ( hd $ sectitle s) ++ "\n" ++
   ( exp2inftext $ secIntro s) ++
   ( unlines $ map (exportItem m) $ items s )

-- | exports an item in a section 
exportItem :: M.Map String String -> Item -> String
exportItem mfii it = ( exp2inftext $ description it ) ++ "\n" ++
                ( exportFormalItem mfii $ formalItem it )

-- | exports formal items
exportFormalItem :: M.Map String String -> FormalItem -> String
exportFormalItem mfii (Definition id sp df) = ( bf "Definition\n" ) ++ ( isar2latex df )
exportFormalItem mfii (Locale nm (parent,vars) itms) = rmdnl $ 
   (bf "Locale ") ++ nm ++ 
   (if null parent then "\n" else " = " ++ parent ++ " " ++ (unwords vars) ++ " +\n") ++
   (unlines $ map exportLocaleItem itms)
exportFormalItem mfii (FormalItem p) = exportProposition mfii p

-- | export locale item
exportLocaleItem :: LocaleItem -> String
exportLocaleItem (Fixes nmnt) = ""
--   (bf "fixes ") ++ (concatMap exportNameNotation nmnt)

exportLocaleItem (Defines nm ld) =
   ( bf "defines " ) ++ (isar2latex ld)

exportLocaleItem (Assumes clms ) =
   ( bf "assumes " ) ++ (exportClaims clms)

-- | exports a single name with notation, as used in the "fixes"
-- locale item, just write the name, the notation is 
-- evident from the "defines" line
exportNameNotation :: NameNotation -> String
exportNameNotation (NameNotation nm nt) = (isar2latex nm) ++ ", "  

-- | exports proposition
exportProposition ::  M.Map String String -> Proposition -> String
exportProposition mfii p = 
   ( bf $ proptype p ) ++ " " ++
   (inContext $ context p) ++
   ( noWiki $ propname p ) ++ ":\n" ++
   (if not $ null $ propprems p then
      ( bf "   assumes " ) ++ ( exportClaims $ propprems p )
   else "" ) ++
   ( bf "   shows " ) ++ ( exportClaims $ claims p ) ++
   (exportProof mfii $ propproof p)

-- | exports a sequence of labelled claims, breaking
exportClaims :: [(String,[String])] -> String
exportClaims =  concat . (intersperse (bf "and ") ) . (map exportOneClaim )

-- | break a sequence of labelled claims into lines
breakClaims :: [String] -> String
breakClaims claims = sunlines $ foldr breakline [last claims] (init claims)
   where breakline new (cur:prev) =
            if (length new) + (length cur) > 100
            then (new ++ (bf " and")):cur:prev
            else (new ++ (bf " and ") ++ cur):prev

-- | exports a single labelled claim
exportOneClaim :: (String,[String]) -> String
exportOneClaim (label,claims) = 
   label ++ (if null label then "" else ": ") ++ 
   ( concat $ append2Init ",  " $ map isar2latex claims )


-- | exports proof
exportProof :: M.Map String String -> Proof -> String
exportProof mfii ( UsingBy useunf uprops tac ) =
   if not $ null uprops then 
      (bf useunf) ++ " " ++ (unwords $ intersperse ", " $ map (getRefSlider mfii) uprops )
   else " "

exportProof mfii (ByRule s) =
   (bf "   by (rule ") ++ (getRefSlider mfii s) ++ ( bf ")")

exportProof mfii (LongProof d ps) =
   slider "proof" 
      ( rmdnl $ ( unlines $ map ( exportProofStep mfii ) ps ) ++ ( bf "qed" ) )

-- | exports a proof step 
exportProofStep ::  M.Map String String -> ProofStep -> String

exportProofStep mfii (LongReasoning rs mbs) =
   ( exportReasoning mfii rs ) ++ "\n" ++
   ( sunlines $ map (exportMoreoverBody mfii) mbs)

exportProofStep _ (FixStep v) =
   (bf "fix ") ++ (unwords $ map isar2latex v)

exportProofStep _ (LetStep v d) =
   (bf "let ") ++ ( isar2latex (v ++ " = " ++ d) )

exportProofStep _ Next = bf "next "


-- | exports a MoreoverBody
exportMoreoverBody :: M.Map String String -> MoreoverBody -> String
exportMoreoverBody mfii mb =
   ( unlines $ map (prepKeyWord . (exportReasoning mfii)) (mrvMrvs mb) ) ++
   ( bf ultfinal ) ++ (exportProofCommand mfii (ultimfinal mb)) ++ "\n" ++
   ( sunlines $ map (exportConnectedStep mfii) (followup mb) )
   where
      s = mrvalso mb
      prepKeyWord = (((bf s) ++ " ") ++)
      ultfinal = if s == "also" then "finally " else "ultimately "
   

-- | exports a reasoning
exportReasoning :: M.Map String String -> Reasoning -> String
exportReasoning mfii (Reasoning is css ) = 
	( exportInitStep mfii is) ++ "\n" ++
        ( sunlines $ map (exportConnectedStep mfii) css )

-- | exports Init Step
exportInitStep :: M.Map String String  -> InitStep -> String
exportInitStep mfii (InitStep loclabs pc) = 
   (if not $ null loclabs then 
      ( exportLocRefs "from " loclabs )
   else "") ++
   (exportProofCommand mfii pc)

exportInitStep mfii (StepBlock pss) =
   ( bquote "{" ( unlines $ map (exportProofStep mfii) pss ) ) ++
   ( bf "}" )

exportInitStep mfii (Assume clms) = (bf "assume " ) ++ exportClaims clms

exportInitStep mfii (Note refs) = 
   ( bf "note " ) ++ (unwords $ map exportLocRef refs)


-- | exports init or connected proof local references
-- wft can be "with", "from", "then"
exportLocRefs :: String -> [String] -> String
exportLocRefs wft loclabs =
   (bf wft) ++ (concat $ append2Init ", " $ map exportLocRef loclabs) ++ " "

-- | export local reference. It may be a label or latex text in backquotes`
exportLocRef :: String -> String
exportLocRef s = if (head s) == '`' then isar2latex $ tail $ init s
                 else s

-- | exports a connected step (like "with", "then" etc, those that
-- refere to the previous step
exportConnectedStep ::  M.Map String String -> ConnectedStep -> String
exportConnectedStep mfii (WithStep loclabs pc) =
   ( exportLocRefs "with " loclabs ) ++ ( exportProofCommand mfii pc )

exportConnectedStep mfii ( ThenStep pc) =
   (bf "then ") ++ ( exportProofCommand mfii pc)

exportConnectedStep mfii (HenceThus ht clms tac) =
   (bf ht) ++ " " ++ (exportClaims clms)

-- | exports a proof command
exportProofCommand :: M.Map String String -> ProofCommand -> String
exportProofCommand mfii (PChaveShow hs cp) = 
   (bf hs ) ++ " " ++ (exportClaimProof mfii cp)
exportProofCommand mfii (PCbtain vars cp) =
   (bf "obtain ") ++ (unwords $ map isar2latex vars) ++ 
   (bf "where ") ++ (exportClaimProof mfii cp)

-- | exports a claim with the proof
exportClaimProof ::M.Map String String -> ClaimProof -> String
exportClaimProof mfii cp = 
   ( exportClaims (cpclaims cp) ) ++
   ( exportProof mfii (cpproof cp) )


-- | takes the map of formal items and a name of theorem,
-- returns a floating slider with the theorem statement
getRefSlider :: M.Map String String -> String -> String
getRefSlider mfii nm = if isJust lookres then fromJust lookres
                       else (noWiki nm)
                          where
                             nmAfterDot = reverse $ takeWhile (/='.') $ reverse nm
                             lookres = (M.lookup nmAfterDot) mfii


-- | exports a list of theories to wiki form 
exportTheories :: [Theory] -> [(String,String)]
exportTheories tt = map expThrs tt where 
   mfii = getRefsInfo $ getThmsDefsFromTheories tt 
   expThrs t = (name t ++ ".tid", exportTheory mfii t)

-- converts exported theories (the output of exportTheories) to pairs
-- [("theory name placeholder", "internal tiddly theory markup"]
wiki2internal :: (String,String) -> (String,String)
wiki2internal (nm,th) = 
   ( "This is " ++ (takeWhile (/='.') nm) ++ " placeholder", toInternalCode th)

-- | fills the tiddler placeholders in an empty wiki
-- with theories. The expected input is ax produced by
-- exportTheories
fillWiki :: [(String,String)] -> String -> String
fillWiki theories = replaceAll (map wiki2internal theories)


-- | create a lookup table of descriptions of referenced theorems
getRefsInfo :: [FormalItemInfo] -> (M.Map String String)
getRefsInfo =  M.fromList . (map addKey)
   where addKey fii = ( getSFIname $ fimitem fii, exportFormalItemInfo fii)
   
-- | exports a single simple (unlabelled) claim
exportOneSimpleClaim :: [String] -> String
exportOneSimpleClaim claims =  
   breakWords $ append2Init ",  " $ map isar2latex claims 

-- | exports simple form of claims    
exportSimpleClaims :: [[String]] -> String
-- exportSimpleClaims = (breakClaims 0) . (map exportOneSimpleClaim)
exportSimpleClaims = concat . intersperse " ''and'' " . (map exportOneSimpleClaim)

-- | breaks a sequence of words into lines that are not too long
breakWords :: [String] -> String 
breakWords ws = concat $ foldr breakline [""] ws
   where breakline new (cur:prev) =
            if (length new) + (length cur) > 100
            then ( new ++ "\n" ):cur:prev
            else  ( (new ++ " " ++ cur):prev )
   
-- | break a sequence of labelled claims into lines
{- breakClaims :: Int -> [String] -> String
breakClaims n claims = unlines $ foldr breakline [last claims] (init claims)
   where breakline new (cur:prev) =
            if (length new) + (length cur) > 75 
            then (new ++ (bf " and")):(indent n cur):prev
            else (new ++ (bf " and ") ++cur):prev -}


-- | translates inner Isar to LaTeX
isar2latex :: String -> String
isar2latex s = remElems "`?" $ "$ " ++ 
   (replace "\n" "$\n$" $ replaceAll inner2LatexSym $ convBraces s) ++ "$"

-- | The text of a tiddler in the TiddlyWiki code as (read by an editor)
-- seems to different only by replacing "<" with "&lt;" and ">" with "&gt;"
-- and we are using those only in the the <nowiki>text</nowiki> construct

toInternalCode :: String -> String
toInternalCode = replaceAll [("<nowiki>","&lt;nowiki&gt;")
                            ,("</nowiki>","&lt;/nowiki&gt;")]



{-+++[Comments|click to leave comment]
&lt;html&gt; &lt;div&gt; &lt;iframe style=&quot;width:50%;height:400px&quot; src=&quot;http://www.haloscan.com/comments/slawekk/tiddlyformalmath&quot;&gt;&lt;/iframe&gt; &lt;/div&gt; &lt;/html&gt;

-}
