(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES LOSS OF USE, DATA, OR PROFITS OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header{*\isaheader{ZF1.thy}*}

theory ZF1 imports equalities

begin

text{*Standard Isabelle distribution contains lots of facts about basic set
  theory. This theory file adds some more.*}

section{*Lemmas in Zermelo-Fraenkel set theory*}

text{*Here we put lemmas from the set theory that we could not find in 
the standard Isabelle distribution.*}

text{*If all sets of a nonempty collection are the same, then its union 
  is the same.*}

lemma ZF1_1_L1: assumes "C\<noteq>0" and "\<forall>y\<in>C. b(y) = A" 
  shows "(\<Union>y\<in>C. b(y)) = A" using assms by blast
  
text{*The union af all values of a constant meta-function belongs to 
the same set as the constant.*}

lemma ZF1_1_L2: assumes A1:"C\<noteq>0" and A2: "\<forall>x\<in>C. b(x) \<in> A" 
  and A3: "\<forall>x y. x\<in>C \<and> y\<in>C \<longrightarrow> b(x) = b(y)"
  shows "(\<Union>x\<in>C. b(x))\<in>A"
proof -
  from A1 obtain x where D1:"x\<in>C" by auto
  with A3 have "\<forall>y\<in>C. b(y) = b(x)" by blast
  with A1 have "(\<Union>y\<in>C. b(y)) = b(x)" 
    using ZF1_1_L1 by simp
  with D1 A2 show ?thesis by simp
qed

text{*If two meta-functions are the same on a cartesian product,
  then the subsets defined by them are the same. I am surprised Isabelle
  can not handle this automatically.*}

lemma ZF1_1_L4: assumes A1: "\<forall>x\<in>X.\<forall>y\<in>Y. a(x,y) = b(x,y)"
  shows "{a(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y} = {b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y}"
proof
  show "{a(x, y). \<langle>x,y\<rangle> \<in> X \<times> Y} \<subseteq> {b(x, y). \<langle>x,y\<rangle> \<in> X \<times> Y}"
  proof
    fix z assume "z \<in> {a(x, y) . \<langle>x,y\<rangle> \<in> X \<times> Y}"
    with A1 show  "z \<in> {b(x,y).\<langle>x,y\<rangle> \<in> X\<times>Y}" by auto   
  qed
  show "{b(x, y). \<langle>x,y\<rangle> \<in> X \<times> Y} \<subseteq> {a(x, y). \<langle>x,y\<rangle> \<in> X \<times> Y}"
  proof
    fix z assume "z \<in> {b(x, y). \<langle>x,y\<rangle> \<in> X \<times> Y}"
    with A1 show "z \<in> {a(x,y).\<langle>x,y\<rangle> \<in> X\<times>Y}" by auto
  qed
qed

text{*If two meta-functions are the same on a cartesian product,
  then the subsets defined by them are the same. 
  This is similar to @{text "ZF1_1_L4"}, except that
  the set definition varies over @{text "p\<in>X\<times>Y"} rather than 
  @{text "<x,y>\<in>X\<times>Y"}.*}

lemma ZF1_1_L4A: assumes A1: "\<forall>x\<in>X.\<forall>y\<in>Y. a(<x,y>) = b(x,y)"
  shows "{a(p). p \<in> X\<times>Y} = {b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y}"
proof
  { fix z assume "z \<in> {a(p). p\<in>X\<times>Y}"
    then obtain p where D1: "z=a(p)" "p\<in>X\<times>Y" by auto
    let ?x = "fst(p)" let ?y = "snd(p)"
    from A1 D1 have "z \<in> {b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y}" by auto
  } then show "{a(p). p \<in> X\<times>Y} \<subseteq> {b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y}" by blast
next 
  { fix z assume "z \<in> {b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y}"
    then obtain x y where D1: "\<langle>x,y\<rangle> \<in> X\<times>Y" "z=b(x,y)" by auto
    let ?p = "<x,y>" 
    from A1 D1 have "?p\<in>X\<times>Y" "z = a(?p)" by auto
    then have "z \<in> {a(p). p \<in> X\<times>Y}" by auto
  } then show "{b(x,y). \<langle>x,y\<rangle> \<in> X\<times>Y} \<subseteq> {a(p). p \<in> X\<times>Y}" by blast
qed

text{*If two meta-functions are the same on a set, then they define the same
  set by separation.*}

lemma ZF1_1_L4B: assumes "\<forall>x\<in>X. a(x) = b(x)"
  shows "{a(x). x\<in>X} = {b(x). x\<in>X}"
  using assms by simp

text{*A set defined by a constant meta-function is a singleton.*}

lemma ZF1_1_L5: assumes "X\<noteq>0" and "\<forall>x\<in>X. b(x) = c"
  shows "{b(x). x\<in>X} = {c}" using assms by blast

text{* Most of the time, @{text "auto"} does this job, but there are strange 
  cases when the next lemma is needed. *}

lemma subset_with_property: assumes "Y = {x\<in>X. b(x)}"
  shows "Y \<subseteq> X" 
  using assms by auto

text{*We can choose an element from a nonempty set.*}

lemma nonempty_has_element: assumes "X\<noteq>0" shows "\<exists>x. x\<in>X"
  using assms by auto

text{*In Isabelle/ZF the intersection of an empty family is 
  empty. This is exactly lemma @{text "Inter_0"} from Isabelle's
  @{text "equalities"} theory. We repeat this lemma here as it is very
  difficult to find. This is one reason we need comments before every 
  theorem: so that we can search for keywords.*}

lemma inter_empty_empty: shows "\<Inter>0 = 0" by (rule Inter_0);

text{*If an intersection of a collection is not empty, then the collection is
  not empty. We are (ab)using the fact the the intesection of empty collection 
  is defined to be empty.*}

lemma inter_nempty_nempty: assumes "\<Inter>A \<noteq> 0" shows "A\<noteq>0"
  using assms by auto;

text{*For two collections $S,T$ of sets we define the product collection
  as the collections of cartesian products $A\times B$, where $A\in S, B\in T$.*}

definition
  "ProductCollection(T,S) \<equiv> \<Union>U\<in>T.{U\<times>V. V\<in>S}"

text{*The union of the product collection of collections $S,T$ is the 
  cartesian product of $\bigcup S$ and  $\bigcup T$. *}

lemma ZF1_1_L6: shows "\<Union> ProductCollection(S,T) = \<Union>S \<times> \<Union>T"
  using ProductCollection_def by auto

text{*An intersection of subsets is a subset.*}

lemma ZF1_1_L7: assumes A1: "I\<noteq>0" and A2: "\<forall>i\<in>I. P(i) \<subseteq> X"
  shows "( \<Inter>i\<in>I. P(i) ) \<subseteq> X"
proof -
  from A1 obtain i\<^isub>0 where "i\<^isub>0 \<in> I" by auto
  with A2 have "( \<Inter>i\<in>I. P(i) ) \<subseteq> P(i\<^isub>0)" and "P(i\<^isub>0) \<subseteq> X"
    by auto
  thus "( \<Inter>i\<in>I. P(i) ) \<subseteq> X" by auto
qed

text{*Isabelle/ZF has a "THE" construct that allows to define an element
  if there is only one such that is satisfies given predicate.
  In pure ZF we can express something similar using the indentity proven below.*}

lemma ZF1_1_L8: shows "\<Union> {x} = x" by auto

text{*Some properties of singletons.*}

lemma ZF1_1_L9: assumes A1: "\<exists>! x. x\<in>A \<and> \<phi>(x)"
  shows 
  "\<exists>a. {x\<in>A. \<phi>(x)} = {a}"
  "\<Union> {x\<in>A. \<phi>(x)} \<in> A"
  "\<phi>(\<Union> {x\<in>A. \<phi>(x)})"
proof -
  from A1 show "\<exists>a. {x\<in>A. \<phi>(x)} = {a}" by auto
  then obtain a where I: "{x\<in>A. \<phi>(x)} = {a}" by auto
  then have "\<Union> {x\<in>A. \<phi>(x)} = a" by auto
  moreover
  from I have "a \<in> {x\<in>A. \<phi>(x)}" by simp
  hence "a\<in>A" and "\<phi>(a)" by auto
  ultimately show "\<Union> {x\<in>A. \<phi>(x)} \<in> A" and "\<phi>(\<Union> {x\<in>A. \<phi>(x)})"
    by auto
qed

text{*Adding an element of a set to that set does not change the set.*}

lemma set_elem_add: assumes "x\<in>X" shows "X \<union> {x} = X" using assms 
  by auto

text{*Here we define a restriction of a collection of sets to a given set. 
In romantic math this is typically denoted $X\cap M$ and means 
$\{X\cap A : A\in M \} $. Note there is also restrict$(f,A)$ 
defined for relations in ZF.thy.*}

definition
  RestrictedTo (infixl "{restricted to}" 70) where
  "M {restricted to} X \<equiv> {X \<inter> A . A \<in> M}"

text{*Next we show a technical identity that is used to prove sufficiency 
  of some condition for a collection of sets to be a base for a topology. *}

lemma ZF1_1_L10: assumes A1: "\<forall>U\<in>C. \<exists>A\<in>B. U = \<Union>A" 
  shows "\<Union>\<Union> {\<Union>{A\<in>B. U = \<Union>A}. U\<in>C} = \<Union>C"
proof
  show "\<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A}) \<subseteq> \<Union>C" by blast
  show "\<Union>C \<subseteq> \<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A})"
  proof
    fix x assume "x \<in> \<Union>C" 
    show "x \<in> \<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A})"
    proof -
      from `x \<in> \<Union>C` obtain U where "U\<in>C \<and> x\<in>U" by auto
      with A1 obtain A where "A\<in>B \<and> U = \<Union>A" by auto
      from `U\<in>C \<and> x\<in>U` `A\<in>B \<and> U = \<Union>A` show "x\<in> \<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A})" 
	by auto
    qed
  qed
qed;

end
