{- 
	This file is part of tiddlyisar - a too for rendering IsarMathLib 
	theories in TiddlyWiki format.
    Copyright (C) 2008  Slawomir Kolodynski

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
-}



module IsarSym2Latex
   where

import Utils(appBetween,appToParts,nestLevel)
import Data.List(isInfixOf)
import Data.Char(isLetter)

-- | this module contains tables for translating Isabelle symbols to LaTeX

-- | list of translations from Isar symbols to LaTeX symbols
inner2LatexSym = [("\\<longrightarrow>", "\\longrightarrow ")
                 ,(".", ".\\ ")
                 ,("\\<and>", "\\wedge ")
                 ,("\\<approx>", "\\approx ")
                 ,("\\<subseteq>", "\\subseteq ")
                 ,("\\<times>", "\\times ") 
                 ,("\\<rightarrow>", "\\rightarrow ")
                 ,("\\<langle>", "\\langle ")
                 ,("\\<lesssim>", "\\preceq ")
                 ,("\\<rangle>", "\\rangle ")
                 ,("\\<cdot>", "\\cdot ")
                 ,("\\<dots>", "\\ldots ")
                 ,("\\<equiv>", "\\equiv ")
                 ,("\\<inter>", "\\cap ")
                 ,("\\<Inter>", "\\bigcap ")
                 ,("\\<le>", "\\leq")
                 ,("\\<longleftrightarrow>", "\\longleftrightarrow ")
                 ,("\\<M>", "\\mathcal{M} ")
                 ,("\\<noteq>", "\\neq ")
                 ,("\\<notin>", "\\notin ")
                 ,("\\<one>", "1 ")
                 ,("\\<oplus>","\\oplus ")
                 ,("\\<or>", "\\vee ")
                 ,("\\<partial>", "\\partial ")
                 ,("\\<phi>", "\\phi ")
                 ,("\\<tau>", "\\tau ")
                 ,("\\<union>", "\\cup ")
                 ,("\\<Union>", "\\bigcup ")
                 ,("{is associative on}","\\text{\\{is associative on\\}}")
                 ,("\\<forall>", "\\forall ")
                 ,("\\<exists>", "\\exists ")
                 ,("\\<in>", "\\in ")
                 ,("#", "\\ \\sharp ")
                 ,("IsAgroup","\\text{IsAgroup}")
                 ,("IsAmonoid","\\text{IsAmonoid}")
                 ,("THE", "\\text{The }")
                 ,("\\<^isub>", "_")
                 ,("_", "\\_") 
                 ,("-``", "^{-1}")
                 ,("\\<inverse>", "^{-1}")]
 

convBraces :: String -> String
convBraces = fst . unzip . (convTextsLevel (0::Int)) . (nestLevel '{' '}')

-- | returns tru if a character is not a letter or space
isNotAlphaSpaceBrace :: (Char, Int) -> Bool
isNotAlphaSpaceBrace (c,n) = not $ (isLetter c || c==' ' || c=='{' || c=='}') 

-- | takes a list of (character, nest level) pairs and decides 
-- if this represents a set comprehension. Set comprehensions are
-- those that have a dot on this nest level
isSetCompr :: Int -> [(Char,Int)] -> Bool
isSetCompr lev s = (length s < 11) || (any isNotAlphaSpaceBrace s ) 
   -- (any ( \(c,n) -> (c == '.' &&  n == (lev+1) ) ) s)

-- | converts a string zipped with nesting level information
-- convTextsNestLev :: [(Char,Int)] -> [(Char,Int)]
-- convTextsNestLev 

convTextsLevel :: Int -> [(Char,Int)] -> [(Char,Int)]
convTextsLevel n = appToParts ((> n) . snd) (convSetComprPred  n)

-- | converts a string representing a set comprehension
-- or a predicate starting from given nest level
-- assumes that the input starts with ('{',n) and ends with ('}',n)

convSetComprPred :: Int -> [(Char,Int)] -> [(Char,Int)]
convSetComprPred n sp = if isSetCompr n sp then convSetCompr n sp
                      else convPredicate n sp

-- | converts a set comprehension with nest level information
convSetCompr :: Int -> [(Char,Int)] -> [(Char,Int)]
convSetCompr n sp = 
   [( '\\', n), ( '{', n)] ++ 
   (tail $ init $ convTextsLevel (n+1)sp) ++ 
   [( '\\', n), ( '}', n)]

-- converts a predicate
convPredicate n sp =
   ( zip "\\text{ " (replicate 7 n) ) ++ 
   (tail $ init sp) ++ 
   [( ' ',n),( '}', n)]

