{- 
	This file is part of tiddlyisar - a too for rendering IsarMathLib 
	theories in TiddlyWiki format.
    Copyright (C) 2008  Slawomir Kolodynski

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
-}


module IMLParser
   where

import Text.ParserCombinators.Parsec
import IMLP_datatypes
import Utils

keywords = [ "have", "show", "obtain", "where", "proof"
           , "ultimately", "finally", "assume", "let", "then"
           , "moreover", "also", "from"]

-- | parses and returns any text between start and end.
textBetween :: String -> String -> Parser String
textBetween start end = do
   string start
   t <- manyTill anyChar (try (string end))
   return t

-- | parses and returns Isar comments
comment :: Parser String
comment	= "(*" `textBetween` "*)"

-- | parses and returns the informal text (the text between 
-- "text{*" and "*}"
informalText :: Parser String
informalText = "text{*" `textBetween` "*}"

-- | parses the header line, returns the header. Note that
-- the header should be the same as the name of the theory
headerline :: Parser String
headerline = "header{*\\isaheader{" `textBetween` "}*}"

-- | parses a valid theory or proposition name, also labels.
itemName :: Parser String
itemName = do
	notAnyOf $ map string keywords
	many1 (alphaNum <|> ( char '_') <|> char '.')

-- | parses a variable name
varname :: Parser String
varname = do
   notAnyOf $ map string keywords
   many1 ( alphaNum <|> (char '_') <|> (char '\\') <|> (char '<') <|> (char '>') <|> (char '^') )

-- | parses a quote reference. Isabelle 2007 allows to
-- refere by writing a fact, like `a \<in>B` instead of A1
literalQuote :: Parser String
literalQuote = do
   q <- textBetween "`" "`"
   return ("`" ++ q ++ "`")

-- | parses label reference name. Labels references 
-- in Isabelle 2007 can be either like item names or 
labelRef :: Parser String
labelRef = itemName <|> literalQuote

notAnyOf :: [Parser tok] -> Parser ()
notAnyOf words = try ( do { try (choice words)
                          ; unexpected ""} 
                      <|> return () )

 
-- | parses inner text (the stuff between "")
innerText :: Parser String
innerText = ("\"" `textBetween` "\"") <|> string "?thesis"

-- | parses a preamble of definition, smth. like 
-- "IsATopology ("_ {is a topology}" [90] 91) where"

defpreamble :: Parser (String, String)
defpreamble = do
   name <- itemName
   spaces
   spec <- textBetween "(" ")"
   spaces
   string "where"
   return (name, spec) 


-- | parses definitions. Isabelle 2007 uses the keyword "definition"
definition :: Parser FormalItem
definition = do
   string "definition"
   spaces
   (name,spec) <- option ("","") defpreamble
   spaces 
   d <- innerText
   return Definition
      { defname = if (name /= "") then name else takeWhile (/= '(') d
      , defspec = spec
      , def = d } 

-- | parses formal items
{-parseFormalItem :: Parse FormalItem
parseFormalItem = do
   fi <- ( --}

-- | parses locale 
locale :: Parser FormalItem
locale = do
   string "locale"
   spaces
   n <- itemName
   spaces
   char '='
   spaces
   li <- localeItem `sepEndBy1` spaces
   return (Locale { localename = n, localeItems = li })

-- | parses a locale item: "fixes", "defines" or "assumes"
localeItem :: Parser LocaleItem
localeItem = locItemfixes <|> locItemDefines <|> locItemAssumes 

-- | parses a "fixes" locale item
locItemfixes :: Parser LocaleItem
locItemfixes = do
   string "fixes"
   spaces
   iname <- varname
   spaces
   notation <- ("(" `textBetween` ")") <|> return ""
   return Fixes { fixedName = iname, fixedNotation = notation }

-- | parses a "defines" locale item
locItemDefines :: Parser LocaleItem
locItemDefines = do
   string "defines"
   spaces
   dn <- varname
   spaces
   string "[simp]:"
   spaces
   ld <- "\"" `textBetween` "\""
   return ( Defines { definedName = dn, locdef = ld } )

-- | parses an "assumes" locale item
locItemAssumes :: Parser LocaleItem
locItemAssumes = do
   string "assumes"
   spaces
   a <- listLabStatLists
   return (Assumes a )

-- | parses a section
section :: Parser Section
section = do
   st <- "section{*" `textBetween` "*}"
   spaces
   si <- "text{*" `textBetween` "*}"
   spaces
   --its <- itemWdescription `sepEndBy1` spaces
   its <- itemWdescription `sepEndBy` spaces
   return Section { sectitle = st, secIntro = si, items = its }

-- | parses formal item with its description
itemWdescription :: Parser Item
itemWdescription = do
   desc <- informalText
   spaces
   fitem <- definition <|> try locale <|> proposition -- (l)ocale like (l)emma
   return ( Item { description = desc, formalItem = fitem } ) 
   
-- | parses a theory
theoryParser :: Parser Theory
theoryParser = do
   --lic      <- comment
   spaces
   h        <- headerline
   spaces
   string "theory"
   char ' '
   n        <- itemName
   char ' '
   string "imports "
   importlist <- itemName `sepBy1` (char ' ')
   spaces
   string "begin"
   spaces
   i <- informalText
   spaces
   ss <- section `sepEndBy1` spaces
   string "end"
   return (Theory { header  = h
                  , name    = n
                  , imports = importlist 
                  , thIntro = i
                  , thsections = ss} )

-- parses a locale name in a proposition
incontext :: Parser String
incontext = textBetween "(in "  ") "

-- | parses a proposition
proposition :: Parser FormalItem
proposition = do
   t <- string "theorem" <|> string "lemma" <|> string "corollary"
   char ' '
   c <- option "" incontext
   n <- itemName
   char ':'
   spaces
   p <- ( premises <|> return [] ) 
   string "shows"
   cl <- listLabStatLists
   spaces
   pr <- proof
   return (FormalItem Proposition 
                      {proptype = t
                      , context = c
                      , propname = n
                      , propprems = p
                      , claims = cl
                      , propproof = pr } )

-- | parses premises
premises :: Parser [(String,[String])]
premises = do
   string "assumes"
   spaces
   p <- listLabStatLists
   return p


-- | parses labelled statement list
-- like A1: "a \<in> A"  "b \<in> B"
labelledStatList :: Parser (String,[String])
labelledStatList = do
   label <- statLabel <|> return ""
   spaces
   inners <- innerText `sepEndBy1` spaces
   return (label, inners)

-- | parses a list of labelled statement lists
listLabStatLists :: Parser [(String,[String])]
listLabStatLists = labelledStatList `sepBy1` (spaces >> (try $ string "and") >> spaces)

   
-- | parses a statement label
statLabel :: Parser String
statLabel = do 
   n <- itemName
   char ':'
   return n
 
-- | parses short proof that has only a tactic in it 
shortProofByTac :: Parser Proof
shortProofByTac = do
   string "by"
   spaces 
   tac <- tactic 
   return ( UsingBy { useunfold = "", usedprops = [], ptactic = tac } ) 

-- | parses short proof that has "using" or "unfolding" keyword with references
shortProofRefTac :: String -> Parser Proof
shortProofRefTac meth = do
   string meth
   spaces
   itms <- (manyTill (do {i <- itemName; spaces; return i}) (try (string "by")))
   spaces
   tac <- tactic 
   return ( UsingBy { useunfold = meth, usedprops = itms, ptactic = tac } )


{-shortProofRefTac = do
   lr  <- optItemsBetween "using" "by"
   spaces
   tac <- tactic
   return ( UsingBy {usedprops = lr, ptactic = tac } )

-- | a more general parser parsing an optional sequence terminated 
-- by special item
optItemsBetween :: String -> String -> Parser [String]
optItemsBetween beg end = 
   (do { string end ; return [] } ) <|>
   ( do
     string beg
     spaces
     itms <- 
        (manyTill (do {i <- itemName; spaces; return i}) (try (string end)))
     return itms )-}

-- parsers a tactic
tactic :: Parser String
tactic = choice $ map string ["simp", "auto", "blast", "fast"] 

-- parses "by rule(... )" type of proof
shortProofByRule :: Parser Proof
shortProofByRule = do
   string "by (rule "
   r <- itemName
   char ')'
   return (ByRule r)

-- | parses an item with space
itemSpace :: Parser String
itemSpace = do
   i <- itemName
   spaces
   return i

-- | parses obtain ... where... with proof
proofcomobtain :: Parser ProofCommand
proofcomobtain = do
   string "obtain"
   spaces
   vars <- varname `sepEndBy` spaces
   string "where"
   spaces
   cp <- claimproof
   return ( PCbtain vars cp )

-- | parses have with proof or show with proof
proofcomhaveshow :: Parser ProofCommand
proofcomhaveshow = do
   hs <- string "have" <|> string "show"
   spaces
   cp <- claimproof
   return $ PChaveShow hs cp
  
-- | parses a proof command - have, show or obtain
proofcom :: Parser ProofCommand
proofcom = proofcomobtain <|> proofcomhaveshow

-- | parses a False constant. This is treated as a kind of claim
claimfalse :: Parser [(String,[String])]
claimfalse = do
   string "False"
   return [("",["False"])]

-- | parses a claim with a proof
claimproof :: Parser ClaimProof
claimproof = do
   c <- (try listLabStatLists) <|> claimfalse
   spaces
   p <- proof
   return ClaimProof { cpclaims = c, cpproof = p }

-- | parses assume ...
assume :: Parser InitStep
assume = do
   string "assume"
   spaces
   a <- listLabStatLists
   return (Assume a)

-- | parses "note"
note :: Parser InitStep
note = do
   ss <- locrefs "note"
   return (Note ss)

-- | parses a proof block (is this OK with nested blocks?)
stepblock :: Parser InitStep
stepblock = do
   char '{'
   spaces
   ps <- proofstep `sepEndBy1` spaces
   char '}'
   return (StepBlock ps)

-- | parses an initit statement in a resoning
initStep :: Parser InitStep
initStep = stepblock <|> assume <|> note <|> 
  ( do 
       lr <- option [] (locrefs "from")
       spaces
       pc <- proofcom
       return (InitStep lr pc) 
       )

-- Parses "then" type of connected step
thenstep :: Parser ConnectedStep
thenstep = do
   string "then"
   spaces
   pc <- proofcom
   return (ThenStep pc)

-- Parses "hence" or "thus" type of connected step
hencethus :: Parser ConnectedStep
hencethus = do
   ht <- string "hence" <|> string "thus"
   spaces
   c <- listLabStatLists
   spaces
   string "by"
   char ' '
   tac <- tactic
   return $ HenceThus { henceorthus = ht
                    , ttclaims = c
                    , tttactic = tac }
 
-- | Parse "with" type of connected step
withstep :: Parser ConnectedStep
withstep = do
   lr <- locrefs "with"
   spaces
   pc <- proofcom
   return ( WithStep lr pc )

-- parses a "fix" directive
fixstep :: Parser ProofStep
fixstep = do
   string "fix "
   vs <- varname `sepEndBy1` spaces
   return (FixStep vs)

--parses a "let" directive
letstep :: Parser ProofStep
letstep = do
   string "let ?"
   v <- varname
   string " = "
   d <- innerText
   return (LetStep v d)

-- | parses a connected step
connectedStep :: Parser ConnectedStep
connectedStep = withstep <|> try thenstep <|> try hencethus


-- parses reasoning chain
reasoning :: Parser Reasoning
reasoning = do
   is <- initStep
   spaces
   cs <- connectedStep `sepEndBy` spaces
   return ( Reasoning is cs)

-- | parses a statement starting with "moreover"
moreoveritem :: String -> Parser Reasoning
moreoveritem s = do
   string s
   spaces
   r <- reasoning
   return r

-- | parses one moreover body, that is a construct like
-- moreover A
-- moreover B
-- ultimately
-- also parses "also" construct

moreoverbody :: String -> Parser MoreoverBody
moreoverbody s = do
   items <- (moreoveritem s) `sepEndBy1` spaces
   if s=="also" then string "finally" else string "ultimately"
   spaces
   pc <- proofcomhaveshow
   spaces
   cs <- connectedStep `sepEndBy` spaces
   return MoreoverBody { mrvMrvs = items
                       , ultimfinal = pc
                       , followup = cs }

-- parses a "moreover" construct
moreoveralso :: String -> Parser ProofStep
moreoveralso s = do
   ir <- reasoning
   spaces
   mbs <- (if (s=="also") then sepEndBy1 else sepEndBy) (moreoverbody s) spaces
   return (MoreoverAlso s ir mbs)


-- | parses the "next" as a proof step
next :: Parser ProofStep
next = do
   string "next"
   return Next


-- parses a proof step
proofstep :: Parser ProofStep
proofstep = letstep <|> try fixstep <|> 
   try (moreoveralso "also") <|> try (moreoveralso "moreover") <|> 
   next -- moreoveralso may start with "note" which starts like "next", that's why we need to try it

-- parses a long proof
longproof :: Parser Proof
longproof = do
   string "proof"
   spaces
   d <- option ' ' (char '-')
   spaces
   ps <- proofstep `sepEndBy1` spaces
   string "qed"
   return (LongProof { dash = (d == '-'), proofSteps = ps } ) 
   
-- | parses a proof, for now only the short one
proof :: Parser Proof
proof = longproof <|>
        (try shortProofByTac) <|> shortProofByRule <|> 
        (try $ shortProofRefTac "using") <|> (shortProofRefTac "unfolding")

-- | parses local references starting from "from" or "with" or "note"
locrefs :: String -> Parser [String]
locrefs s = do
   string s
   spaces
   r <- labelRef `sepEndBy1` spaces
   return r


-- | converts a list of Either's to a list of values, stopping at error
eithers2vals :: [Either ParseError a] -> [a]
eithers2vals = map either2val
   where 
      either2val (Left err) = error ("parse error at " ++ (show err) )
      either2val (Right x)  = x

-- | parses a list of theory texts
parseTheories :: [(String,String)] -> [Theory]
parseTheories = eithers2vals . map (\(n,t) -> (parse theoryParser n) (prepTheory t) )


-- | preprocesses a string with a theory by removing comments
-- and semicolons. We need to add a space to work around 
-- the fact that appBetween does not work when the 
-- string begins with a beginning mark
prepTheory :: String -> String
prepTheory = ( filter (/=';') ) . ( dropWhile (/='h') ). ( appBetween (\x-> []) "(*" "*)" ) . ( ' ': )

-- | test with parseTest theoryParser x
   


   

