(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2008  Slawomir Kolodynski

    This program is free software Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES LOSS OF USE, DATA, OR PROFITS OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header{*\isaheader{Finite\_ZF.thy}*}
theory Finite_ZF imports Nat_ZF_IML Cardinal

begin

text{*Standard Isabelle Finite.thy contains a very useful 
  notion of finite powerset: the set of finite subsets of a given set.
  The definition, however, is specific to Isabelle and based on the notion
  of "datatype", obviously not something that belongs to ZF set theory.
  This theory file devolopes the notion of finite powerset similarly as
  in Finite.thy, but based on standard library's 
  Cardinal.thy. This theory file is intended to 
  replace IsarMathLib's @{text "Finite1"} and @{text "Finite_ZF_1"} theories
  that are currently derived from the "datatype" approach.
  *}

section{*Definition and basic properties of finite powerset*}

text{*The goal of this section is to prove an induction theorem about
  finite powersets: if the empty set has some property and this property is 
  preserved by adding a single element of a set, 
  then this property is true for all finite subsets of this set. *}


text{*We defined the finite powerset @{text "FinPow(X)"} as those elements 
  of the powerset that are finite.*}

definition
  "FinPow(X) \<equiv> {A \<in> Pow(X). Finite(A)}"

text{*The cardinality of an element of finite powerset is a natural number.*}

lemma card_fin_is_nat: assumes "A \<in> FinPow(X)" 
  shows "|A| \<in> nat" and "A \<approx> |A|"
  using assms FinPow_def Finite_def cardinal_cong nat_into_Card 
    Card_cardinal_eq by auto
  
text{*We can decompose the finite powerset into collection of
  sets of the same natural cardinalities.*}

lemma finpow_decomp: 
  shows "FinPow(X) = (\<Union>n \<in> nat. {A \<in> Pow(X). A \<approx> n})"
  using Finite_def FinPow_def by auto

text{*Finite powerset is the union of sets of cardinality
  bounded by natural numbers.*}

lemma finpow_union_card_nat: 
  shows "FinPow(X) = (\<Union>n \<in> nat. {A \<in> Pow(X). A \<lesssim> n})"
proof -
  have "FinPow(X) \<subseteq> (\<Union>n \<in> nat. {A \<in> Pow(X). A \<lesssim> n})"
    using finpow_decomp FinPow_def eqpoll_imp_lepoll
    by auto
  moreover have 
    "(\<Union>n \<in> nat. {A \<in> Pow(X). A \<lesssim> n}) \<subseteq> FinPow(X)"
    using lepoll_nat_imp_Finite FinPow_def by auto
  ultimately show ?thesis by auto
qed

text{*A different form of @{text "finpow_union_card_nat"} (see above) -
  a subset that has not more elements than a given natural number 
  is in the finite powerset.*}

lemma lepoll_nat_in_finpow: 
  assumes "n \<in> nat"   "A \<subseteq> X"  "A \<lesssim> n"
  shows "A \<in> FinPow(X)"
  using assms finpow_union_card_nat by auto

text{*If we remove an element and put it back we get the set back.
  *}

lemma rem_add_eq: assumes "a\<in>A" shows "(A-{a}) \<union> {a} = A"
  using assms by auto

text{*Induction for finite powerset. This is smilar to the
  standard Isabelle's @{text "Fin_induct"}. *}

theorem FinPow_induct: assumes A1: "P(0)" and
  A2: "\<forall>A \<in> FinPow(X). P(A) \<longrightarrow> (\<forall>a\<in>X. P(A \<union> {a}))" and
  A3: "B \<in> FinPow(X)"
  shows "P(B)"
proof -
  { fix n assume "n \<in> nat"
    moreover from A1 have I: "\<forall>B\<in>Pow(X). B \<lesssim> 0 \<longrightarrow> P(B)"
      using lepoll_0_is_0 by auto
    moreover have "\<forall> k \<in> nat. 
      (\<forall>B \<in> Pow(X). (B \<lesssim> k \<longrightarrow> P(B))) \<longrightarrow> 
      (\<forall>B \<in> Pow(X). (B \<lesssim> succ(k) \<longrightarrow> P(B)))"
    proof -
      { fix k assume A4: "k \<in> nat"
	assume A5: "\<forall> B \<in> Pow(X). (B \<lesssim> k \<longrightarrow> P(B))"
	fix B assume A6: "B \<in> Pow(X)"  "B \<lesssim> succ(k)"
	have "P(B)"
	proof -
	  have "B = 0 \<longrightarrow> P(B)"
	  proof -
	    { assume "B = 0"
	      then have "B \<lesssim> 0" using lepoll_0_iff
		by simp
	      with I A6 have "P(B)" by simp 
	    } thus "B = 0 \<longrightarrow> P(B)" by simp
	  qed
	  moreover have "B\<noteq>0 \<longrightarrow> P(B)"
	  proof -
	    { assume "B \<noteq> 0"
	      then obtain a where II: "a\<in>B" by auto
	      let ?A = "B - {a}"
	      from A6 II have "?A \<subseteq> X" and "?A \<lesssim> k" 
		using Diff_sing_lepoll by auto
	      with A4 A5 have "?A \<in> FinPow(X)" and "P(?A)" 
		using lepoll_nat_in_finpow finpow_decomp 
		by auto
	      with A2 A6 II have " P(?A \<union> {a})"
		by auto
	      moreover from II have "?A \<union> {a} = B"
		by auto
	      ultimately have "P(B)" by simp 
	    } thus "B\<noteq>0 \<longrightarrow> P(B)" by simp
	  qed
	  ultimately show "P(B)" by auto
	qed
      } thus ?thesis by blast
    qed
    ultimately have "\<forall>B \<in> Pow(X). (B \<lesssim> n \<longrightarrow> P(B))"
      by (rule ind_on_nat)
  } then have "\<forall>n \<in> nat. \<forall>B \<in> Pow(X). (B \<lesssim> n \<longrightarrow> P(B))"
    by auto
  with A3 show "P(B)" using finpow_union_card_nat
    by auto
qed

text{*If a family of sets is closed with respect to taking intersections
  of two sets then it is closed with respect to taking intersections 
  of any nonempty finite collection.*}

lemma inter_two_inter_fin: 
  assumes A1: "\<forall>V\<in>T. \<forall>W\<in>T. V \<inter> W \<in> T" and
  A2: "N \<noteq> 0" and A3: "N \<in> FinPow(T)"
  shows "(\<Inter>N \<in> T)"
proof -
  have "0 = 0 \<or> (\<Inter>0 \<in> T)" by simp
  moreover have "\<forall>M \<in> FinPow(T). (M = 0 \<or> \<Inter>M \<in> T) \<longrightarrow> 
    (\<forall>W \<in> T. M\<union>{W} = 0 \<or> \<Inter>(M \<union> {W}) \<in> T)"
  proof -
    { fix M assume "M \<in> FinPow(T)"
      assume A4: "M = 0 \<or> \<Inter>M \<in> T"
      { assume "M = 0"
	hence "\<forall>W \<in> T. M\<union>{W} = 0 \<or> \<Inter>(M \<union> {W}) \<in> T"
	  by auto }
      moreover
      { assume "M \<noteq> 0" 
	with A4 have "\<Inter>M \<in> T" by simp
	{ fix W assume "W \<in> T"
	  from `M \<noteq> 0` have "\<Inter>(M \<union> {W}) = (\<Inter>M) \<inter> W" 
	    by auto
	  with A1 `\<Inter>M \<in> T` `W \<in> T` have "\<Inter>(M \<union> {W}) \<in> T"
	    by simp
	} hence "\<forall>W \<in> T. M\<union>{W} = 0 \<or> \<Inter>(M \<union> {W}) \<in> T"
	  by simp }
      ultimately have "\<forall>W \<in> T. M\<union>{W} = 0 \<or> \<Inter>(M \<union> {W}) \<in> T"
	by blast
    } thus ?thesis by simp
  qed
  moreover note `N \<in> FinPow(T)`
  ultimately have "N = 0 \<or> (\<Inter>N \<in> T)"
    by (rule FinPow_induct)
  with A2 show "(\<Inter>N \<in> T)" by simp
qed

text{*If a family of sets contains the empty set and
  is closed with respect to taking unions
  of two sets then it is closed with respect to taking unions 
  of any finite collection.*}

lemma union_two_union_fin:
  assumes A1: "0 \<in> C" and A2: "\<forall>A\<in>C. \<forall>B\<in>C. A\<union>B \<in> C" and 
  A3: "N \<in> FinPow(C)"
  shows "\<Union>N \<in> C"
proof -
  from `0 \<in> C` have "\<Union>0 \<in> C" by simp
  moreover have "\<forall>M \<in> FinPow(C). \<Union>M \<in> C \<longrightarrow> (\<forall>A\<in>C. \<Union>(M \<union> {A}) \<in> C)"
  proof -
    { fix M assume "M \<in> FinPow(C)"
      assume "\<Union>M \<in> C"
      fix A assume "A\<in>C"
      have "\<Union>(M \<union> {A}) = (\<Union>M) \<union> A" by auto
      with A2 `\<Union>M \<in> C`  `A\<in>C` have  "\<Union>(M \<union> {A}) \<in> C"
	by simp
    } thus ?thesis by simp
  qed
  moreover note `N \<in> FinPow(C)`
  ultimately show "\<Union>N \<in> C" by (rule FinPow_induct)
qed

text{*Empty set is in finite power set.*}

lemma empty_in_finpow: shows "0 \<in> FinPow(X)"
  using FinPow_def by simp

text{*Singleton is in the finite powerset.*}

lemma singleton_in_finpow: assumes "x \<in> X"
  shows "{x} \<in> FinPow(X)" using assms FinPow_def by simp

text{*Union of two finite subsets is a finite subset.*}

lemma union_finpow: assumes "A \<in> FinPow(X)" and "B \<in> FinPow(X)"
  shows "A \<union> B \<in> FinPow(X)"
  using assms FinPow_def by auto

text{*An image of a finite set is finite.*}
  
lemma fin_image_fin: assumes "\<forall>V\<in>B. K(V)\<in>C" and "N \<in> FinPow(B)"
  shows "{K(V). V\<in>N} \<in> FinPow(C)"
proof -
  have "{K(V). V\<in>0} \<in> FinPow(C)" using FinPow_def
    by auto
  moreover have "\<forall>A \<in> FinPow(B). 
    {K(V). V\<in>A} \<in> FinPow(C) \<longrightarrow> (\<forall>a\<in>B. {K(V). V \<in> (A \<union> {a})} \<in> FinPow(C))"
  proof -
    { fix A assume "A \<in> FinPow(B)"
      assume  "{K(V). V\<in>A} \<in> FinPow(C)"
      fix a assume "a\<in>B"
      have  "{K(V). V \<in> (A \<union> {a})} \<in> FinPow(C)"
      proof -
	have "{K(V). V \<in> (A \<union> {a})} = {K(V). V\<in>A} \<union> {K(a)}"
	  by auto
	moreover note `{K(V). V\<in>A} \<in> FinPow(C)`
	moreover from `\<forall>V\<in>B. K(V)\<in>C`  `a\<in>B` have "{K(a)} \<in>  FinPow(C)"
	  using singleton_in_finpow by simp
	ultimately show ?thesis using union_finpow by simp
      qed
    } thus ?thesis by simp
  qed
  moreover note `N \<in> FinPow(B)`
  ultimately show "{K(V). V\<in>N} \<in> FinPow(C)"
    by (rule FinPow_induct)
qed



end
