(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2007  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Semigroup\_ZF.thy}*}

theory Semigroup_ZF imports func_ZF Fold_ZF

begin

text{*It seems that the minimal setup needed to talk about a product of a 
  sequence is a set with a binary operation. 
  Such object is called "magma". However, interesting properties
  show up when the binary operation is associative and such alebraic structure
  is called a semigroup. 
  In this theory file we define and study sequences of partial 
  products of sequences of magma and semigroup elements.*}

section{*Products of sequences of semigroup elements*}

text{*Semigroup is a a magma in which the binary operation is associative.
  In this section we mostly study the products of sequences of elements 
  of semigroup. The goal is to establish the fact that taking the product of 
  a sequence is distributive with respect to concatenation of sequences, 
  i.e for two sequences $a,b$ of the semigroup elements we have 
  $\prod (a\sqcup b) = (\prod a)\cdot (\prod b)$, where "$a \sqcup b$" 
  is concatenation of $a$ and $b$ ($a$@{text "++"}$b$ in Haskell notation).
  Less formally, we want to show that we can discard parantheses in 
  expressions of the form 
  $(a_0\cdot a_1\cdot .. \cdot a_n)\cdot (b_0\cdot .. \cdot b_k)$.
  *}

text{*The definition of the @{text "semigr0"} context below introduces notation
  for writing about finite sequences and semigroup products. 
  In the context we fix the carrier and
  denote it $G$. The binary operation on $G$ is called $f$. 
  All theorems proven in the context @{text "semigr0"} 
  will implicitly assume that $f$ is an associative operation on $G$.
  We will use the multiplicative notation for the semigroup operation.
  The product of a sequence $a$ is denoted $\prod a$. We will write
  $a\hookleftarrow x$ for the result of appending an element $x$ to
  the finite sequence (list) $a$. This is a bit nonstandard, 
  but I don't have a better idea for the "append" notation. Finally,
  $a\sqcup b$ will denote the concatenation of the lists $a$ and $b$.*}

locale semigr0 =
  
  fixes G and f

  assumes assoc_assum: "f {is associative on} G"

  fixes prod (infixl "\<cdot>" 72)
  defines prod_def [simp]: "x \<cdot> y \<equiv> f`\<langle>x,y\<rangle>"

  fixes seqprod :: "i\<Rightarrow>i" ("\<Prod> _" 71)
  defines seqprod_def [simp]: "\<Prod> a \<equiv> Fold(f,a`(0),Tail(a))"

  fixes append (infix "\<hookleftarrow>" 72)
  defines append_def [simp]: "a \<hookleftarrow> x \<equiv> Append(a,x)"

  fixes concat (infixl "\<squnion>" 69)
  defines concat_def [simp]: "a \<squnion> b \<equiv> Concat(a,b)";

text{*The next lemma shows our assumption on the associativity
  of the semigroup operation in the notation defined in in the 
  @{text "semigr0"} context.*}

lemma (in semigr0) semigr_assoc: assumes "x \<in> G"  "y \<in> G"  "z \<in> G"
  shows "x\<cdot>y\<cdot>z = x\<cdot>(y\<cdot>z)"
  using prems assoc_assum IsAssociative_def by simp;

text{*In the way we define associativity the assumption that
  the $f$ is associative on $G$ also implies that it is a binary
  operation on $X$. *}

lemma (in semigr0) semigr_binop: shows "f : G\<times>G \<rightarrow> G"
  using assoc_assum IsAssociative_def by simp;

text{*Lemma @{text "append_1elem"} written in the notation used in 
  the @{text "semigr0"} context.*}

lemma (in semigr0) append_1elem_nice: 
  assumes "n \<in> nat" and "a: n \<rightarrow> X" and "b : 1 \<rightarrow> X"
  shows "a \<squnion> b = a \<hookleftarrow> b`(0)"
  using prems append_1elem by simp;

text{*Lemma @{text "concat_init_last_elem"} rewritten
  in the notation used in the @{text "semigr0"} context.*}

lemma (in semigr0) concat_init_last: 
  assumes "n \<in> nat"  "k \<in> nat" and 
  "a: n \<rightarrow> X"  and "b : succ(k) \<rightarrow> X"
  shows "(a \<squnion> Init(b)) \<hookleftarrow> b`(k) = a \<squnion> b"
  using prems concat_init_last_elem by simp;

text{*The product of semigroup (actually, magma -- we don't
   need associativity for this) elements is in the semigroup.*}

lemma (in semigr0) prod_type: 
  assumes "n \<in> nat" and "a : succ(n) \<rightarrow> G"
  shows "(\<Prod> a) \<in> G"
proof -
  from assms have 
    "succ(n) \<in> nat"  "f : G\<times>G \<rightarrow> G"  "Tail(a) : n \<rightarrow> G"
    using semigr_binop tail_props by simp_all;
  moreover from assms have "a`(0) \<in> G" and "G \<noteq> 0"
    using empty_in_every_succ apply_funtype
    by auto;
  ultimately show "(\<Prod> a) \<in> G" using fold_props
    by simp;
qed;

text{*What is the product of one element list?*}

lemma (in semigr0) prod_of_1elem: assumes A1: "a: 1 \<rightarrow> G"
  shows "(\<Prod> a) = a`(0)"
proof -
  have "f : G\<times>G \<rightarrow> G" using semigr_binop by simp;
  moreover from A1 have "Tail(a) : 0 \<rightarrow> G" using tail_props
    by blast;
  moreover from A1 have "a`(0) \<in> G" and "G \<noteq> 0" 
    using apply_funtype by auto;
  ultimately show "(\<Prod> a) =  a`(0)" using fold_empty by simp;
qed;

text{*What happens to the product of a list when we append an element 
  to the list?*}

lemma (in semigr0) prod_append: assumes A1: "n \<in> nat" and
  A2: "a : succ(n) \<rightarrow> G" and A3: "x\<in>G"
  shows "(\<Prod> a\<hookleftarrow>x) = (\<Prod> a) \<cdot> x"
proof -
  from A1 A2 have I: "Tail(a) : n \<rightarrow> G"  "a`(0) \<in> G"
    using tail_props empty_in_every_succ apply_funtype
    by simp_all;
  from assms have "(\<Prod> a\<hookleftarrow>x) = Fold(f,a`(0),Tail(a)\<hookleftarrow>x)"
    using head_of_append tail_append_commute by simp;
  also from A1 A3 I have "\<dots> = (\<Prod> a) \<cdot> x"
    using semigr_binop fold_append by simp;
  finally show ?thesis by simp;
qed;

text{*The main theorem of the section: taking the product of 
  a sequence is distributive with respect to concatenation of sequences.
  The proof is by induction on the length of the second list.*}

theorem (in semigr0) prod_conc_distr: 
  assumes A1: "n \<in> nat"  "k \<in> nat" and
  A2: "a : succ(n) \<rightarrow> G"   "b: succ(k) \<rightarrow> G"
  shows "(\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)"
proof -
  from A1 have "k \<in> nat" by simp;
  moreover have "\<forall>b \<in> succ(0) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)"
  proof -
    { fix b assume A3: "b : succ(0) \<rightarrow> G"
      with A1 A2 have
	"succ(n) \<in> nat"  "a : succ(n) \<rightarrow> G"  "b : 1 \<rightarrow> G" 
	by auto;
      then have "a \<squnion> b = a \<hookleftarrow> b`(0)" by (rule append_1elem_nice);
      with A1 A2 A3 have "(\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)"
	using apply_funtype prod_append semigr_binop prod_of_1elem
	by simp;
    } thus ?thesis by simp;
  qed;
  moreover have "\<forall>j \<in> nat. 
    (\<forall>b \<in> succ(j) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)) \<longrightarrow>
    (\<forall>b \<in> succ(succ(j)) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b))"
  proof -
    { fix j assume A4: "j \<in> nat" and 
      A5: "(\<forall>b \<in> succ(j) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b))"
      { fix b assume A6: "b : succ(succ(j)) \<rightarrow> G"
	let ?c = "Init(b)"
	from A4 A6 have  T: "b`(succ(j)) \<in> G" and
	  I: "?c : succ(j) \<rightarrow> G" and II: "b = ?c\<hookleftarrow>b`(succ(j))"
	  using apply_funtype init_props by auto;
	from A1 A2 A4 A6 have
	  "succ(n) \<in> nat"  "succ(j) \<in> nat"
	  "a : succ(n) \<rightarrow> G"  "b : succ(succ(j)) \<rightarrow> G"
	  by auto;
	then have III: "(a \<squnion> ?c) \<hookleftarrow> b`(succ(j)) = a \<squnion> b"
	  by (rule concat_init_last);
	from A4 I T have "(\<Prod> ?c\<hookleftarrow>b`(succ(j))) = (\<Prod> ?c) \<cdot> b`(succ(j))"
	  by (rule prod_append);
	with II have 
	  "(\<Prod> a) \<cdot> (\<Prod> b) = (\<Prod> a) \<cdot> ((\<Prod> ?c) \<cdot> b`(succ(j)))"
	  by simp;
	moreover from A1 A2 A4 T I have
	  "(\<Prod> a) \<in> G"  "(\<Prod> ?c) \<in> G"  "b`(succ(j)) \<in> G"
	  using prod_type by auto;
	ultimately have 
	  "(\<Prod> a) \<cdot> (\<Prod> b) =  ((\<Prod> a) \<cdot> (\<Prod> ?c)) \<cdot> b`(succ(j))"
	  using semigr_assoc by auto;
	with A5 I have "(\<Prod> a) \<cdot> (\<Prod> b) = (\<Prod> (a \<squnion> ?c))\<cdot>b`(succ(j))"
	  by simp;
	moreover
	from A1 A2 A4 I have
	  T1: "succ(n) \<in> nat"  "succ(j) \<in> nat" and
	  "a : succ(n) \<rightarrow> G"   "?c : succ(j) \<rightarrow> G"
	  by auto;
	then have "Concat(a,?c): succ(n) #+ succ(j) \<rightarrow> G"
	  by (rule concat_props);
	with A1 A4 T have
	  "succ(n #+ j) \<in> nat"   
	  "a \<squnion> ?c : succ(succ(n #+j)) \<rightarrow> G"
	  "b`(succ(j)) \<in> G"
	  using succ_plus by auto;
	then have 
	  "(\<Prod> (a \<squnion> ?c)\<hookleftarrow>b`(succ(j))) = (\<Prod> (a \<squnion> ?c))\<cdot>b`(succ(j))"
	  by (rule prod_append);
	with III have "(\<Prod> (a \<squnion> ?c))\<cdot>b`(succ(j)) =  \<Prod> (a \<squnion> b)"
	  by simp;
	ultimately have "(\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)"
	  by simp;
      } hence "(\<forall>b \<in> succ(succ(j)) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b))"
	by simp;
    } thus ?thesis by blast;
  qed;
  ultimately have "\<forall>b \<in> succ(k) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)"
    by (rule ind_on_nat);
  with A2 show "(\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)" by simp;
qed;
    
  
end
